/******************************************************************************
* TASTE (Testability Analysis SuiTE) version 1.00
* Copyright (c) 2008 Josef Strnadel, all rights reserved
* Brno University of Technology, Faculty of Information Technology
* -----------------------------------------------------------------------------
* This is a free software: you can redistribute it and/or modify it 
* under the terms of the latest version of the GNU Lesser General Public License 
* as published by the Free Software Foundation.
* 
* This software is distributed in the hope that it will be useful, 
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
* See the GNU General Public License and the GNU Lesser General Public License 
* for more details.
* 
* You should have received a copy of the GNU General Public License
* and GNU Lesser General Public License along with this software. 
* If not, see <http://www.gnu.org/licenses/>.
* -----------------------------------------------------------------------------
* filename:     main.cc
* version:      1.00
* started:      25. 4. 2007
* last revised: 21. 7. 2008
* language:     ANSI C++
* author:       Josef Strnadel
*               web:      http://www.fit.vutbr.cz/~strnadel/index.php.en
*               e-mail:   strnadel@fit.vutbr.cz
*               address:  Bozetechova 2, 61266 Brno, Czech Republic
*               phone:    +420 54114-1211
*               fax:      +420 54114-1270
* -----------------------------------------------------------------------------
* description:	main module
* -----------------------------------------------------------------------------
* notes:        -
* -----------------------------------------------------------------------------
* history of modifications:
*               -
* -----------------------------------------------------------------------------
* known bugs:   -
******************************************************************************/
//! \file main.cc main module 

#include <iostream>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <utility>

#include <stdio.h>
#include <stdlib.h>

#include "getpot/GetPot"

#include "defs.h"
#include "fileio.h"
#include "misc.h"
#include "model.h"

using namespace std;

/******************************************************************************
*	local copy of license versions
******************************************************************************/
string lgpl = "licensing/lgpl3.license";
string warranty_liability = "licensing/warranty_liability.txt";
string taste = "licensing/taste.license";

/******************************************************************************
*	globals
******************************************************************************/
extern unsigned long nerrors, nwarns;
extern char act_fname[MAX_STR_LEN];

/*!\brief debug/normal mode switch. In debug mode, implementation and low-level model related data are sent to std. Undocumented at present. */
bool debug_mode=false;

/*!\brief scan mode switch 
*/
/*! switching the mode ON, scan will be implemented into the circuit structure. 
 *  The mode can be changed using a command-line switch (see tool's help).
 */
bool scan_mode=false;

/*!\brief circuit mode switch 
*/
/*! switching the mode ON, all actions will be performed over particular circuit within 
 *  the design. Both the mode can be selected and circuit name set by means of 
 *  command-line commands (see tool's help).
 */
bool cir_mode=false;

/*!\brief design mode switch 
*/
/*! switching the mode ON, all actions will be performed over whole design. 
 *  The mode can be selected by means of command-line switch (see tool's help).
 */
bool design_mode=false;

/*!\brief operation used to evaluate penalty during mark transportation process
*/
/*! Implicitly set to PP_EXP.
 *  Undocumented at present.
 */
tProPen mark_operation = PP_EXP;

/*!\brief type of sequential length estimation
*/
/*! Implicitly set to SE_BASIC.
 *  Undocumented at present.
 */
tSeqEst clk_cnt = SE_BASIC;

/*!\brief string to store an application name. The name is taken from command line (real executable name is taken) */
string appName = "";

/*!\brief string to store a file name circuit/design data will be read from. The name is set by means of a command line command */
string  fname4TA = "";

/*!\brief string to store a circuit name. The name is set by means of a command line command */
string  cirname = "";

/*!\brief string to store a file name information about circuit registers will be read from. The name is set by means of a command line command  */
string freg;

/*!\brief string to store a file name information about scan will be read from. The name is set by means of a command line command */
string fscan;

/*!\brief pointer to the top clsDesign object */
clsDesign *dsgnPtr = NULL;

/*!\brief information header of the tool */
const char *header[] =
{
  "-------------------------------------",
  "TASTE (Testability Analysis SuiTE) version 1.00",
  "Copyright (c) 2008 Josef Strnadel", 
  "Developped at Brno University of Technology, Faculty of Information Technology",
  "This software comes with ABSOLUTELY NO WARRANTY; for details use `-show w'.",
  "You are welcome to use, modify and/or redistribute the software under certain conditions; use `-show c' for details.",
  "-------------------------------------",
  NULL
};

/*!\brief function to print an information header */
void print_header()
{
  int i=0;
  for(const char *s=header[i]; (s=header[i]); i++) cout << s << endl;
}

/*!\brief function to print an information footer */
void print_footer(int ncircs, int nmodules, int nnodes, int nnodebits)
{
/*  cout << "-------------------------------------" << endl;
  cout << "Stats:" << endl;
  cout << ncircs << " circuit(s) created" << endl;
  cout << nmodules << " modules(s) created" << endl;
  cout << nnodes << " ports(s) created" << endl;
  cout << nnodebits << " ports(s) created" << endl;
  cout << "-------------------------------------" << endl;*/
  cout << "TASTE ended with " << nerrors << " error(s) and " << nwarns << " warning(s)." << endl;
}

/*!\brief function to print a help of the tool */
void print_help()
{
  cout << "SYNTAX: " << appName << " -iTA <file> [option(s)]" << endl;
  cout << "options:" << endl;
  cout << "   -license_read\tuse only if you have already read license conditions" << endl;
  cout << "   -h\t\t\tdisplay (this) help information" << endl;
  cout << "   -scan\t\tturn scan mode ON" << endl;
  cout << "   -design\t\tturn design mode ON" << endl;
  cout << "   -cir <name>\t\tset name of circuit and turns circuit mode ON" << endl;
  cout << "   -i <file>\t\tspecify file containing circuit/design data" << endl;
  cout << "   -seqest \t\tspecify way of sequential length estimation" << endl;
  cout << "   -propen \t\tspecify way a propagation penalty is evaluated" << endl;
  cout << "   -ta\t\t\trun testability analysis of given circuit" << endl;
  cout << "   -o <format(s)>\toutput format(s) for storing  testability analysis" << endl;
  cout << "                 \tresults, where 'formats' can be one or comma-separated" << endl;
  cout << "                 \tcombination of html, tex, txt" << endl;
  cout << "   -T <file(s)>\t\tspecify explicit library/ies of available module-types" << endl;
  cout << "             \t\t(otherwise, implicit library taste.templates taken)" << endl;
  cout << "   -freg <file>\t\tspecify explicit file with circuit/design register list" << endl;
  cout << "             \t\t(otherwise, implicit *.reg will be taken)" << endl;
  cout << "   -fscan <file>\tspecify explicit file with circuit/design scan notation" << endl;
  cout << "             \t\t(otherwise, implicit *.scan will be taken)" << endl;
  cout << "TASTE ended with " << nerrors << " error(s) and " << nwarns << " warning(s)." << endl;
  exit(0);
}

/*!\brief main function 
*/
/*! in the function, command line is processed and corresponding actions are started.
 *
 */
int main(int argc, char *argv[])
{
  FILE *f_in=NULL;
  string str_tmp;

  print_header();

  GetPot cl(argc, argv);
  vector<string> GetPotUFOs;
  
  appName = cl[0];

  if(cl.size() == 1) { cout << "Please, read all related LICENSE CONDITIONS before using TASTE!\nPress ENTER to continue..."; getchar(); print_help(); } 

  if(cl.search("-h")) { print_help(); } 

  if(cl.search("-show"))
  {
     FILE *fp;
     char c;
     if(cl.follow("", "-show") == "c")
     { 
        if((fp=fopen(taste.c_str(), "r")) == NULL) {error("License file '%s' missing! It should be available in taste/licensing directory of the distribution package. Please look for it, because reading the license is an necessary condition for using TASTE!\n", taste.c_str()); print_help();}
        else
        {
          ifstream fin;
          fin.open(taste.c_str());
          while(!fin.eof()) {c=(char)fin.get(); if(c!=EOF) cout<<c;}
          fin.close();
        }

        cout << "\n\n================================================================================\n\n" << endl;

        if((fp=fopen(lgpl.c_str(), "r")) == NULL) {error("License file '%s' missing! It should be available in taste/licensing directory of the distribution package. Please look for it, because reading the license is an necessary condition for using TASTE!\n", lgpl.c_str()); print_help();}
        else
        {
          ifstream fin;
          fin.open(lgpl.c_str());
          while(!fin.eof()) {c=(char)fin.get(); if(c!=EOF) cout<<c;}
          fin.close();
        }

        exit(0);
     }

     if(cl.follow("", "-show") == "w")
     { 
        if((fp=fopen(warranty_liability.c_str(), "r")) == NULL) {error("File '%s' containing information about warranty and liability related to the software missing! It should be available in taste/licensing directory of the distribution package. Please look for it, because reading it is an necessary condition for using TASTE!\n", warranty_liability.c_str()); print_help();}
        else
        {
          ifstream fin;
          fin.open(warranty_liability.c_str());
          while(!fin.eof()) {c=(char)fin.get(); if(c!=EOF) cout<<c;}
          fin.close();
        }
        exit(0);
     }
  }

  if(!cl.search("-license_read")) 
  {
    cout << "CANNOT CONTINUE until license conditions are read!" << endl;
    cout << "After reading all license conditions related to TASTE, you are allowed to use 'license_read' flag. See help for more information, please." << endl;
    exit(0); 
  }

  if(cl.search("-i"))
  {
    fname4TA = cl.follow("", "-i");
    if((f_in = fopen(fname4TA.c_str(), "r")) == NULL) {error("Input file '%s' does not exist!\n", fname4TA.c_str()); print_help();}
  }
  else {if(fname4TA == "") {error("Input file not specified! Use '-i' option.\n"); print_help();}}

  freg = fname4TA + ".reg";

  if(cl.search("-debug")) {debug_mode = true; }

  if(cl.search("-scan")) { scan_mode = true; } 
   
  dsgnPtr = new clsDesign("top_level");
  cout << "Processing design named '" << dsgnPtr->getName() << "'..." << endl;

  strcpy(act_fname, string("").c_str());
  if(cl.search("-T"))
  {
    vector<string> vs;
    vector<string>::iterator vs_it;
    vs = cl.nominus_followers("-T");
    for(vs_it = vs.begin(); vs_it != vs.end(); vs_it++) loadTemplates((*vs_it).c_str(), dsgnPtr);
  }
  else loadTemplates("taste.templates", dsgnPtr);

  strcpy(act_fname, string("").c_str());
  if(f_in){loadNetlist(f_in, fname4TA.c_str(), dsgnPtr); fclose(f_in);}

  if(cl.search("-design")){ design_mode = true; }
  else if(cl.search("-cir"))
  {
    cirname = cl.follow("", "-cir");
    cir_mode = true;
  }

  if(scan_mode && cir_mode)
  {
      if(cl.search("-freg"))
      {
        freg = cl.follow("", "-freg");
        if((fopen(freg.c_str(), "r")) == NULL) {error("Input file '%s' containing list of registers does not exist!\n", freg.c_str()); print_help();}
        else {cout << "List of registers will be taken from explicit '" << freg << "' file." << endl;}
      }
      else {if(fname4TA != "") { cout << "List of registers will be taken from implicit '" << (freg = fname4TA + ".reg") << "' file." << endl; }}
      
      if(cl.search("-fscan"))
      {
        fscan = cl.follow("", "-fscan");
        if((fopen(fscan.c_str(), "r")) == NULL) {error("Input file '%s' containing scan notation does not exist!\n", fscan.c_str()); print_help();}
        else {cout << "Data related to scan will be loaded from explicit '" << fscan << "' file." << endl;}
      }
      else {if(fname4TA != "") { cout << "Data related to scan will be taken from implicit '" << (fscan = fname4TA + ".scan") << "' file." << endl; }}

      dsgnPtr->implementDft(cirname);
  }

  if(cl.search("-propen"))
  {
    str_tmp = cl.follow("", "-propen");
    if(str_tmp == "linear"){ mark_operation = PP_LIN; }
    else { mark_operation = PP_EXP; }
  }

  if(cl.search("-seqest"))
  {
    str_tmp = cl.follow("", "-seqest");
    if(str_tmp == "precise"){ clk_cnt = SE_PRECISE; }
    else { clk_cnt = SE_BASIC; }
  }

  if(cl.search("-ta"))
  {
    clsCircuit *cptr = dsgnPtr->findCir(cirname);
    if(!cptr) {error("Circuit '%s' for testability analysis does not exists!\n", cirname.c_str()); print_help();}
    else 
    {
      cout << "Testability analysis of circuit '" << cirname << "' started.";
      cptr->testability_analysis();
      cout << "\rTestability analysis of circuit '" << cirname << "' done with parameters:" << endl;
      cout << "...sequential length estimation used: " << (clk_cnt==1?"basic":"precise") << endl;
      cout << "...function used to evaluate propagation penalty: " << (mark_operation==1?"linear":"exponential") << endl;

      if(cl.search("-o"))
      {
        vector<string> vs;
        vector<string>::iterator vs_it;
        vs = cl.nominus_followers("-o");
        bool selected = false;
        for(vs_it = vs.begin(); vs_it != vs.end(); vs_it++) 
        {
          if((*vs_it) == "html") {cptr->export_TAresults_htm(fname4TA); selected=true;}
          if((*vs_it) == "tex") {cptr->export_TAresults_tex(fname4TA); selected=true;}
          if((*vs_it) == "txt") {cptr->export_TAresults_txt(fname4TA); selected=true;}
        }
        if(!selected) warn("Testability results not saved because of BAD information after '-o'\n");
      }
    }
  }
  else {if(fname4TA == "") {error("Input file not specified!\n"); print_help();}}

  // --- user-actions over the design
  // ---------------------------------------------------------------------------

  cout << "-------------------" << endl;
  cout << "Design area: " <<dsgnPtr->getArea() << endl;
  cout << "Design power: " <<dsgnPtr->getPower() << endl;

  if(dsgnPtr) delete(dsgnPtr);
  print_footer(0,0,0,0);
  return(0);
}

