/******************************************************************************
 *      "Copyright (C) 2006, ApS s.r.o Brno, All Rights Reserved"             *
 ******************************************************************************
 *============================================================================*
 *   Version �number�    Header/source file for the project   "file name"     *
 *----------------------------------------------------------------------------*
 * VERSION   DATE    TIME  BY   CHANGE/COMMENTS                               *
 *----------------------------------------------------------------------------*
 *============================================================================*/

#ifndef RES_DECLS_H_
#define RES_DECLS_H_

#include <string>
#include <sstream>
#include <pthread.h>

using namespace std;

#include "msiprotl.h"
#include "rstatl.h"

//============================================================
//									REGISTER
//============================================================
	//fast compiled simulator macros
#define reg_read(regname) (regname##_rstat++, regname)

#define reg_freeread(regname) (regname)

#define reg_write(regname, u) (\
	regname##_changed = true,\
	regname##_wstat++, regname = (u))

#define reg_freewrite(regname, u) (regname = (u))

#define reg_get_stat(regname) (TStat(regname##_rstat, regname##_wstat, 0))

#define reg_reset_stat(regname) (regname##_rstat = 0, regname##_wstat = 0)

#define reg_stat_str(regname) (({\
	stringstream ss (stringstream::in | stringstream::out);\
	ss << "\tvalue: " << regname << "\treads: " << regname##_rstat\
	   << "\twrites: " << regname##_wstat;\
	rstatl_obj()->_csretstr = ss.str();\
	}),rstatl_obj()->_csretstr)

// mas: 280407
// TODO: store all data as signed int

class TRegister
{
public:
	int size;      		 //bit size
	int data;     		 	 //space for data
	mutable int wstat;      //write counter
	mutable int rstat;      //read counter
	bool changed;			// was the value changed
	bool shared;				//flag to specify if resource is shared
	char *name;			//we need to name to be able to ask for resource by its name
	tResState flag;		// MSI register state for parallel simulation
	MSIL *msil;				//instnace of MSIL needed for reading/writing shared resources
	pthread_mutex_t guard; //mutex to prevent threads accesing shared resource at the same moment
    pthread_mutex_t flag_mutex;//flag guard
	
	
	TRegister();
	TRegister(bool shr, const char *rname,MSIL* msi);
	TRegister(int data, int ssize = -1);
	
	TRegister(const TRegister & src);
	
	TRegister& operator=(int d);
	TRegister& operator=(TRegister d);
	
	
	bool operator==(int d);
	bool operator!=(int d);
	bool operator<(int d);
	bool operator>(int d);
	bool operator<=(int d);
	bool operator>=(int d);
	
	friend TRegister operator&(const TRegister &x, const int y);
		
	int read();
	int free_read();
	void write(int d);
	void free_write(int d);
	
	const char * get_stat();
	
	TStat dbg_get_stat();
	void dbg_reset_stat();
	
	int as_int();
	
	int get_data();
	void set_data(int d);
	//flag function
	tResState get_flag();
	void set_flag(tResState flag);
		//comparing operator overloading
	bool operator<(const TRegister &x);
	bool operator>(const TRegister &x);
	bool operator<=(const TRegister &x);
	bool operator>=(const TRegister &x);
	bool operator==(const TRegister &x);
	bool operator!=(const TRegister &x);
	
	  //logical operator overloading
	friend TRegister operator&&(const TRegister &x, const TRegister &y);
	friend TRegister operator||(const TRegister &x, const TRegister &y);
	TRegister operator!();
	
		//aritmetic operator overloading
	friend TRegister operator+(const TRegister &x, const TRegister &y);
	friend TRegister operator-(const TRegister &x, const TRegister &y);
	friend TRegister operator*(const TRegister &x, const TRegister &y);
	friend TRegister operator/(const TRegister &x, const TRegister &y);
	friend TRegister operator%(const TRegister &x, const TRegister &y);
	
	TRegister& operator+=(const TRegister& x);
	TRegister& operator-=(const TRegister& x);
	TRegister& operator*=(const TRegister& x);
	TRegister& operator/=(const TRegister& x);
	TRegister& operator%=(const TRegister& x);
	
	TRegister& operator++();		//prefix
	TRegister operator++(int);	//postfix
	TRegister& operator--();
	TRegister operator--(int);
	
		//bit operator overloading
	friend TRegister operator&(const TRegister &x, const TRegister &y);
	friend TRegister operator|(const TRegister &x, const TRegister &y);
	friend TRegister operator^(const TRegister &x, const TRegister &y);
	friend TRegister operator<<(const TRegister &x, const TRegister &y);
	friend TRegister operator>>(const TRegister &x, const TRegister &y);
	TRegister operator~();

	TRegister& operator&=(const TRegister& x);
	TRegister& operator|=(const TRegister& x);
	TRegister& operator^=(const TRegister& x);
	TRegister& operator<<=(const TRegister& x);
	TRegister& operator>>=(const TRegister& x);
};

int as_int(TRegister reg);
int as_int(int);

//============================================================
//						REGISTER FILE
//============================================================
	//fast compiled simulator macros
#define filereg_read(fregname,i) \
	(({unsigned int _csind = (i);\
	fregname##_rstat[(_csind) - fregname##_low]++; \
	rstatl_obj()->_csretval = fregname[(_csind) - fregname##_low];}),rstatl_obj()->_csretval)

#define filereg_freeread(fregname,i) (fregname[(i)-fregname##_low])

#define filereg_write(fregname,i,d) ({\
	unsigned int _csind = (i);\
	fregname##_wstat[_csind - fregname##_low]++;\
	fregname[_csind - fregname##_low] = (d);})

#define filereg_freewrite(fregname,i,d) (fregname[(i) - fregname##_low] = (d))

#define filereg_get_stat(fregname,i) (({\
	unsigned int _csind = (i) - fregname##_low;\
	rstatl_obj()->_csrettstat = TStat(fregname##_rstat[_csind], fregname##_wstat[_csind], 0);}),\
	rstatl_obj()->_csrettstat)
	
#define filereg_reset_stat(fregname,i) ({\
	unsigned int _csind = (i) - fregname##_low;\
	fregname##_rstat[_csind] = 0; fregname##_wstat[_csind] = 0;})

#define filereg_stat_str(fregname,i) (({\
	unsigned int _csind = (i) - fregname##_low;\
	stringstream ss (stringstream::in | stringstream::out);\
	ss << "\tvalue: " << fregname[_csind] << "\treads: " << fregname##_rstat[_csind]\
	   << "\twrites: " << fregname##_wstat[_csind];\
	rstatl_obj()->_csretstr = ss.str();\
	}),rstatl_obj()->_csretstr)

class TRegisterFile
{
public:
	int size;      //bit size
	TRegister *data;
	int low;       //lower index
	int high;      //higher index
	
	TRegisterFile(int ssize, int lowi, int highi);
	
	TRegister& operator[](const TRegister& i);
	
	TRegister& read(const TRegister& i);
	TRegister& free_read(const TRegister& i);
	void write(const TRegister& i, const TRegister& d);
	void free_write(const TRegister& i, const TRegister& d);
	
	TStat dbg_get_stat(unsigned int addr);
	void dbg_reset_stat(unsigned int addr);
	
	const char * get_stat(int i);
};
//============================================================
//									MEMORY
//============================================================
// prototypes for mapping class
class ISACTMap
{
public:	
	virtual int execute(unsigned int addr)=0;
};

class ISACTMapFastSim: public ISACTMap{
public:	
	virtual int read(unsigned int addr)=0;
	virtual int free_read(unsigned int addr)=0;
	virtual void write(unsigned int addr, int data)=0;
	virtual void free_write(unsigned int addr, int data)=0;
	virtual TStat dbg_get_stat(unsigned int addr)=0;
	virtual void dbg_reset_stat(unsigned int addr)=0;	
};

class ISACTMapSlowSim: public ISACTMap
{
public:
	virtual TRegister& operator[](const TRegister& adr)=0;
	virtual TRegister& read(const TRegister& adr)=0;
	virtual TRegister& free_read(const TRegister& adr)=0;
	virtual void write(const TRegister& adr, const TRegister& data)=0;
	virtual void free_write(const TRegister& adr, const TRegister& data)=0;
	virtual TStat dbg_get_stat(unsigned int addr)=0;
	virtual void dbg_reset_stat(unsigned int addr)=0;
};
	
	//fast compiled simulator macros
#define mem_read(memname,i) (({\
	unsigned int _csind = (i);\
	memname##_rstat[_csind - memname##_low]++;\
	rstatl_obj()->_csretval = memname[_csind - memname##_low];}), rstatl_obj()->_csretval )

#define mem_freeread(memname,i) (memname[(i)-memname##_low])

#define mem_write(memname,i,d) ({\
	unsigned int _csind = (i);\
	memname##_wstat[_csind - memname##_low]++;\
	memname[_csind - memname##_low] = (d);})

#define mem_freewrite(memname,i,d) (memname[(i) - memname##_low] = (d))

#define mem_execute(memname,i) (({\
	unsigned int _csind = (i);\
	memname##_estat[_csind - memname##_low]++;\
	rstatl_obj()->_csretval = memname[_csind - memname##_low];}), rstatl_obj()->_csretval)

#define mem_get_stat(memname,i) (({\
	unsigned int _csind = (i) - memname##_low;\
	rstatl_obj()->_csrettstat = TStat(memname##_rstat[_csind], memname##_wstat[_csind], memname##_estat[_csind]);}),\
	rstatl_obj()->_csrettstat)
	
#define mem_reset_stat(memname,i) ({\
	unsigned int _csind = (i) - memname##_low;\
	memname##_rstat[_csind] = 0, memname##_wstat[_csind] = 0, memname##_estat[_csind] = 0;})

#define mem_stat_str(memname,i) (({\
	unsigned int _csind = (i) - memname##_low;\
	stringstream ss (stringstream::in | stringstream::out);\
	ss << "\tvalue: " << memname[_csind] << "\treads: " << memname##_rstat[_csind]\
	   << "\twrites: " << memname##_wstat[_csind] << "\texecutes: " << memname##_estat[_csind];\
	rstatl_obj()->_csretstr = ss.str();\
	}),rstatl_obj()->_csretstr)

class TMemory
{
public:
                  //memory parameters
  int size;   
  int block_size;
  int sub_block_size;
  int cnt_blocks;
  bool flags[3];
  int low_addr_ind;
  int high_addr_ind;
  int latency;
  
  int mem_size;
  TRegister *data;				//memory values
  int *exstat;
  
  TMemory(int ssize, int bl_size, int sblock_size, int bCount,
          bool fl1,bool fl2, bool fl3, int low, int high, int lat);
  
  TRegister& operator[](const TRegister& i);
  
  TRegister& read(const TRegister& i);
  TRegister& free_read(const TRegister& i);
  void write(const TRegister& i, const TRegister& d);
  void free_write(const TRegister& i, const TRegister& d);
  int execute(unsigned int i);
  
  unsigned int get_mem_size();
  
  TStat dbg_get_stat(unsigned int addr);
  void dbg_reset_stat(unsigned int addr);
  
  const char * get_stat(int i);
};

//============================================================
//                               CACHE
//============================================================
#define cache_freeread(cname, i, memname) (memname[(i)-memname##_low])

#define cache_read(cname, address, mem_name) (({\
	unsigned int addr = (address);\
	if (cname##_counter == ((unsigned int)-1)) {\
		cname##_counter = 0;\
		for (unsigned int i = 0; i < cname##_csize; ++i) {\
			cname##_time[i] = 0;\
		}\
	}\
	++cname##_counter;\
	int set_start = (addr / (cname##_mem_size / cname##_set_count)) * cname##_asociativity;\
	unsigned int free_addr;\
	bool free = false;\
	unsigned int lru_min = cname##_counter;\
	unsigned int lru_addr;\
	bool found = false;\
	for (unsigned int i=set_start; i<(set_start + cname##_asociativity); ++i) {\
		if (cname##_time[i] < lru_min) {\
			lru_min = cname##_time[i];\
			lru_addr = i;\
		}\
		if (!cname##_full[i]) {\
			free = true;\
			free_addr = i;\
		} else if ((addr) == cname##_addr[i]) {\
			cname##_hits++;\
			cname##_time[i] = cname##_counter;\
			rstatl_obj()->_csretval = cname[i];\
			found = true;\
			break;\
		}\
	}\
	if (!found) {\
	++cname##_misses;\
	if (free) {\
		cname##_addr[free_addr] = addr;\
		cname[free_addr] = (mem_name##_rstat[addr - mem_name##_low]++, mem_name[addr - mem_name##_low]);\
		cname##_dirty[free_addr] = false;\
		cname##_full[free_addr] = true;\
		cname##_time[free_addr] = cname##_counter;\
		rstatl_obj()->_csretval = cname[free_addr];\
		found = true;\
	}\
	if (!found) {\
	if (cname##_LRU) {\
		free_addr = lru_addr;\
	} else {\
		free_addr = set_start + ((int)((rand() / (float)RAND_MAX) * cname##_asociativity));\
	}\
	\
	if (cname##_dirty[free_addr]) {\
		mem_name##_wstat[cname##_addr[free_addr] - mem_name##_low]++;\
		mem_name[cname##_addr[free_addr] - mem_name##_low] = cname[free_addr];\
	}\
	cname##_addr[free_addr] = addr;\
	cname[free_addr] = (mem_name##_rstat[addr - mem_name##_low]++, mem_name[addr - mem_name##_low]);\
	cname##_dirty[free_addr] = false;\
	cname##_time[free_addr] = cname##_counter;\
	rstatl_obj()->_csretval = cname[free_addr];\
	}\
	}\
  }),rstatl_obj()->_csretval)

//===================================================================   
#define cache_freewrite(cname,address,dat,memname) (memname[(address) - memname##_low] = (dat))
   
#define cache_write(cname,address,dat,memname) ({\
	unsigned int addr = (address);\
	unsigned int d = (dat);\
    if (cname##_counter == ((unsigned int)-1)) {\
		cname##_counter = 0;\
		for (unsigned int i = 0; i < cname##_csize; ++i) {\
			cname##_time[i] = 0;\
		}\
	}\
	++cname##_counter;\
	int set_start = (addr / (cname##_mem_size / cname##_set_count)) * cname##_asociativity;\
	unsigned int free_addr;\
	bool free = false;\
	unsigned int lru_min = cname##_counter;\
	unsigned int lru_addr;\
	bool found = false;\
	for (unsigned int i=set_start; i<(set_start + cname##_asociativity); ++i) {\
		if (cname##_time[i] < lru_min) {\
			lru_min = cname##_time[i];\
			lru_addr = i;\
		}\
		if (!cname##_full[i]) {\
			free = true;\
			free_addr = i;\
		} else if (addr == cname##_addr[i]) {\
			cname##_hits++;\
			cname[i] = d;\
			if (cname##_WB) {\
				memname##_wstat[addr - memname##_low]++;\
 				memname[addr - memname##_low] = d;\
				cname##_dirty[i] = false;\
			} else cname##_dirty[i] = true;\
			cname##_time[i] = cname##_counter;\
			found = true;\
			break;\
		}\
	}\
	if (!found) {\
	++cname##_misses;\
	if (cname##_WA) {\
		if (!free) {\
			if (cname##_LRU) {\
				free_addr = lru_addr;\
			} else {\
				free_addr = set_start + ((int)((rand() / (float)RAND_MAX) * cname##_asociativity));\
			}\
			if (cname##_dirty[free_addr]) \
				memname##_wstat[cname##_addr[free_addr] - memname##_low]++;\
 				memname[cname##_addr[free_addr] - memname##_low] = cname[free_addr];\
		}\
	\
		cname[free_addr] = d;\
		if (cname##_WB) {\
			memname##_wstat[addr - memname##_low]++;\
			memname[addr - memname##_low] = d;\
			cname##_dirty[free_addr] = false;\
		} else cname##_dirty[free_addr] = true;\
		cname##_addr[free_addr] = addr;\
		cname##_full[free_addr] = true;\
		cname##_time[free_addr] = cname##_counter;\
	} else {\
		memname##_wstat[addr - memname##_low]++;\
		memname[addr - memname##_low] = d;\
	}\
	}\
	})
	
//===============================================================
	
#define cache_execute(cname,address,memname) (({\
	unsigned int addr = (address);\
    if (cname##_counter == ((unsigned int)-1)) {\
		cname##_counter = 0;\
		for (unsigned int i = 0; i < cname##_csize; ++i) {\
			cname##_time[i] = 0;\
		}\
	}\
	++cname##_counter;\
	int set_start = (addr / (cname##_mem_size / cname##_set_count)) * cname##_asociativity;\
	unsigned int free_addr;\
	bool free = false;\
	unsigned int lru_min = cname##_counter;\
	unsigned int lru_addr;\
	bool found = false;\
	for (unsigned int i=set_start; i<(set_start + cname##_asociativity); ++i) {\
		if (cname##_time[i] < lru_min) {\
			lru_min = cname##_time[i];\
			lru_addr = i;\
		}\
		if (!cname##_full[i]) {\
			free = true;\
			free_addr = i;\
		} else if (addr == cname##_addr[i]) {\
			cname##_hits++;\
			cname##_time[i] = cname##_counter;\
			memname##_estat[addr - memname##_low]++;\
			rstatl_obj()->_csretval = cname[i];\
			found = true;\
			break;\
		}\
	}\
	if (!found) {\
	++cname##_misses;\
	if (free) {\
		cname##_addr[free_addr] = addr;\
 		cname[free_addr] = memname[addr - memname##_low];\
		memname##_estat[addr - memname##_low]++;\
		cname##_dirty[free_addr] = false;\
		cname##_full[free_addr] = true;\
		cname##_time[free_addr] = cname##_counter;\
		rstatl_obj()->_csretval = cname[free_addr];\
		found = true;\
	}\
	if (!found) {\
	if (cname##_LRU) {\
		free_addr = lru_addr;\
	} else {\
		free_addr = set_start + ((int)((rand() / (float)RAND_MAX) * cname##_asociativity));\
	}\
	\
	if (cname##_dirty[free_addr]) {\
		memname##_wstat[cname##_addr[free_addr] - memname##_low]++;\
		memname[cname##_addr[free_addr] - memname##_low] = cname[free_addr];\
	}\
	cname##_addr[free_addr] = addr;\
	memname##_estat[addr - memname##_low]++;\
	cname[free_addr] = memname[addr - memname##_low];\
	cname##_dirty[free_addr] = false;\
	cname##_time[free_addr] = cname##_counter;\
	rstatl_obj()->_csretval = cname[free_addr];\
	}\
	}\
	}),rstatl_obj()->_csretval)

#define cache_get_stat(cname, i, memname) (({\
	unsigned int _csind = (i) - memname##_low;\
	rstatl_obj()->_csrettstat = TStat(memname##_rstat[_csind], memname##_wstat[_csind], memname##_estat[_csind]);}),\
	rstatl_obj()->_csrettstat)
	
#define cache_reset_stat(cname, i, memname) ({\
	unsigned int _csind = (i) - memname##_low;\
	memname##_rstat[_csind] = 0; memname##_wstat[_csind] = 0; memname##_estat[_csind] = 0;})
	
#define cache_hit_stats(cname) (cname##_hits)
#define cache_misses_stats(cname) (cname##_misses)

#define cache_stat_str(cname) (({\
	stringstream ss (stringstream::in | stringstream::out);\
	ss << "hits: " << cname##_hits << "\tmisses: " << cname##_misses;\
	rstatl_obj()->_csretstr = ss.str();\
	}), rstatl_obj()->_csretstr)

  //types of cache strategies
typedef enum {
	WA_POLICY = 0,
	WB_POLICY = 1,
	LRU_POLICY = 2,
	TSTRATEGY_TYPE_SIZE
} Tstrategy;

class TCache {
public:
  TRegister *data;      	 //local memory values
  unsigned int *c_addr;    //addresses of localy stored memory data
  bool *c_free;						 //identifies full/empty cache cell
  bool *c_dirty;					 //dirty bit
  unsigned int *c_time;		 //time stamp for LRU policy calculation
  int size;                //memory parameters
  int block_size;
  int sub_block_size;
  int cnt_blocks;
  bool flags[3];
  int low_addr_ind;
  int high_addr_ind;
  int latency;
     //cache parameters
  int asociativity;
  int lineSize;
  bool strategy[TSTRATEGY_TYPE_SIZE];
  int wb_size;
  TMemory *mem;            //pointer to asociated memory
  
  		//statistical variables
  int hits;
  int misses;
  
      //auxilary variables
  int cache_size;
  int set_count;
  unsigned int lru_counter;
  
  TCache(int ssize, int bl_size, int sblock_size, int b_count,
  bool fl1, bool fl2, bool fl3, int low, int high, int lat,
  int asoc, int line, bool wa_strat, bool wb_strat, bool lru_strat, int wb, TMemory *m);
  
  TRegister& operator[](const TRegister& address);
  
  TRegister& read(const TRegister& address);
  TRegister& free_read(const TRegister& address);
  void write(const TRegister& address, const TRegister& dat);
  void free_write(const TRegister& address, const TRegister& dat);
  int execute(unsigned int addr);
  
  TStat dbg_get_stat(unsigned int addr);
  void dbg_reset_stat(unsigned int addr);
  	
  const char * get_stat();
  const char * get_data_stat(int i);
};


class TPipeLineState
{
	public:
		TPipeLineState(const char* sID) :
			ID(sID),
			op("")
		{}
		TPipeLineState(const char* sID, int p_countLatch, int p_countState, ...);
		~TPipeLineState()
		{
			delete[] latch_regs;
			delete[] state_regs;
		}
		// registers witch can be used in state
		int countLatch;
		TRegister **latch_regs;
		int countState;
		TRegister **state_regs;
		// name of state
		string ID;	
		// how many times we used state
		int ustat;
		// flag if state is empty
		bool empty;
		// name of operation, wich is executed in state
		string op;
		
		void flush();
		void stall();
		void shift(TPipeLineState *state);
		
};

class TPipeLine
{
	public:
		TPipeLine(int count, ...);
		~TPipeLine() { delete[] pipeLineStates; }
		
		void stall(const char* state);
		void flush(const char* state);
		void shift(const char* state);
		
		int state_to_num(const char* state);
		
		int countState;
		TPipeLineState **pipeLineStates;
		
};
			

#endif /*RES_DECLS_H_*/

