#ifndef __UMF_PNP_SOLVER_H
#define __UMF_PNP_SOLVER_H

#include <vector>
#include <string>
#include <Eigen/Core>
#include <Eigen/Geometry>

namespace umf {

class Correspondence
{
public:
    Correspondence(float projectX, float projectY, float modelX, float modelY, float modelZ)
    {
        this->px = projectX;
        this->py = projectY;
        this->mx = modelX;
        this->my = modelY;
        this->mz = modelZ;
    }

    float px;
    float py;
    float mx;
    float my;
    float mz;

    inline bool operator<(const Correspondence &other) const
    {
        return this->mx < other.mx ||
                (this->mx == other.mx && (this->my < other.my ||
                 (this->my == other.my && this->mz < other.mz)));
    }
};

typedef std::vector<Correspondence> CorrespondenceSet;


enum PnPFlags {
    PNP_FLAG_COMPUTE_CAMERA = 1,
    PNP_FLAG_GL_PROJECTION_MV = 2
};

class PnPSolver
{
public:
    PnPSolver();
    virtual ~PnPSolver() {}

    void addCorrespondences(CorrespondenceSet correspondences);
    bool computeCameraPose(CorrespondenceSet correspondencesOrig, Eigen::Vector2i imageSize, int flags = PNP_FLAG_COMPUTE_CAMERA);

    void setCameraMatrix( Eigen::Matrix3d &cameraMatrix) { this->cameraMatrix = cameraMatrix; }
    Eigen::Matrix3d &getCameraMatrix() { return this->cameraMatrix; }

    void setDistortionCoeffs( Eigen::VectorXd &distCoeffs ) {this->distCoeffs = distCoeffs; }
    Eigen::VectorXd &getDistortionCoeffs() {return this->distCoeffs; }

    static PnPSolver *GetPnPSolver(std::string type = "EPNP");

    /**
     * @brief getWorldTransformMatrix - this is only rotation and transformation
     * @return
     */
    Eigen::Matrix4d &getWorldTransformMatrix() { return this->worldTransformMatrix; }

    /**
     * @brief getCameraTransformMatrix - this is only rotation and transformation of the camera - inverse of the #getWorldTransformMatrix
     * @return
     *
     * the #computeCameraPose has to be called with flag #PNP_FLAG_COMPUTE_CAMERA
     */
    Eigen::Matrix4d &getCameraTransformMatrix() { return this->cameraTransformMatrix; }


    /**
     * @brief getMVP get model view projection matrix for openGL based on the focal lengthes and
     * @return mvp
     *
     * the #computeCameraPose has to be called with flag #PNP_FLAG_GL_PROJECTION_MV
     */
    Eigen::Matrix4d &getMVP() { return this->mvp; }


    /**
     * @brief getCameraPos - get the camera position in 3D as if the world was fixed
     * @return
     *
     * The #computeCameraPose has to be called with flag #PNP_FLAG_COMPUTE_CAMERA
     */
    Eigen::Vector3d &getCameraPos() { return this->cameraPos; }

    /**
     * @brief getWorldPos - get the world translation relative to the camera
     * @return
     */
    Eigen::Vector3d &getWorldPos() { return this->worldPos; }

    /**
     * @brief getWorldQuaternion the world rotation only quaternion
     * @return
     */
    Eigen::Quaterniond &getWorldQuaternion() { return this->worldQuat; }

    /**
     * @brief getCameraQuaternion - get the world camera rotation quaternion
     * @return
     *
     * The #computeCameraPose has to be called with flag #PNP_FLAG_COMPUTE_CAMERA
     */
    Eigen::Quaterniond &getCameraQuaternion() { return this->cameraQuat; }

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

protected:

    virtual bool pnpSolve(CorrespondenceSet correspondences) = 0;

    Eigen::Matrix3d cameraMatrix;
    Eigen::VectorXd distCoeffs;

    Eigen::Matrix4d mvp;
    Eigen::Matrix4d worldTransformMatrix;
    Eigen::Matrix4d cameraTransformMatrix;
    Eigen::Vector3d cameraPos;
    Eigen::Vector3d worldPos;

    Eigen::Quaterniond cameraQuat;
    Eigen::Quaterniond worldQuat;

};



}
#endif // PNP_SOLVER_H
