#include "model.h"
#include "util/umfdebug.h"
#include "util/draw.h"
#include "util/corner_detector.h"
#include "util/distort.h"
#include "util/pnp_solver.h"
#include "umf.h"
#include <algorithm>

#ifdef UMF_USE_OPENCV
#include <opencv2/calib3d/calib3d.hpp>
#endif

namespace umf {

template<int NCHAN>
Model<NCHAN>::Model()
{
    this->marker = NULL;
    this->useCornerSearch = true;
    this->useSubPixel = true;
#ifdef UMF_USE_OPENCV
    this->pnp = PnPSolver::GetPnPSolver("OPENCV");
#else 
    this->pnp = PnPSolver::GetPnPSolver("EPNP");
#endif

    Eigen::Matrix3d cameraMatrix;
    float focal = 700;
    Eigen::Vector2f imgSize(640,480);
    cameraMatrix << focal, 0, imgSize[0]/2,
            0, focal, imgSize[1]/2,
            0, 0, 1;

    Eigen::VectorXd distCoeffs(8);
    distCoeffs << 0, 0, 0, 0, 0, 0, 0, 0;

    //desire hd
    focal = 1200;
    imgSize[0] = 1280;
    imgSize[1] = 720;

    cameraMatrix << focal, 0, imgSize[0]/2,
            0, focal, imgSize[1]/2,
            0, 0, 1;
    /*
    //ADAMOVA kamera <- odstranit ;)
    cameraMatrix << 1939.05, 0, 959.5,
            0, 1939.05, 539.5,
            0, 0, 1;
    distCoeffs << -0.228015, 0.360624, 0.0133008, 0.00272655, -0.169941, 0, 0, 0;
    */
    this->pnp->setCameraMatrix(cameraMatrix);
    this->pnp->setDistortionCoeffs(distCoeffs);
}


template<int NCHAN>
Model<NCHAN>::~Model()
{
    if(this->marker)
    {
        delete this->marker;
    }
}


template<int NCHAN>
void Model<NCHAN>::setCameraProperties(Eigen::Matrix3d &camMatrix, Eigen::VectorXd &distCoeffs)
{
    this->pnp->setCameraMatrix(camMatrix);
    this->pnp->setDistortionCoeffs(distCoeffs);
}

template<int NCHAN>
void Model<NCHAN>::addMarker(Marker<NCHAN> *marker)
{
    if(this->marker)
    {
        delete this->marker;
        this->marker = NULL;
    }
    this->marker = marker;
}


static inline int clamp(int x, int a, int b)
{
    return x < a ? a : (x > b ? b : x);
}

template<int NCHAN> template<class T>
bool Model<NCHAN>::matchModel(Image<T, NCHAN> *img, unsigned short subW, unsigned short subH,
                              std::vector< typename Marker<NCHAN>::DirectionType > &edgeDir,
                              std::vector< Eigen::Vector2f > &extractionPoints, Location &loc,
                              bool show)
{

    if(this->marker == NULL)
    {
        return false;
    }

    bool showCorners = false && show;
    bool showCorrespondences = true && show;

    //TODO maybe make a list of the corners instead of a map, that way we can add the position, too
    std::vector<unsigned char> corners;
    loc = this->marker->getLocation(subW, subH, edgeDir, corners);

    bool success = !(loc == Location::invalid);

    if(!success)
    {
        return success;
    }

#ifdef UMF_DEBUG
    if(showCorners)
    {
        this->showCorners(corners, extractionPoints, subW, subH);
    }//show
#endif

    //now we have our location and good corners -> find the corresponding corner points in the image, and match them to 3d points
    success = this->matchPoints(img, loc, extractionPoints, corners, subW, subH);

    if(!success)
    {
        return success;
    }

#ifdef UMF_DEBUG
    if(showCorrespondences)
    {
        this->showCorrespondences();
    }//show
#endif

    return success;
}

template<int NCHAN>
bool Model<NCHAN>::computeHomography(bool show)
{

#ifdef UMF_DEBUG
	if(show)
	{
		this->showCorrespondences(true);
	}
#endif
#ifdef UMF_USE_OPENCV
    cv::Mat srcPoints(this->correspondences.size(), 1, CV_32FC2);
    cv::Mat dstPoints(this->correspondences.size(), 1, CV_32FC2);
    for(int i = 0; i < this->correspondences.size(); i++)
    {
        srcPoints.at<cv::Point2f>(i) = cv::Point2f(this->correspondences[i].mx, this->correspondences[i].my);
        dstPoints.at<cv::Point2f>(i) = cv::Point2f(this->correspondences[i].px, this->correspondences[i].py);
    }
    cv::Mat HCV = cv::findHomography(srcPoints, dstPoints);

    for(int i = 0; i < 3; i++)
    {
        for(int j = 0; j < 3; j++)
        {
            this->homography(i, j) = HCV.at<double>(i, j);
        }
    }
    
    //this->homography << 0, 0, 0,  0, 0, 0,  0, 0, 0;
    return true;
#else
    return false;
#endif
}

template<int NCHAN> template<class T>
bool Model<NCHAN>::computeCameraPosition(Image<T, NCHAN> *img, unsigned short subW, unsigned short subH,
                                         Location &loc, bool refine, bool show)
{

    bool success = this->pnp->computeCameraPose(this->correspondences, Eigen::Vector2i(img->width, img->height), PNP_FLAG_COMPUTE_CAMERA | PNP_FLAG_GL_PROJECTION_MV);

    if(refine && success)
    {
        Eigen::MatrixXi visited(this->marker->h, this->marker->w);
        Eigen::MatrixXi mask(this->marker->h, this->marker->w);
        Location rloc = loc;
        changeBackLocation(rloc, this->marker->w, this->marker->h, subW, subH);

        int xmin = (std::max)(rloc.c, 0);
        int xmax = (std::min)(rloc.c + subW, (int)this->marker->w);
        int ymin = (std::max)(rloc.r, 0);
        int ymax = (std::min)(rloc.r + subH, (int) this->marker->h);
        visited.setZero();
        visited.block(ymin, xmin, ymax - ymin, xmax - xmin).setOnes();
        int stepWidth = 3;
        int iterationCount = 5;
        for(int iterCount = 0; iterCount < iterationCount; iterCount++)
        {
            xmin = (std::max)(xmin - stepWidth, 0);
            xmax = (std::min)(xmax + stepWidth, (int) this->marker->w);
            ymin = (std::max)(ymin - stepWidth, 0);
            ymax = (std::min)(ymax + stepWidth, (int) this->marker->h);
            mask = visited;
            visited.block(ymin, xmin, ymax-ymin, xmax-xmin).setOnes();
            mask = visited - mask;

            int count = this->getNewPositions(img, mask);

            if(count == 0)
            {
                break;
            }
        }
#ifdef UMF_DEBUG
        if(show)
        {
            this->showCorrespondences(true);
        }
#endif

        success = this->pnp->computeCameraPose(this->correspondences, Eigen::Vector2i(img->width, img->height), PNP_FLAG_COMPUTE_CAMERA | PNP_FLAG_GL_PROJECTION_MV);
    }


#ifdef UMF_DEBUG
    if(success && show)
    {
        this->showBox();
    }
#endif

    return success;
}


template<int NCHAN> template<class T>
int Model<NCHAN>::getNewPositions(Image<T,NCHAN> *img, const Eigen::MatrixXi &mask)
{
    int counter = 0;
    std::vector<Eigen::Vector3f> modelPoints(4);
    std::vector<Eigen::Vector2f> projPoints(4);
    std::vector<Location> locations;

    std::vector<Eigen::Vector2f> projCorners;
    std::vector<Eigen::Vector3f> modelCorners;

    for(int row = 0; row < mask.rows(); row++)
    {
        for(int col = 0; col < mask.cols(); col++)
        {
            if(mask(row,col) == 0){
                continue;
            }
            counter++;

            Location p; p.c = col; p.r = row; p.rotation = LOC_ROT_0;
            Eigen::Vector2f cornerLoc;
            typename Marker<NCHAN>::DirectionType edges[4];

            int type = this->marker->getCornerType(p, cornerLoc, edges);

            if(type == CORNER_TYPE_NONE)
            {
                continue;
            }

            modelPoints[0] = Eigen::Vector3f(cornerLoc[0], cornerLoc[1], 0.0f);
            modelPoints[1] = Eigen::Vector3f(cornerLoc[0] + 1.f, cornerLoc[1], 0.0f);
            modelPoints[2] = Eigen::Vector3f(cornerLoc[0] + 1.f, cornerLoc[1] + 1.f, 0.0f);
            modelPoints[3] = Eigen::Vector3f(cornerLoc[0], cornerLoc[1] + 1.f, 0.0f);

            //we want the position in the 3d model - corners are +0.5f moved
            cornerLoc[0] += 0.5f;
            cornerLoc[1] += 0.5f;

            this->projectPoints(modelPoints, projPoints, Eigen::Vector2i(img->width, img->height), true, true);

            Eigen::Vector2f corner = findMarkerCorner(img, projPoints, type);
            if((corner.array() >= 0).all())
            {
                if(this->useSubPixel)
                {
                    projCorners.push_back(corner);
                    modelCorners.push_back(Eigen::Vector3f(cornerLoc[0], cornerLoc[1], 0));
                } else {
                    this->correspondences.push_back(Correspondence(corner[0], corner[1], cornerLoc[0], cornerLoc[1], 0));
                }
            }
        }
    }


    //if subpixel search is on
    if(this->useSubPixel)
    {
        counter = findCornersSubpixel(img, projCorners);

        if(counter > 0)
        {
            for(unsigned int ind = 0; ind < projCorners.size(); ind++)
            {
                if(projCorners[ind][0] >= 0)
                {
                    this->correspondences.push_back(Correspondence(projCorners[ind][0], projCorners[ind][1], modelCorners[ind][0], modelCorners[ind][1], modelCorners[ind][2]));
                }
            }
        }
    }
    return counter;
}

template <int NCHAN>
void Model<NCHAN>::showBox()
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();
    if(imgDbg != NULL)
    {
        std::vector<Eigen::Vector2f> points;
        std::vector<Eigen::Vector3f> modelPoints(8);

        int w = 4;

        modelPoints[0] = Eigen::Vector3f(0, 0, 0);
        modelPoints[1] = Eigen::Vector3f(0, w, 0);
        modelPoints[2] = Eigen::Vector3f(w, w, 0);
        modelPoints[3] = Eigen::Vector3f(w, 0, 0);
        modelPoints[4] = Eigen::Vector3f(0, 0, -w);
        modelPoints[5] = Eigen::Vector3f(0, w, -w);
        modelPoints[6] = Eigen::Vector3f(w, w, -w);
        modelPoints[7] = Eigen::Vector3f(w, 0, -w);

        this->projectPoints(modelPoints, points, Eigen::Vector2i(imgDbg->width, imgDbg->height), true, false);

        Eigen::Vector3i lineColor(255, 199, 95);
        Eigen::Vector3i lineColorY(0, 255, 0);
        Eigen::Vector3i lineColorX(255, 0, 0);
        Eigen::Vector3i lineColorZ(0, 0, 255);


        drawLine(imgDbg, points[0].template cast<int>(), points[1].template cast<int>(), lineColorY, 4 ); //y
        drawLine(imgDbg, points[0].template cast<int>(), points[3].template cast<int>(), lineColorX, 4 ); //x
        drawLine(imgDbg, points[0].template cast<int>(), points[4].template cast<int>(), lineColorZ, 4); //Z

        for(int i = 1; i < 3; i++)
        {
            drawLine(imgDbg, points[i].template cast<int>(), points[(i+1)%4].template cast<int>(), lineColor, 4 ); //floor
        }
        for(int i = 1; i < 4; i++)
        {
            drawLine(imgDbg, points[i].template cast<int>(), points[i + 4].template cast<int>(), lineColor, 4); //sides
        }
        for(int i = 0; i < 4; i++)
        {
            drawLine(imgDbg, points[4+ i].template cast<int>(), points[4 + (i+1)%4].template cast<int>(), lineColor, 4);//roof
        }

    }
}

template <int NCHAN>
void Model<NCHAN>::showCorrespondences(bool refine)
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();
    if(imgDbg != NULL)
    {
        Eigen::Vector3i lineColor(0, 255, 0);
        int linewidth = 2;

        if(refine)
        {
            lineColor = Eigen::Vector3i(150, 0, 200);
            linewidth = 1;
        }
        for(CorrespondenceSet::iterator it = this->correspondences.begin(); it != this->correspondences.end(); it++)
        {
            drawCircle(imgDbg, Eigen::Vector2i(it->px, it->py), 5, lineColor, linewidth);
        }
    }
}

template <int NCHAN>
void Model<NCHAN>::showCorners(std::vector<unsigned char> &corners, std::vector< Eigen::Vector2f > &extractionPoints, unsigned short subW, unsigned short subH)
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();
    if(imgDbg != NULL)
    {
        int rows = subH - 1;
        int cols = subW - 1;
        for(int row = 0; row < rows; row++)
        {
            for(int col = 0; col < cols; col++)
            {
                int pindex = row*(cols+1) + col;
                unsigned char type = corners[row*cols + col];
                Eigen::Vector3i color(255, 0, 0);
                switch(type)
                {
                case CORNER_TYPE_CROSS:
                    color = Eigen::Vector3i(0, 255, 255);
                    break;
                case CORNER_TYPE_NONE:
                    color = Eigen::Vector3i(0, 0, 255);
                    break;
                case CORNER_TYPE_LEFT_TOP:
                    color = Eigen::Vector3i(255, 255, 0);
                    break;
                case CORNER_TYPE_LEFT_BOTTOM:
                    color = Eigen::Vector3i(255, 0, 0);
                    break;
                case CORNER_TYPE_RIGHT_BOTTOM:
                    color = Eigen::Vector3i(255, 0, 0);
                    break;
                case CORNER_TYPE_RIGHT_TOP:
                    color = Eigen::Vector3i(255, 255, 0);
                    break;
                default:
                    color = Eigen::Vector3i((type - CORNER_TYPE_LEFT_TOP)*63, 200, 0);
                }

                if(type != CORNER_TYPE_NONE)
                {
                    drawCircle(imgDbg,
                               ((extractionPoints[pindex] + extractionPoints[pindex+1] + extractionPoints[pindex + cols + 1] + extractionPoints[pindex + cols + 2])*0.25).template cast<int>(),
                            8, color, 1);
                }
            }//cols
        }//rows
    }//imgDbg
}


template<int NCHAN> template<class T>
bool Model<NCHAN>::matchPoints(Image<T, NCHAN> *img, Location &loc,
                               std::vector< Eigen::Vector2f > &extractionPoints,
                               std::vector<unsigned char> &corners,
                               unsigned short subW,
                               unsigned short subH)
{
    this->correspondences.clear();

    //////////////////////////////////////////////////////////////////////////
    // using corner search
    if(useCornerSearch)
    {
        std::vector<Eigen::Vector2f> projCorners;
        std::vector<Eigen::Vector3f> modelCorners;
        for(int row = 0; row < subW - 1; row++)
        {
            for(int col = 0; col < subH - 1; col++)
            {
                int cornerIndex = row*(subW - 1) + col;
                if(corners[cornerIndex] == CORNER_TYPE_NONE)
                {
                    continue;
                }

                //get model pos
                Location currLoc = loc;
                currLoc.c += col;
                currLoc.r += row;
                changeBackLocation(currLoc, this->marker->w, this->marker->h, 2, 2);

                //now we got the actual position - in order for the model's 0,0 to be at the marker's edge
                Eigen::Vector2f cornerLoc(currLoc.c + 1.0f, currLoc.r + 1.0f);

                //great, now we have the model position, find the more precise image position

                std::vector<Eigen::Vector2f> subPos(4);

                //get the four points around the corner
                for(int subRow = 0; subRow < 2; subRow++)
                {
                    for(int subCol = 0; subCol < 2; subCol++)
                    {
                        int sid = (row + subRow)*subW + col + subCol;
                        subPos[subRow*2 + subCol] = extractionPoints[sid];
                    }
                }

                std::swap(subPos[2], subPos[3]);

                Eigen::Vector2f corner = findMarkerCorner(img, subPos, corners[cornerIndex]);
                if((corner.array() >= 0).all())
                {
                    if(this->useSubPixel)
                    {
                        projCorners.push_back(corner);
                        modelCorners.push_back(Eigen::Vector3f(cornerLoc[0], cornerLoc[1], 0));
                    } else {
                        correspondences.push_back(Correspondence(corner[0], corner[1], cornerLoc[0], cornerLoc[1], 0));
                    }
                }
            }

        }

        //if subpixel search is on
        if(this->useSubPixel)
        {
            int count = findCornersSubpixel(img, projCorners);

            if(count > 0)
            {
                for(unsigned int ind = 0; ind < projCorners.size(); ind++)
                {
                    if(projCorners[ind][0] >= 0)
                    {
                        correspondences.push_back(Correspondence(projCorners[ind][0], projCorners[ind][1], modelCorners[ind][0], modelCorners[ind][1], modelCorners[ind][2]));
                    }
                }
            }
        }
    } else {
        ////////////////////////////////////////////////////////////////////////////////////
        // not using corner search, just get the good points

        for(int row = 0; row < subW - 1; row++)
        {
            for(int col = 0; col < subH - 1; col++)
            {
                bool addPoint = false;
                //test if some of the corners around are good points and it's OK to add this
                for(int subRow = 0; subRow < 2; subRow++)
                {
                    for(int subCol = 0; subCol < 2; subCol++)
                    {
                        int y = (std::max)((std::min)(row - subRow,subH - 1), 0);
                        int x = (std::max)((std::min)(col - subCol,subW - 1), 0);

                        int cornerIndex = y*(subW - 1) + x;
                        if(corners[cornerIndex] != CORNER_TYPE_NONE)
                        {
                            addPoint = true;
                            subCol = subRow = 3;
                            break;
                        }
                    }
                }

                if(!addPoint)
                {
                    continue;
                }

                //get model pos
                Location currLoc = loc;
                currLoc.c += col;
                currLoc.r += row;
                changeBackLocation(currLoc, this->marker->w, this->marker->h);

                Eigen::Vector2f imagePoint = extractionPoints[row*subW + col];
                //shift by 0.5f so that the origin is at the marker's edge, not the field center
                this->correspondences.push_back(Correspondence(imagePoint[0], imagePoint[1], currLoc.c + 0.5f, currLoc.r + 0.5f, 0.0f));
            }
        }
    }

    return this->correspondences.size() > 4; //the minimum we need for camera localization
}

template<int NCHAN>
void Model<NCHAN>::projectPoints(std::vector<Eigen::Vector3f> &modelPoints, std::vector<Eigen::Vector2f> &imagePoints, Eigen::Vector2i imageSize, bool distort, bool swap)
{
    std::vector<Eigen::Vector2f> points;
    Eigen::Matrix4d &mv = this->pnp->getWorldTransformMatrix();
    Eigen::Matrix3d &camera = this->pnp->getCameraMatrix();
    Eigen::Vector4d currentPos;
    Eigen::Vector3d projected;
    double k = 1;
    for(unsigned int i = 0; i < modelPoints.size(); i++)
    {
        //switch x and y, cos we do that
        Eigen::Vector4d pos3d(modelPoints[i][0], modelPoints[i][1], modelPoints[i][2], 1.0);
        if(swap)
        {
            //when we want the mask coordinate system, not the real 3d coordinate system, we have to do this
            std::swap(pos3d[0], pos3d[1]);
        }
        currentPos = mv*pos3d;

        if(distort) //the distortion will compute the multiplication with the camer matrix, so we only need the transformed point
        {
            //the current position probably is still going to be normalized, since we don't do any projection in the model view matrix
            //but just in case:
            currentPos *= 1.0/currentPos[3];

            //now that we have this if i get this right we just do 1/z in projection, so (the camera matrix has 1 at the diagonal for the z point
            k = currentPos[2] ? 1./currentPos[2] : 1;
            points.push_back(Eigen::Vector2f(currentPos[0]*k, currentPos[1]*k));
        } else {
            //if we are going to just project it move everything to the imagePoints at once
            projected = camera*Eigen::Vector3d(currentPos[0], currentPos[1], currentPos[2]);

            k = projected[2] ? 1./projected[2] : 1;

            imagePoints.push_back(Eigen::Vector2f(projected[0]*k, projected[1]*k));

        }

    }

    //now we do some kind of distorting based on opencl. It will also scale the points based on camera matrix
    if(distort)
    {
        distortPoints(points, imagePoints, camera.template cast<float>(), this->pnp->getDistortionCoeffs().template cast<float>());
    }

    //transform back the y
    for(unsigned int i = 0; i < imagePoints.size(); i++)
    {
        imagePoints[i](1) = imageSize[1] - imagePoints[i](1);
    }
}

template<int NCHAN>
int Model<NCHAN>::getCameraPosRot(double cameraPos[3], double rotationQuat[4])
{
    Eigen::Vector3d ecameraPos = this->pnp->getCameraPos();
    Eigen::Quaterniond ecameraQuat = this->pnp->getCameraQuaternion();

    cameraPos[0] = ecameraPos[0];
    cameraPos[1] = ecameraPos[1];
    cameraPos[2] = ecameraPos[2];

    rotationQuat[1] = ecameraQuat.x();
    rotationQuat[2] = ecameraQuat.y();
    rotationQuat[3] = ecameraQuat.z();
    rotationQuat[0] = ecameraQuat.w();
    return EXIT_SUCCESS;
}

template<int NCHAN>
int Model<NCHAN>::getWorldPosRot(double cameraPos[3], double rotationQuat[4])
{
    Eigen::Vector3d eWorldPos = this->pnp->getWorldPos();
    Eigen::Quaterniond eWorldQuat = this->pnp->getWorldQuaternion();

    cameraPos[0] = eWorldPos[0];
    cameraPos[1] = eWorldPos[1];
    cameraPos[2] = eWorldPos[2];

    rotationQuat[1] = eWorldQuat.x();
    rotationQuat[2] = eWorldQuat.y();
    rotationQuat[3] = eWorldQuat.z();
    rotationQuat[0] = eWorldQuat.w();
    return EXIT_SUCCESS;
}

template<int NCHAN>
void Model<NCHAN>::updateMaskTracker(Eigen::Vector2i imgSize, int flags)
{
    std::vector<Eigen::Vector3f> modelPoints;
    bool local = (flags & UMF_FLAG_TRACK_LOCAL_POS) != 0;
    bool use_homography = (flags & UMF_FLAG_HOMOGRAPHY) != 0;
    if(local)
    {
        int minX = this->marker->w;
        int minY = this->marker->h;
        int maxX = -1;
        int maxY = -1;
        for(int i = 0; i < this->correspondences.size(); i++)
        {
            if(correspondences[i].mx < minX)
            {
                minX = correspondences[i].mx;
            }
            if(correspondences[i].my < minY)
            {
                minY = correspondences[i].my;
            }
            if(correspondences[i].mx > maxX)
            {
                maxX = correspondences[i].mx;
            }
            
            if(correspondences[i].my > maxY)
            {
                maxY = correspondences[i].my;
            }
        }
        minX--; minY--;
        maxX++; maxY++;
        
        modelPoints.push_back(Eigen::Vector3f(minX, minY, 0));
        modelPoints.push_back(Eigen::Vector3f(maxX, minY, 0));
        modelPoints.push_back(Eigen::Vector3f(maxX, maxY, 0));
        modelPoints.push_back(Eigen::Vector3f(minX, maxY, 0));
    } else {
        modelPoints.push_back(Eigen::Vector3f(0, 0, 0));
        modelPoints.push_back(Eigen::Vector3f(this->marker->w, 0, 0));
        modelPoints.push_back(Eigen::Vector3f(this->marker->w, this->marker->h, 0));
        modelPoints.push_back(Eigen::Vector3f(0, this->marker->h, 0));
        
    }

    std::vector<Eigen::Vector2f> projPoints;
    if(use_homography)
    {
        for(int i = 0; i < modelPoints.size(); i++)
        {
            Eigen::Vector3d pp = this->homography*Eigen::Vector3d(modelPoints[i][0], modelPoints[i][1], 1.0);
            pp /= pp[2];
            projPoints.push_back(Eigen::Vector2f(pp[0], pp[1]));
        }
    } else {
        this->projectPoints(modelPoints, projPoints, imgSize, false, true);
    }

    this->maskTracker.update(projPoints);

}

//INSTANCING
template Model<1>::Model();
template Model<3>::Model();


template Model<1>::~Model();
template Model<3>::~Model();

template void Model<1>::addMarker(Marker<1> *marker);
template void Model<3>::addMarker(Marker<3> *marker);

template int Model<1>::getCameraPosRot(double pos[3], double quat[4]);
template int Model<3>::getCameraPosRot(double pos[3], double quat[4]);
template int Model<1>::getWorldPosRot(double pos[3], double quat[4]);
template int Model<3>::getWorldPosRot(double pos[3], double quat[4]);

template void Model<1>::setCameraProperties(Eigen::Matrix3d &cameraMatrix, Eigen::VectorXd &distCoeffs);
template void Model<3>::setCameraProperties(Eigen::Matrix3d &cameraMatrix, Eigen::VectorXd &distCoeffs);

template bool Model<1>::matchModel(ImageGray *img, unsigned short subW, unsigned short subH,
                                   std::vector< typename Marker<1>::DirectionType > &edgeDir,
                                   std::vector< Eigen::Vector2f > &extractionPoints, Location &loc, bool show);
template bool Model<3>::matchModel(ImageRGB *img, unsigned short subW, unsigned short subH,
                                   std::vector< typename Marker<3>::DirectionType > &edgeDir,
                                   std::vector< Eigen::Vector2f > &extractionPoints, Location &loc, bool show);

template bool Model<1>::computeHomography(bool show);
template bool Model<3>::computeHomography(bool show);
template bool Model<1>::computeCameraPosition(ImageGray *img, unsigned short subW, unsigned short subH, Location &loc, bool refine, bool show);
template bool Model<3>::computeCameraPosition(ImageRGB *img, unsigned short subW, unsigned short subH, Location &loc, bool refine, bool show);

template void Model<1>::updateMaskTracker(Eigen::Vector2i imgSize, int flags);
template void Model<3>::updateMaskTracker(Eigen::Vector2i imgSize, int flags);

} //namespace
