#include "defines.h"
#include "edgel_detector.h"
#include "util/umfdebug.h"
#include "util/draw.h"
#include "util/corner_detector.h"

#include <Eigen/Dense>

namespace umf {

EdgelDetector::EdgelDetector()
{
    this->adaptiveThreshold = 15;
    this->scanlineStep = 25;
    this->scanlineWindow = 10;

    this->edgelStep = 5;
    this->edgelThreshold = 10;
    this->edgelRadius = 7;
    this->edgelLengthThreshold = 15;
    this->edgelDotThreshold = std::cos(3.14/6); //+-30 degree of freedom
}


template<class T, int CHAN>
static void scan(Image<T, CHAN> *input,
                 std::vector<Eigen::Vector2i> &points,
                 ImageGray *mask,
                 const int step,
                 bool vertical,
                 const int bufferSize,
                 const int adaptiveThreshold)
{
    std::vector< Eigen::Matrix<float, CHAN, 1> > buffer(bufferSize);
    unsigned short windowEnd = 0;

    float inv_size = 1.0f/(bufferSize);
    Eigen::Matrix<float, CHAN, 1> mean;
    mean.setZero();

    int x,y;

    int *dirStep, *dirScan;
    int dirStepSize, dirScanSize;

    if(vertical)
    {
        dirStep = &x;
		dirStepSize = input->width - bufferSize;
        dirScan = &y;
        dirScanSize = input->height - bufferSize;
    } else {
        dirStep = &y;
        dirStepSize = input->height - bufferSize;
        dirScan = &x;
        dirScanSize = input->width - bufferSize;
    }


    for(*dirStep=step/2; *dirStep < dirStepSize; (*dirStep) +=step)
    {
        windowEnd = 0;
        mean *= 0;
        //first fill up our buffer - probably clutter - don't really care
        for(*dirScan = 0; *dirScan < bufferSize; (*dirScan)++)
        {
            Eigen::Matrix<float, CHAN, 1> curr = input->get2De(x, y).template cast<float>();

            buffer[windowEnd] = curr;
            windowEnd++;
            mean += curr;
        }
#ifdef UMF_DEBUG_COUNT_PIXELS
        UMFDSingleton::Instance()->addPixels(bufferSize);
#endif

        windowEnd = 0;

        mean *= inv_size;

        bool edgeFound = false;
        int maskedSize = -1;
        Eigen::Vector2i edgeStart(0, 0);
        //now scan the line
#ifdef UMF_DEBUG_COUNT_PIXELS
        UMFDSingleton::Instance()->addPixels(dirScanSize - bufferSize);
#endif
        for(*dirScan = bufferSize; *dirScan < dirScanSize; (*dirScan)++)
        {
            Eigen::Matrix<float, CHAN, 1> curr = input->get2De(x, y).template cast<float>();

            bool masked = (mask == NULL || (*(mask->get2D(x, y)) == 255));
            if(masked && maskedSize <= 0)
            {

                Eigen::Matrix<float, CHAN, 1> diff = mean - curr;
                if( diff.dot(diff) > adaptiveThreshold*adaptiveThreshold)
                {
                    if(!edgeFound)
                    {
                        //good we found and edge
                        edgeFound = true;
                        edgeStart[0] = x;
                        edgeStart[1] = y;
                    }
                } else if (edgeFound)
                {
                    edgeFound = false;
                    Eigen::Vector2i middle = edgeStart;

                    //store the index
                    points.push_back(middle);

                }

            } else {
                edgeFound = false;
            }
            if(!masked)
            {
                maskedSize = bufferSize;
            } else {
                maskedSize--;
            }

            mean -= buffer[windowEnd]*inv_size;
            mean += curr*inv_size;
            buffer[windowEnd] = curr;
            windowEnd = (windowEnd + 1) % bufferSize;
        }
    }

}

/**
 * @brief Detect edges along scanlines horizontally and vertically.
 *
 * \param image The input image for detection
 * \param mask Optional parameter (for example for masking out regions - like greenscreen
 *      The size of the image should be the same as for the input image.
 * \param show Show the scanline in the debug output if it is enabled - see image below
 *
 * \tparam T type of the image (unsigned char,float etc.)
 * \tparam NCHAN number of channels
 *
 * Detect along scanlines with \link EdgelDetector::setScanlineStep \endlink shift with adaptive threshod
 * \link EdgelDetector::setAdaptiveThreshold \endlink over a window sized \link EdgelDetector::setScanlineWindow \endlink.
 * \image html 1_scanline.png
 *
 */
template<class T, int NCHAN>
void EdgelDetector::detectEdges(Image<T, NCHAN> *image, ImageGray *mask, bool show)
{
    this->points.clear();
    //scanline detection
    //first vertical pass
    scan(image, this->points, mask, this->scanlineStep, true, this->scanlineWindow, this->adaptiveThreshold);
    //then horizontal pass
    scan(image, this->points, mask, this->scanlineStep, false, this->scanlineWindow, this->adaptiveThreshold);

#ifdef UMF_DEBUG
    if(show)
    {
        UMFDebug *dbg = UMFDSingleton::Instance();
        ImageRGB *imgDbg = dbg->getImage();

        if(imgDbg)
        {
            Eigen::Vector3i lineColor(235, 235, 90);
            int lineWidth = 3;

			for(int row = this->scanlineStep/2; row < imgDbg->height - this->scanlineWindow; row += this->scanlineStep)
            {
                drawLine(imgDbg, Eigen::Vector2i(0, row), Eigen::Vector2i(imgDbg->width, row), lineColor, lineWidth);

            }
			for(int col = this->scanlineStep/2; col < imgDbg->width - this->scanlineWindow; col += this->scanlineStep)
            {
                drawLine(imgDbg, Eigen::Vector2i(col, 0), Eigen::Vector2i(col, imgDbg->height), lineColor, lineWidth);
            }

            Eigen::Vector3i paintColor(255, 0, 0);
            //Eigen::Vector3i paintColor(255, 255, 0);
            for(std::vector<Eigen::Vector2i>::iterator pit = points.begin(); pit != points.end(); pit++)
            {
                drawCircle(imgDbg, *pit, lineWidth*2, paintColor, -1);
            }
        }
    }
#endif

}

///////////////////////////////////////////////////////////////////////////////////
////EDGEL search

void debugDrawLine(Eigen::Vector2i p1, Eigen::Vector2i p2, Eigen::Vector3i color)
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();

    if(imgDbg)
    {
        drawLine(imgDbg, p1, p2, color, 1);
    }
}

template <typename T> int sgn(T val) {
    return (T(0) < val) - (T(0) >= val);
}

/**
 * @brief Get the edge direction in horizontal and vertical direction
 * @param input the input image
 * @param x The x position in the image
 * @param y The y position in the image
 * @param h The gradient size in horizontal direction (signed)
 * @param v The gradient size in vertical direction (signed)
 * @tparam T the type of the image
 * @tparam NCHAN the number of channels in the image
 *
 * This method simply takes the channel with the maximum gradient in vertical and horizontal
 * directions separately.
 */
template<class T, int NCHAN>
void getSobelResponseMax(Image<T, NCHAN> *input, int x, int y, float &h, float &v)
{
    Eigen::Matrix<float, NCHAN, 1> AB = input->get2De(x, (y-1)).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> BA = input->get2De((x-1), y).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> BC = input->get2De((x+1), y).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> CB = input->get2De(x, (y+1)).template cast<float>();
#ifdef UMF_DEBUG_COUNT_PIXELS
    UMFDSingleton::Instance()->addPixels(4);
#endif

    Eigen::Matrix<float, NCHAN, 1>  horiz = (CB - AB);
    Eigen::Matrix<float, NCHAN, 1>  vert = (BC - BA);

    int row = 0;
    int col = 0;
    h = horiz.array().abs().maxCoeff(&row, &col);
    h *= sgn(horiz[col]);

    v = vert.array().abs().maxCoeff(&row, &col);
    v *= sgn(vert[col]);

}

//

/**
 * @brief Get the edge direction in horizontal and vertical direction
 * @param input the input image
 * @param x The x position in the image
 * @param y The y position in the image
 * @param h The gradient size in horizontal direction (signed)
 * @param v The gradient size in vertical direction (signed)
 * @tparam T the type of the image
 * @tparam NCHAN the number of channels in the image
 *
 * This operator is based on Edge Detection of Color Images Using Directional Operators (J. Scharcanski and A. N. Venetsanopoulos)
 * Separates H+ and H- and in vertical direction V+ and V- (each vector based on the number of channels)
 *
 * The resulting horizontal is:
 * h = | ||H+ - H-|| if ||H+|| > ||H-||
 *     | -||H+ - H-|| else
 * Similarly in the vertical direction
 */
template<class T, int NCHAN>
void getSobelResponseRGB(Image<T, NCHAN> *input, int x, int y, float &h, float &v)
{
    Eigen::Matrix<float, NCHAN, 1> AB = input->get2De(x, (y-1)).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> BA = input->get2De((x-1), y).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> BC = input->get2De((x+1), y).template cast<float>();
    Eigen::Matrix<float, NCHAN, 1> CB = input->get2De(x, (y+1)).template cast<float>();
#ifdef UMF_DEBUG_COUNT_PIXELS
    UMFDSingleton::Instance()->addPixels(4);
#endif

    Eigen::Matrix<float, NCHAN, 1>  horiz = (CB - AB);
    Eigen::Matrix<float, NCHAN, 1>  vert = (BC - BA);

    h = sgn(CB.norm() - AB.norm())*horiz.norm();
    v = sgn(BC.norm() - BA.norm())*vert.norm();
}

/**
 * @brief Finds a connected edges and creates a continues line
 * @param image The input image
 * @param edgel The found edgel is stored here
 * @param point The seeding point where the search is started
 * @tparam T the type of the image
 * @tparam NCHAN the number of channels
 * @return if the edgel search was successful
 *
 * The method is demonstrated on the image below:
 * \image html 2_edgelgrow.png
 * A good example is the seed point under the mouse. The search is separated into two steps - up (reddish) and down (greenish) lines
 * In both directions the algorithm goes like this:
 *  -# get the edge direction estimate ( \link getSobelResponseRGB \endlink, \link getSobelResponseMax \endlink)
 *  -# based on the edge direction jump by a given number of pixels ( \link setEdgelSearchStep \endlink )
 *  -# perpendicular to the edge direction find and edge ( \link setEdgelSearchRadius \endlink ),
 *     In this step binary search is used between the two endpoints. ( \link setEdgelSearchThreshold \endlink must be the minimum distance between the endPoints)
 *     The lines between the endpoints are the short lines perpendicular to the longer lines in the image.
 *  -# if an edge is found the gradient is tested at the given point, so it is similar to the normal direction ( \link setEdgelDotThreshold \endlink)
 *  -# the normal end line direction is updated if the edge was found and use this as a seed and jump to 1.
 *  -# otherwise stop
 *
 * After in both up and down directions the edges were found the two endpoints are connected,
 * and if the edgel is long enough ( \link setEdgelLengthThreshold \endlink ) true is returned
 * and the edgel parameter is set.
 */
template<class T, int NCHAN>
bool EdgelDetector::findEdgel(Image<T, NCHAN> *image,
               Edgel &edgel,
               const Eigen::Vector2i &point, ImageGray *mask)
{
    float h, v;
    //try using L1 norm
    getSobelResponseRGB(image, point[0], point[1], h, v);

    Eigen::Vector2f normal(v, h);
    normal.normalize();
    Eigen::Vector2f edgeDirection(-normal[1], normal[0]);

    //cvLine(draw, up, down, cvScalar(255, 0, 0), 1);
    int upCount = 0;
    int downCount = 0;
    Eigen::Vector2f step;
    Eigen::Vector2i edgeTmp;
    Eigen::Vector2i edge1 = point;
    Eigen::Vector2i edge2 = point;

    int edgelLength = 0;

    //upwards direction
    do{
        step = (upCount + 1)*this->edgelStep*edgeDirection;
        Eigen::Vector2f next = edge1.template cast<float>() + step;

        bool found = binSearchEdge(image, edgeTmp,
                                   (next + normal*this->edgelRadius).template cast<int>(),
                                   (next - normal*this->edgelRadius).template cast<int>(),
                                   this->edgelThreshold);

        if(!found)
        {
            break;
        } else if(mask != NULL && *mask->get2D(edgeTmp[0], edgeTmp[1]) != 255)
        {
            break;
        }

        //filter based on edge gradient
        getSobelResponseRGB(image, edgeTmp[0], edgeTmp[1], h, v);
        Eigen::Vector2f normalTmp(v, h);
        normalTmp.normalize();
        if(std::abs(normalTmp.dot(normal)) < this->edgelDotThreshold)
        {
            break;
        }

        //update normal and edge direction
        edge1 = edgeTmp;
        edgeDirection = (edge1 - point).template cast<float>();
        edgeDirection.normalize();
        normal[0] = -edgeDirection[1];
        normal[1] = edgeDirection[0];

        edgelLength += (upCount + 1)*this->edgelStep;
        upCount++;
    } while(true);

    //downwards
    do{
        step = (downCount + 1)*this->edgelStep*edgeDirection;
        Eigen::Vector2f next = edge2.template cast<float>() - step;

        bool found = binSearchEdge(image, edgeTmp,
                                   (next + normal*this->edgelRadius).template cast<int>(),
                                   (next - normal*this->edgelRadius).template cast<int>(),
                                   this->edgelThreshold);

        if(!found)
        {
            break;
        } else if(mask != NULL && *mask->get2D(edgeTmp[0], edgeTmp[1]) != 255)
        {
            break;
        }

        //filter based on edge gradient
        getSobelResponseRGB(image, edgeTmp[0], edgeTmp[1], h, v);
        Eigen::Vector2f normalTmp(v, h);
        normalTmp.normalize();
        if(std::abs(normalTmp.dot(normal)) < this->edgelDotThreshold)
        {
            break;
        }

        //update normal and edge direction
        edge2 = edgeTmp;
        edgeDirection = (edge1 - edge2).template cast<float>();
        edgeDirection.normalize();
        normal[0] = -edgeDirection[1];
        normal[1] = edgeDirection[0];

        edgelLength += (downCount + 1)*this->edgelStep;
        downCount++;
    } while(true);

    //if the limits are fine
    if(edgelLength >= this->edgelLengthThreshold)
    {
        //we found an edge in both directions, hurray - get the line
        edgel.endPoints[0] = edge1.template cast<float>();
        edgel.endPoints[1] = edge2.template cast<float>();
        edgel.line = Eigen::Vector3f(normal[0], normal[1], - normal.dot(edgel.endPoints[0]));
        edgel.score = edgelLength;
        return true;
    }

    return false;
}

/**
 * @brief Find edgels using the detected edges as seedpoints
 * @tparam T type of the image
 * @tparam NCHAN the number of channels in the image
 * @param image the input image
 * @param show If debugging is enabled can be used to draw the edgels to debug output
 *
 * The result are edgels \link umf::Edgel \endlink that are further used to detect the grid.
 * For each detected edge the \link findEdgels \endlink is run.
 * \image html 3_edgels.png "The detected edgels for scanline step"
 */
template<class T, int NCHAN>
void EdgelDetector::findEdgels(Image<T, NCHAN> *image, ImageGray* mask, bool show)
{
    this->edgels.clear();

    for(unsigned int i = 0; i != this->points.size(); i++)
    {
        Edgel e;
        bool res = this->findEdgel(image, e, this->points[i], mask);
        if(res)
        {
            this->edgels.push_back(e);
        }
    }

#ifdef UMF_DEBUG
    if(show)
    {
        this->showEdgels();
    }
#endif

}

void EdgelDetector::showEdgels()
{
    UMFDebug *dbg = UMFDSingleton::Instance();
    ImageRGB *imgDbg = dbg->getImage();

    if(imgDbg)
    {
        Eigen::Vector3i lineColor(0, 255, 0);
        //Eigen::Vector3i lineColor(255, 0, 0);
        int lineWidth = 2;

        for(std::vector<Edgel>::iterator it = this->edgels.begin(); it != this->edgels.end(); it++)
        {
            drawLine(imgDbg, Eigen::Vector2i(it->endPoints[0][0], it->endPoints[0][1]),
                    Eigen::Vector2i(it->endPoints[1][0], it->endPoints[1][1]), lineColor, lineWidth);
        }
    }
}

template void EdgelDetector::detectEdges(ImageRGB *image, ImageGray *mask, bool show);
template void EdgelDetector::detectEdges(ImageGray *image, ImageGray *mask, bool show);

template void EdgelDetector::findEdgels(ImageRGB *image, ImageGray *mask, bool show);
template void EdgelDetector::findEdgels(ImageGray *image, ImageGray *mask, bool show);

}
