#include <iostream>
#include <fstream>
#include <vector>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>

using namespace std;

typedef struct
{
    char data[512];
    uint32_t len;
    uint32_t addr;
    int type; // 0 normal, 1 addr, -1 no record
} t_record;

typedef char t_srec[600];
typedef uint8_t t_brec[512];

typedef struct
{
    char* ifile;
    int sel_ifile;
    char* ofile;
    int sel_ofile;
    int sel_help;
} params_t;

int32_t srec_line = -1;

/**
 * \brief Print help message.
 */
void usage() {
    printf("\
Usage: srec2brec options\n\
-i srec  Input SREC encoded file.\n\
-o brec  Output BREC encoded file.\n\
-h       This help.\n\
");
}

/**
 * \brief Parse command line parameters.
 * \param argc Number of command line parameters.
 * \param argv Command line parameters.
 * \param params Parsed command line parameters.
 * \return EXIT_SUCCESS if parameters ware parsed OK, EXIT_FAILURE if error 
 *         occured.
 */
int parse_cmdline(int argc, char *argv[], params_t *params)
{
    char c;         /* getopt */
    
    memset(params, 0, sizeof(params_t)); // Set all selected flags to 0
    
    // Parse the command line arguments
    while ((c = getopt(argc, argv, "hi:o:")) != -1) {
        switch (c) {
        case 'h':
            params->sel_help = 1;
            break;
        case 'i':
            params->sel_ifile = 1;
            params->ifile = optarg;
            break;
        case 'o':
            params->sel_ofile = 1;
            params->ofile = optarg;
            break;
        default:
            cerr << "Unknown parameter: " << c << "!"<< endl;
            return EXIT_FAILURE;
            break;
        }
    }
    
    return EXIT_SUCCESS;
}

// Convert one hex digit to uint32_t
uint32_t inline hexdigit2val(char val)
{
    uint32_t ret = 0;
    if (val >= '0' && val <= '9')
        ret = val - '0';
    else if (val >= 'A'  && val <= 'F')
        ret = val - 'A' + 10;
    
    return ret;
}

// get byte from string
uint8_t inline get_byte(char* iter)
{
    return hexdigit2val(*iter) << 4 | hexdigit2val(*++iter);
}

// get 32 bit word from string
uint32_t inline get_word(char* iter)
{
    return hexdigit2val(*iter)   << 28 | hexdigit2val(*++iter) << 24 |
           hexdigit2val(*++iter) << 20 | hexdigit2val(*++iter) << 16 |
           hexdigit2val(*++iter) << 12 | hexdigit2val(*++iter) <<  8 |
           hexdigit2val(*++iter) <<  4 | hexdigit2val(*++iter);
}

uint8_t compute_fcs(uint8_t* iter, uint32_t count, uint32_t init)
{
    uint32_t fcs = init;
    uint32_t i = 0;
    
    for (i = 0; i < count; i++)
    {
        fcs += *iter;
        iter += 1;
    }

    fcs = (~fcs) & 0xFF;
    return fcs;
}

uint32_t check_fcs(char* iter, uint32_t count)
{
    uint32_t fcs = 0;
    uint32_t i = 0;
    
    for (i = 0; i < count; i++)
    {
        fcs += get_byte(iter);
        iter += 2;
    }

    fcs = (~fcs) & 0xFF;
    return (fcs != get_byte(iter)) ? 1 : 0;
}


int parse_line(t_srec srec, t_record &rec)
{
 
    char* type;
    char* iter;
    uint32_t count;
    uint32_t ret;
    uint32_t i;
    uint32_t addr;
    
    srec_line++;
    
    // Set iterator initial position
    iter = (char*)srec;
    
    // Check if line begin with S
    if (*iter++ != 'S')
    {
        cerr << "SREC line " << srec_line << " doesn't start with 'S'!" << endl;
        return EXIT_FAILURE;
    }
    
    // Save type of srecord
    type = iter++;

    // Get count of bytes
    count = get_byte(iter);
        
    // Check srecord check sum
    if (check_fcs(iter, count) == 1)
    {
        cerr << "SREC line " << srec_line << " checksum mismatch!" << endl;
        return EXIT_FAILURE;
    }
    // Skip the count
    iter += 2;

    // Different behavior for srec types
    switch (*type)
    {
        // Record types 0 and 5 are ignored
        case '0':
        case '5':
            rec.type = -1;
            break;
        // Load data from Record type 3 to memory
        case '3':
            // Get address
            addr = get_word(iter);
            iter += 8;
            // Load data
            for(i = 0; i < count - 4; i++)
            {
                rec.data[i] = get_byte(iter);
                iter += 2;
            }
            rec.addr = addr;
            rec.type = 0;
            rec.len = count - 5;
            break;
        case '7':
           rec.addr = get_word(iter);
           rec.type = 1;
           break;
        // Other types of SRECORD cause error
        default:
            cerr << "SREC line " << srec_line << ": Unknown SREC type " << *type << "!" << endl;
            return EXIT_FAILURE;
            break;
    }
    return EXIT_SUCCESS;
}

int parse_srec(params_t *params,  vector<t_record> &srecords)
{
    ifstream ifs(params->ifile , ifstream::in);
    while(ifs.good())
    {
        t_record rec;
        t_srec srec;
        int ret;
        
        ifs.getline(srec, 600);
        if (strlen(srec) > 4){
            ret = parse_line(srec, rec);
            if (ret != EXIT_SUCCESS)
            {
                return ret;
            }
            if (rec.type >= 0)
            {
                srecords.push_back(rec);
            }
        }
    }
    ifs.close();
    return EXIT_SUCCESS;
}

int srec2brec(vector<t_record> &srecords, vector<t_record> &brecords)
{
    
    uint32_t len = srecords[0].len;
    uint32_t start_addr = srecords[0].addr;
    uint32_t curr_addr = start_addr;
    uint32_t start_addr_call;
    
    for (int i = 1; i < srecords.size(); i++)
    {
        if (srecords[i].type == 0)
        {
            len += srecords[i].len;
            
            if (srecords[i-1].addr + srecords[i-1].len != srecords[i].addr)
            {
                cerr << "There is hole in SREC file at line: " << i << "!" << endl;
                cerr << "-1 Addr: " << srecords[i-1].addr << " -1 Len: " << srecords[i-1].len << " 0 Addr: " << srecords[i].addr << endl;
                return EXIT_FAILURE;
            }
        }
    }
    
    if (len % 506 != 0)
        len = ((len / 506) + 1)*506;
    
    int8_t * mem = new int8_t[len];
    int8_t * itr = mem;
    
    for (int i = 0; i < srecords.size(); i++)
    {
        if (srecords[i].type == 0)
        {
            memcpy(itr, srecords[i].data, srecords[i].len);
            itr += srecords[i].len;
        }
        else
        {
            start_addr_call = srecords[i].addr;
        }
    }

    itr = mem;
    for (int i = 0; i < len / 506; i++)
    {
       t_record rec;
       rec.type = 0;
       rec.addr = curr_addr;
       rec.len = 506;
       memcpy(rec.data, itr, 506);
       brecords.push_back(rec);
       itr += 506;
       curr_addr += 506;
    }
    
    t_record rec;
    rec.type = 1;
    rec.addr = start_addr_call;
    rec.len = 506;
    brecords.push_back(rec);
    
    delete[] mem;
    return EXIT_SUCCESS;
}

int save_brec(params_t *params,  vector<t_record> brecords)
{
    ofstream ofs(params->ofile , ofstream::out | ofstream::binary | ofstream::trunc);
    for (int i = 0; i < brecords.size(); i++)
    {
        uint8_t header = 0;
        uint8_t fcs = 0;
        uint8_t *iter = (uint8_t*)&(brecords[i].addr);
        uint32_t a[4];
        a[0] = (*iter++);
        a[1] = (*iter++);
        a[2] = (*iter++);
        a[3] = (*iter++);
        header = brecords[i].type << 7;
        header |= 'B';
        fcs = compute_fcs((uint8_t*)&(brecords[i].data), 506, a[0] + a[1] + a[2] + a[3]);
        ofs.write((const char*)&header, 1);
        ofs.write((const char*)&(brecords[i].addr), 4);
        ofs.write((const char*)&(brecords[i].data), 506);
        ofs.write((const char*)&fcs, 1);
    }
    ofs.close();
    return EXIT_SUCCESS;
}

int main(int argc, char *argv[])
{
    params_t params; // Command line parameters
    
    if (argc == 1)
    {
        usage();
        return EXIT_SUCCESS;
    }
    
    if (parse_cmdline(argc, argv, &params) == EXIT_FAILURE)
        return EXIT_FAILURE;
    
    // If -h parameter is selected print usage and stop
    if (params.sel_help != 0)
    {
        usage();
        return EXIT_SUCCESS;
    }
    
    if (params.sel_ifile && params.sel_ofile)
    {
        int ret;
        vector<t_record> srecords;
        vector<t_record> brecords;
        
        ret = parse_srec(&params, srecords);
        if (ret != EXIT_SUCCESS)
        {
            cerr << "Error while parsing input file!" << endl;
            return ret;
        }
        ret = srec2brec(srecords, brecords);
        if (ret != EXIT_SUCCESS)
        {
            return ret;
        }
        ret = save_brec(&params, brecords);
        if (ret != EXIT_SUCCESS)
        {
            cerr << "Error while saving output file!" << endl;
            return ret;
        }
    }
    else
    {
        if (!params.sel_ifile)
        {
            cerr << "Parameter -i is missing!" << endl;
        }
        if (!params.sel_ofile)
        {
            cerr << "Parameter -o is missing!" << endl;
        }
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}

