/*!
 * \file ethctl_cmd.c
 * \brief Tool for controlling axi-ethernet - Axi-ethernet control 
 *         and communication commands
 * \author Vlastimil Kosar <ikosar@fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: filterctl.c 1469 2012-05-23 15:59:18Z xkekel00 $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <linux/ethtool.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <linux/sockios.h>
#include "ethctl_param.h"
#include "ethctl_cmd.h"
#include "hwio.h"

/*
 * Compatibility table with supported components
 */
struct hwio_comp_spec comp_compat_tab[] = {
//     {"xlnx", ETHERNET_NAME, HWIO_VER3(1, 0, HWIO_VERSION_NA)},
    {"xlnx", ETHERNET_NAME, HWIO_VER3(3, 0, HWIO_VERSION_NA)},
    HWIO_COMPAT_END
};

/**
 * \brief Initialize communication with axi-ethernet.
 * \param dev HWIO device structure.
 * \param fn Device tree path.
 * \param mem Mem device path.
 * \return Status of operation
 */
int ethernet_init(struct hwio_dev *dev, const char *fn, const char *mem) {
    int retcode = hwio_init(dev, mem, fn);
    if (retcode != HWIO_EOK)
    {
        fprintf(stderr,"ERROR: Unable to initialize axi-ethernet!\n");
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}


/**
 * \brief Deinitialize communication with axi-ethernet.
 * \param dev HWIO device structure.
 * \param head Component pointer.
 */
void ethernet_deinit(struct hwio_dev *dev, struct hwio_comp *head) {
    hwio_comp_list_free(head);
    hwio_deinit(dev);
}

/**
 * \brief Read 32 bit word from component.
 * \param comp Component pointer.
 * \param addr Address.
 * \return Read data from component.
 */

uint32_t ethernet_read(struct hwio_comp *comp, uint32_t addr)
{
    uint32_t data;
    hwio_comp_read(comp, addr, &data);
    return data;
}

/**
 * \brief Find ethernet components.
 * \param dev HWIO device structure.
 * \param comp Component pointer.
 * \param params Command line arguments.
 */
int ethernet_find_components(struct hwio_dev *dev, struct hwio_comp **comp, params_t *params)
{   
    int retcode;
    if (params->sel_name != 0)
    {
        retcode = hwio_byname(dev, params->name, comp_compat_tab, comp);
    }
    else
    {
        retcode = hwio_byspec(dev, comp_compat_tab, comp);
    }
    
    if (retcode != HWIO_EOK)
    {
        fprintf(stderr,"ERROR: Unable to enumerate axi-ethernet components!\n");
        return EXIT_FAILURE;
    }
    if (*comp == NULL)
    {
        fprintf(stderr,"WARNING: No axi-ethernet component found!\n");
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}

/**
 * \brief Print axi-ethernet component.
 * \param comp Component pointer.
 * \param ifc Interface number.
 */
void ethernet_print_component(struct hwio_comp *comp, int ifc)
{
    struct hwio_version version = hwio_comp_version(comp);
    printf("[%d:] [%s:%s:%s] [%d.%d.%d]", ifc, hwio_comp_vendor(comp),
        hwio_comp_type(comp), hwio_comp_name(comp),
        version.major, version.minor, version.subminor);
    putchar(' ');
//     if (hwio_comp_addr_space_available(comp))
//         printf("[0x%X:0x%X]", hwio_comp_base(comp),
//             hwio_comp_range(comp));
//     else
//         printf("[N/A]");
    putchar('\n');
}

/**
 * \brief Print status and configuration of axi-ethernet.
 * \param stat Status structure.
 */
void print_status(stat_t *stat)
{
    printf("========== AXI ETHERNET %d ==========\n", stat->id);
    printf("Core version: %s %u.%u.%u\n", (stat->version.type > 0) ? "SOFT" : "HARD", stat->version.major, stat->version.minor, stat->version.patch);
    printf("%s\n", (stat->driver_data != NULL) ? stat->driver_data : "Driver: N/A Version: N/A");
    printf("--------- Core capabilities ---------\n");
    printf("%s %s %s %s %s\n", ((stat->ability & MAC_MASK) > 0) ? "MAC FILTERS," : "", ((stat->ability & STAT_MASK) > 0) ? "STATISTICS," : "", ((stat->ability & MASK_1000) > 0) ? "1000Mbps," : "",((stat->ability & MASK_100) > 0) ? "100Mbps," : "",((stat->ability & MASK_10) > 0) ? "10Mbps," : "");
    printf("----------- Core settings -----------\n");
    printf("Link speed: %d Mbps\n", ((stat->emmc & SPEED_MASK) >> 30 == SEL_1000) ? 1000 : ((stat->emmc & SPEED_MASK) >> 30 == SEL_100) ? 100 : 10);
    printf("Link state: %s\n", (stat->link_state < 0) ? "N/A" : (stat->link_state == ETH_UP) ? "UP" : "DOWN");
    printf("--------- Core RX settings ----------\n");
    printf("Enabled: %s\n", ((stat->rx_cfg & ENABLE_MASK) > 0) ? "Yes" : "No");
    if ((stat->rx_cfg & JUMBO_MASK) > 0)
        printf("MTU: Jumbo\n");
    else if ((stat->rx_mtu & MTU_DEFAULT_MASK) > 0)
        printf("MTU: %dB\n", stat->rx_mtu & MTU_MASK);
    else if ((stat->rx_cfg & VLAN_MASK) > 0)
        printf("MTU: 1522B\n");
    else
        printf("MTU: 1518B\n");
    printf("VLAN: %s\n", ((stat->rx_cfg & VLAN_MASK) > 0) ? "Yes" : "No");
    printf("FCS: %s\n", ((stat->rx_cfg & FCS_MASK) > 0) ? "Pass" : "Strip");
    printf("Length/Type check: %s\n", ((stat->rx_cfg & LTC_MASK) == 0) ? "Yes" : "No");
    printf("Promiscuous mode: %s\n", ((stat->fmi & PROM_MASK) > 0) ? "Yes" : "No");
    printf("Bad frames: %s\n", ((stat->raf & BAD_MASK) > 0) ? "Receive" : "Drop");
    if ((stat->fmi & PROM_MASK) == 0)
    {   
        printf("Filter MAC addresses:\n");
        for (uint32_t i = 0; i < 4; i++)
        {
            printf("%X:%X:%X:%X:%X:%X", stat->mac[i][0], stat->mac[i][1], stat->mac[i][2], stat->mac[i][3], stat->mac[i][4], stat->mac[i][5]);
// TODO: tento kod asi neni potreba
//             if ((stat->fmi & INDEX_MASK) == i)
//                 printf(" <--- used");
            printf("\n");
        }
    }
    printf("--------- Core TX settings ----------\n");
    printf("Enabled: %s\n", ((stat->tx_cfg & ENABLE_MASK) > 0) ? "Yes" : "No");
    if ((stat->tx_cfg & JUMBO_MASK) > 0)
        printf("MTU: Jumbo\n");
    else if ((stat->tx_mtu & MTU_DEFAULT_MASK) > 0)
        printf("MTU: %dB\n", stat->tx_mtu & MTU_MASK);
    else if ((stat->tx_cfg & VLAN_MASK) > 0)
        printf("MTU: 1522B\n");
    else
        printf("MTU: 1518B\n");
    printf("VLAN: %s\n", ((stat->tx_cfg & VLAN_MASK) > 0) ? "Yes" : "No");
    printf("FCS: %s\n", ((stat->tx_cfg & FCS_MASK) > 0) ? "Pass" : "Generate");
    if ((stat->ability & STAT_MASK) > 0)
    {
        printf("--------- Basic statistics ----------\n");
        printf("RX Frames OK: %"PRIu64"\n", stat->rx_frames_ok);
        printf("RX Frames FCS error: %"PRIu64"\n", stat->rx_frames_bad);
        printf("RX Bytes: %"PRIu64"\n", stat->rx_bytes);
        printf("TX Frames OK: %"PRIu64"\n", stat->tx_frames_ok);
        printf("TX Bytes: %"PRIu64"\n", stat->tx_bytes);
    }
    printf("=====================================\n");
}

/**
 * \brief Dealocate dynamicaly allocated members of status structure.
 * \param stat Status structure.
 */
void dealoc_stat_members(stat_t *stat)
{
    free(stat->driver_data);
}

/**
 * Get link state via ethtool interface.
 * \param ifc Interface number.
 * \returns ETH_ERR if error occured, ETH_UNSUP if not supported, ETH_UP if 
 *          link is up and ETH_DOWN if link is down.
 */
int get_link_state(uint32_t ifc)
{
    struct ethtool_value command_data;
    int ret;
    struct ifreq interface_req;
    int sock;
    char device_name[128];
    
    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0)
    {
        fprintf(stderr, "Warning: Can not open ethernet iterface eth%d. \
Link state can not be determinated.\n", ifc);
        return ETH_ERR;
    }

    sprintf(device_name, "eth%d", ifc);
    if (strlen(device_name) + 1 > IFNAMSIZ)
    {
        fprintf(stderr, "Warning: Interface name %s\n is longer than maximal \
supported size of %d.\n", device_name, IFNAMSIZ);
        close(sock);
        return ETH_ERR;
    }
    memset(&interface_req, 0, sizeof(interface_req));
    memcpy(interface_req.ifr_name, device_name, IFNAMSIZ*sizeof(char));
    command_data.cmd = ETHTOOL_GLINK;
    interface_req.ifr_data = (char*)&command_data;
    ret = ioctl(sock, SIOCETHTOOL, &interface_req);
    
    if (ret == 0)
    {
        if (command_data.data != 0)
        {
            close(sock);
            return ETH_UP;
        }
        else
        {
            close(sock);
            return ETH_DOWN;
        }            
    }
    else
    {
        if (errno == EOPNOTSUPP)
        {
            fprintf(stderr, "Warning: Link state can not be determinated as \
this operation is not supported for ethernet interface eth%d.\n", ifc);
            close(sock);
            return ETH_UNSUP;
        }
        else
        {
            fprintf(stderr, "Warning: Link state can not be determinated as \
this operation caused an error for ethernet interface eth%d.\n", ifc);
            close(sock);
            return ETH_ERR;
        }
    }
    close(sock);
    return ETH_ERR;
}

/**
 * Get driver informations via ethtool interface.
 * \param ifc Interface number.
 * \returns String consisting of driver information. NULL if informations can 
 *          not be determinated.
 */
char* get_driver_data(uint32_t ifc)
{
    struct ethtool_drvinfo command_data;
    int ret;
    struct ifreq interface_req;
    int sock;
    char device_name[128];
    char * data = NULL;
    
    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0)
    {
        fprintf(stderr, "Warning: Can not open ethernet iterface eth%d. \
Driver data can not be determinated.\n", ifc);
        return data;
    }

    sprintf(device_name, "eth%d", ifc);
    if (strlen(device_name) + 1 > IFNAMSIZ)
    {
        fprintf(stderr, "Warning: Interface name %s\n is longer than maximal \
supported size of %d.\n", device_name, IFNAMSIZ);
        close(sock);
        return data;
    }
    memset(&interface_req, 0, sizeof(interface_req));
    memcpy(interface_req.ifr_name, device_name, IFNAMSIZ*sizeof(char));
    command_data.cmd = ETHTOOL_GDRVINFO;
    interface_req.ifr_data = (char*)&command_data;
    ret = ioctl(sock, SIOCETHTOOL, &interface_req);
    
    if (ret < 0)
    {
        fprintf(stderr, "Warning: Driver informations can not be determinated \
as this operation caused an error for ethernet interface eth%d.\n", ifc);
        close(sock);
        return data;
    }
    else
    {
        data = (char*) malloc(1024 * sizeof(char));
        if (data == NULL)
        {
            fprintf(stderr, "Warning: Driver informations can not be \
determinated as this operation caused a memory allocation error for ethernet \
interface eth%d.\n", ifc);
            close(sock);
            return data;
        }
        sprintf(data, "Driver: %s Version: %s", command_data.driver, \
                command_data.version);
        close(sock);
        return data;
    }
    close(sock);
    return data;
}

/**
 * \brief Read status and configuration from axi-ethernet.
 * \param comp Component pointer.
 * \param stat Status structure.
 */
void read_status(struct hwio_comp *comp, stat_t *stat)
{
    // Read values from ethernet device driver
    stat->link_state = get_link_state(stat->id);
    stat->driver_data = get_driver_data(stat->id);
    // Read values from axi ethernet
    stat->raf = ethernet_read(comp, RAF);
    stat->ability = ethernet_read(comp, AR);
    stat->rx_cfg = ethernet_read(comp, RCW1);
    stat->tx_cfg = ethernet_read(comp, TC);
    stat->emmc = ethernet_read(comp, EMMC);
    stat->rx_mtu = ethernet_read(comp, RXFC);
    stat->tx_mtu = ethernet_read(comp, TXFC);
    stat->fmi = ethernet_read(comp, FMI);
    uint32_t id = ethernet_read(comp, ID);
    stat->version.patch = id & PATCH_MASK;
    stat->version.type = (id & TYPE_MASK) >> 8;
    stat->version.minor = (id & MINOR_MASK) >> 16;
    stat->version.major = (id & MAJOR_MASK) >> 24;
    if ((stat->ability & STAT_MASK) > 0)
    {
        stat->rx_bytes = ((uint64_t)ethernet_read(comp, RXBU) << 32) | (uint64_t)ethernet_read(comp, RXBL);
        stat->tx_bytes = ((uint64_t)ethernet_read(comp, TXBU) << 32) | (uint64_t)ethernet_read(comp, TXBL);
        stat->rx_frames_ok = ((uint64_t)ethernet_read(comp, RXFU) << 32) | (uint64_t)ethernet_read(comp, RXFL);
        stat->tx_frames_ok = ((uint64_t)ethernet_read(comp, TXFU) << 32) | (uint64_t)ethernet_read(comp, TXFL);
        stat->rx_frames_bad = ((uint64_t)ethernet_read(comp, RXFCSERU) << 32) | (uint64_t)ethernet_read(comp, RXFCSERL);
    }
    if ((stat->fmi & PROM_MASK) == 0)
    {
        uint32_t val;
        for (uint32_t i = 0; i < 4; i++)
        {
            // select MAC register
            val = ethernet_read(comp, FMI);
            val &= ~INDEX_MASK;
            hwio_comp_write(comp, FMI, val | i);
            // Read lower bits (31:0)
            val = ethernet_read(comp, MAC0);
            stat->mac[i][0] = val & 0xFF;
            stat->mac[i][1] = (val >> 8) & 0xFF;
            stat->mac[i][2] = (val >> 16) & 0xFF;
            stat->mac[i][3] = (val >> 24) & 0xFF;
            // Read upper bits (47:32)
            val = ethernet_read(comp, MAC1);
            stat->mac[i][4] = val & 0xFF;
            stat->mac[i][5] = (val >> 8) & 0xFF;
        }
    }
}

/**
 * \brief Dump all statistics to the output.
 * \param comp Component pointer.
 */
void dump_all_statistics(struct hwio_comp *comp)
{
    for (uint32_t i = STAT_START; i <= STAT_END; i+=4)
    {
        printf("%08X\n", ethernet_read(comp, i));
    }
}



/**
 * Enable or disable axi-ethernet.
 * \param comp Component pointer.
 * \param enable Specifies if device is to be disabled (0) or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void enable_axi_ethernet(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RCW1);
        if (enable != 0)
            val |= ENABLE_MASK;
        else
            val &= ~ENABLE_MASK;
        hwio_comp_write(comp, RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = ethernet_read(comp, TC);
        if (enable != 0)
            val |= ENABLE_MASK;
        else
            val &= ~ENABLE_MASK;
        hwio_comp_write(comp, TC, val);
    }
}

/**
 * Enable or disable jumbo frames.
 * \param comp Component pointer.
 * \param enable Specifies if the support for jumbo frames is to be 
 *               disabled (0) or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_jumbo(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RCW1);
        if (enable != 0)
            val |= JUMBO_MASK;
        else
            val &= ~JUMBO_MASK;
        hwio_comp_write(comp, RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = ethernet_read(comp, TC);
        if (enable != 0)
            val |= JUMBO_MASK;
        else
            val &= ~JUMBO_MASK;
        hwio_comp_write(comp, TC, val);
    }
}

/**
 * Enable or disable basic VLAN support. 
 * \param comp Component pointer.
 * \param enable Specifies if the basic VLAN support is to be disabled (0) 
 *               or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_vlan(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RCW1);
        if (enable != 0)
            val |= VLAN_MASK;
        else
            val &= ~VLAN_MASK;
        hwio_comp_write(comp, RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = ethernet_read(comp, TC);
        if (enable != 0)
            val |= VLAN_MASK;
        else
            val &= ~VLAN_MASK;
        hwio_comp_write(comp, TC, val);
    }
}

/**
 * Enable or disable FCS passing or striping.
 * \param comp Component pointer.
 * \param enable Specifies if device strips the FCS from frames (0) or if 
 *               the FCS is passed to the processing blocks (1)[RX interface]. 
 *               Or is the FCS generated by the axi-ethernet (0) or the FCS 
 *               from the processing blocks is used (1)[TX interface].
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_fcs(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RCW1);
        if (enable != 0)
            val |= FCS_MASK;
        else
            val &= ~FCS_MASK;
        hwio_comp_write(comp, RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = ethernet_read(comp, TC);
        if (enable != 0)
            val |= FCS_MASK;
        else
            val &= ~FCS_MASK;
        hwio_comp_write(comp, TC, val);
    }
}

/**
 * Enable or disable type/len check on RX frames.
 * \param comp Component pointer.
 * \param enable Specifies if the check is enabled (0) or disabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_type_check(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RCW1);
        if (enable != 0)
            val |= LTC_MASK;
        else
            val &= ~LTC_MASK;
        hwio_comp_write(comp, RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        fprintf(stderr, "WARNING: Type/Len check is not supported by TX interface.\n");
    }
}

/**
 * Enable or disable receive of bad RX frames.
 * \param comp Component pointer.
 * \param enable Specifies if the check is enabled (1) or disabled (0).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void enable_bad_frames(struct hwio_comp *comp, uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RAF);
        if (enable != 0)
            val |= BAD_MASK;
        else
            val &= ~BAD_MASK;
        hwio_comp_write(comp, RAF, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        fprintf(stderr, "WARNING: Bad frames receive is not supported by TX interface.\n");
    }
}

/**
 * Set speed of axi-ethernet (10/100/1000Mbps). NOTE: This parameter does not 
 * set the phyter.
 * \param comp Component pointer.
 * \param speed Selected speed.
 */
void set_speed(struct hwio_comp *comp, uint32_t speed)
{
    uint32_t val;

    val = ethernet_read(comp, EMMC);
    val &= ~SPEED_MASK;
    val |= speed << 30;
    hwio_comp_write(comp, EMMC, val);
}

/**
 * Set MTU.
 * \param comp Component pointer.
 * \param mtu MTU value. If 0, default MTU is used.
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_mtu(struct hwio_comp *comp, uint32_t mtu, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = ethernet_read(comp, RXFC);
        if (mtu != 0)
        {
            val |= MTU_DEFAULT_MASK;
            val &= ~MTU_MASK;
            val |= mtu;
        }
        else
        {
            val |= ~MTU_DEFAULT_MASK;
        }
        hwio_comp_write(comp, RXFC, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = ethernet_read(comp, TXFC);
        if (mtu != 0)
        {
            val |= MTU_DEFAULT_MASK;
            val &= ~MTU_MASK;
            val |= mtu;
        }
        else
        {
            val |= ~MTU_DEFAULT_MASK;
        }
        hwio_comp_write(comp, TXFC, val);
    }
}

/**
 * Set promiscuous mode.
 * \param comp Component pointer.
 * \param enable Specifies if the promiscuous mode is to be disabled (0) 
 *               or enabled (1).
 */
void set_promiscuite_mode(struct hwio_comp *comp, uint32_t enable)
{
    uint32_t val;

    val = ethernet_read(comp, FMI);
    if (enable != 0)
        val |= PROM_MASK;
    else
        val &= ~PROM_MASK;
    hwio_comp_write(comp, FMI, val);
}

/**
 * Set MAC for MAC filter at specified index.
 * \param comp Component pointer.
 * \param index Index to the MAC addresses table (0-3).
 * \param mac MAC address.
 */
void set_mac(struct hwio_comp *comp, uint32_t index, mac_t mac)
{
    uint32_t val;
    
    // set the index
    val = ethernet_read(comp, FMI);
    val &= ~INDEX_MASK;
    val |= index;
    hwio_comp_write(comp, FMI, val);
    
    // set the MAC address
    val = mac[3];
    val <<= 8;
    val |= mac[2];
    val <<= 8;
    val |= mac[1];
    val <<= 8;
    val |= mac[0];
    hwio_comp_write(comp, MAC0, val);
    val = mac[5];
    val <<= 8;
    val |= mac[4];
    hwio_comp_write(comp, MAC1, val);
}

/**
 * \brief Perform commands on single interface.
 * \param comp Component pointer.
 * \param params Command line parameters.
 * \param ifc Inteface number
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int perform_cfg_update(struct hwio_comp *comp, params_t *params, int ifc)
{       
    // Perform requested commands on axi-ethernet
    // print full statistics dump
    if (params->sel_full_dump != 0)
        dump_all_statistics(comp);
    // List all interfaces
    if (params->sel_list != 0)
    {
        ethernet_print_component(comp, ifc);
    }
    // Enable or disable support of jumbo frames
    if (params->sel_jumbo != 0)
        set_jumbo(comp, params->jumbo, params->direction);
    // Enable or disable basic support for VLAN frames
    if (params->sel_vlan != 0)
        set_vlan(comp, params->vlan, params->direction);
    // Enable or disable FCS passing
    if (params->sel_fcs != 0)
        set_fcs(comp, params->fcs, params->direction);
    // Enable or disable type/length check of RX frames
    if (params->sel_type != 0)
        set_type_check(comp, params->type, params->direction);
    // Select speed of interface
    if (params->sel_speed != 0)
        set_speed(comp, params->speed);
    // Select MTU
    if (params->sel_mtu != 0)
        set_mtu(comp, params->mtu, params->direction);
    // Enable or disable promiscuous mode
    if (params->sel_promiscuite != 0)
        set_promiscuite_mode(comp, params->promiscuite);
    // Set MAC for filter at index
    if (params->sel_index != 0 && params->sel_mac != 0)
        set_mac(comp, params->index, params->mac);
    // List status and configuration of interface
    if (params->list_only != 0)
    {
        stat_t stat;
        memset(&stat, 0, sizeof(stat_t));
        stat.id = ifc;
        read_status(comp, &stat);
        print_status(&stat);
        dealoc_stat_members(&stat);
    }
    // Enable or disable reception of bad frames
    if (params->sel_bad != 0)
        enable_bad_frames(comp, params->bad, params->direction);
    // Enable or disable interface of axi-ethernet
    if (params->sel_enable != 0)
        enable_axi_ethernet(comp, params->enable, params->direction);

    // Report success
    return EXIT_SUCCESS;
}

/**
 * Get ethernet device index (ethX, X == index) from axi-ethernet name.
 * \param dev HWIO device.
 * \param params Command line parameters
 * \return Ethernet device index or ETH_NO_DEV if no axi-ethernet with matching 
 *         name was found.
 */
int get_device_index_by_name(struct hwio_dev *dev, params_t *params)
{
    if (params->sel_name != 0)
    {
        int retcode;
        struct hwio_comp *comp = NULL;
        
        retcode = hwio_byspec(dev, comp_compat_tab, &comp);

        if (retcode != HWIO_EOK)
        {
            fprintf(stderr,"ERROR: Unable to enumerate axi-ethernet components!\n");
            return ETH_NO_DEV;
        }
        if (comp == NULL)
        {
            fprintf(stderr,"ERROR: No axi-ethernet component found!\n");
            return ETH_NO_DEV;
        }
   
        // Determinate index
        uint32_t ifc = 0;
        int found = -1;
        while (comp != NULL) {
            if (strcmp(params->name, hwio_comp_name(comp)) == 0)
            {
                found = ifc;
            }
            ifc++;
            comp = hwio_comp_next(comp);
        }
        hwio_comp_list_free(comp);
        
        if (found == -1)
        {
            fprintf(stderr, "ERROR: No axi-ethernet component with name %s found!\n", params->name);
            return ETH_NO_DEV;
        }
        else
        {
            return found;
        }
        
    }
    else
    {
        fprintf(stderr,"ERROR: Device index can not be determinated: Parameter -u was not set!\n");
        return ETH_NO_DEV;
    }
    return ETH_NO_DEV;
}

/**
 * \brief Perform commands
 * \param params Command line parameters
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int perform_commands(params_t *params)
{
    int ret;
    uint32_t ifc = 0;
    char *path;
    char *device;
    struct hwio_dev dev;
    struct hwio_comp *comp = NULL;
    // Use either user provided path to device tree or the default one
    path = (params->sel_dtree_path != 0) ? params->dtree_path : DTREE_PATH;
    // Use either user provided path to mem device or the default one
    device = (params->sel_device != 0) ? params->device : MEM_PATH;
    // Map axi-ethernet memory space to userspace and do any other init
    if((ret = ethernet_init(&dev, path, device)) == EXIT_FAILURE)
        return ret;
    
    if (params->sel_name != 0)
    {
        int ret;
        ret = get_device_index_by_name(&dev, params);
        if (ret == ETH_NO_DEV)
        {
            return EXIT_FAILURE;
        }
        else
        {
            ifc = ret;
        }
    }
    
    ret = ethernet_find_components(&dev, &comp, params);
    if (ret == EXIT_FAILURE)
        return ret;
    
    
    while (comp != NULL) {
        // If interface number is set work with this axi_ethernet
        if ((params->sel_ifc_num != 0 && ifc == params->ifc_num) || params->sel_ifc_num == 0)
        {
            ret = perform_cfg_update(comp, params, ifc);
            if (ret != EXIT_SUCCESS)
            {
                // Relese the address space and do any other deinit
                ethernet_deinit(&dev, comp);
                return ret;
            }
        }
        ifc++;
        comp = hwio_comp_next(comp);
    }
    
    // Relese the address space and do any other deinit
    ethernet_deinit(&dev, comp);
    return EXIT_SUCCESS;
}
