/*!
 * \file ethctl_param.h
 * \brief Tool for controlling axi-ethernet - command line parameters 
 *        processing header file
 * \author Vlastimil Kosar <ikosar@fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: filterctl.c 1469 2012-05-23 15:59:18Z xkekel00 $
 *
 */

#ifndef ETHCTL_PARAM_H_INCLUDED
#define ETHCTL_PARAM_H_INCLUDED

/// Command line arguments
#define ARGUMENTS   "SAi:ld:e:j:v:f:t:s:m:p:n:a:x:hb:"

/// Enable RX
#define ENABLE_RX 2
/// enable TX
#define ENABLE_TX 1
/// set 10 Mb
#define SEL_10 0
/// set 100 Mb
#define SEL_100 1
/// set 1000 Mb
#define SEL_1000 2
/// command enabled
#define CMD_EN 1
/// command disabled
#define CMD_DIS 0
/// MAC filter minimal index
#define MAC_INDEX_MIN 0
/// MAC filter maximal index
#define MAC_INDEX_MAX 3
/// Minimal ethernet frame size for MTU check
#define MIN_MTU 64
/// Maximal ethernet frame size for MTU check
/// This is the maximal size allowed by axi-ethernet
/// The real maximal value is dependent on axi-ethernet configuration
#define MAX_MTU 32768

typedef uint32_t mac_t[6];

/// This structure is used to store commmand line parameters
typedef struct{
    uint32_t sel_full_dump;   /// -S parameter selected
    uint32_t sel_all;         /// -A parameter selected
    uint32_t sel_ifc_num;     /// -i parameter selected
    uint32_t ifc_num;         /// -i parameter value
    uint32_t sel_list;        /// -l parameter selected
    uint32_t sel_direction;   /// -d parameter selected
    uint32_t direction;       /// -d parameter value, only lower 2 bits are 
                              ///    valid, lsb(0) set to 1 means TX is enabled,
                              ///    msb(1) set to 1 means RX is enabled
    uint32_t sel_enable;      /// -e parameter selected
    uint32_t enable;          /// -e parameter value
    uint32_t sel_jumbo;       /// -j parameter selected
    uint32_t jumbo;           /// -j parameter value
    uint32_t sel_vlan;        /// -v parameter selected
    uint32_t vlan;            /// -v parameter value
    uint32_t sel_fcs;         /// -f parameter selected
    uint32_t fcs;             /// -f parameter value
    uint32_t sel_type;        /// -t parameter selected
    uint32_t type;            /// -t parameter value
    uint32_t sel_speed;       /// -s parameter selected
    uint32_t speed;           /// -s parameter value
    uint32_t sel_mtu;         /// -m parameter selected
    uint32_t mtu;             /// -m parameter value
    uint32_t sel_promiscuite; /// -p parameter selected
    uint32_t promiscuite;     /// -p parameter value
    uint32_t sel_index;       /// -n parameter selected
    uint32_t index;           /// -n parameter value
    uint32_t sel_mac;         /// -a parameter selected
    mac_t    mac;             /// -a parameter value
    uint32_t sel_dtree_path;  /// -x parameter selected
    char*    dtree_path;      /// -x parameter value
    uint32_t list_only;       /// only parameters -i or -A are used
    uint32_t sel_help;        /// -h parameter selected
    uint32_t sel_bad;         /// -b parameter selected
    uint32_t bad;             /// -b parameter value
} params_t;

/**
 * \brief Print help message.
 */
void usage();

/**
 * \brief Check command line parameters - are they sane?
 * \param params Command line parameters
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int check_params(params_t *params);

/**
 * \brief Parse command line parameters.
 * \param argc Number of command line parameters.
 * \param argv Command line parameters.
 * \param params Parsed command line parameters.
 * \return EXIT_SUCCESS if parameters ware parsed OK, EXIT_FAILURE if error 
 *         occured.
 */
int parse_cmdline(int argc, char *argv[], params_t *params);

#endif // ETHCTL_PARAM_H_INCLUDED
