/*!
 * \file ethctl_cmd.c
 * \brief Tool for controlling axi-ethernet - Axi-ethernet control 
 *         and communication commands
 * \author Vlastimil Kosar <ikosar@fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: filterctl.c 1469 2012-05-23 15:59:18Z xkekel00 $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include "ethctl_param.h"
#include "ethctl_cmd.h"
#include "hwio.h"

/**
 * \brief Initialize communication with axi-ethernet.
 * \param fn Path to device tree.
 * \param index Index of axi-ethernet in device tree.
 * \return Status of operation
 */
int ethernet_init(const char *fn, uint32_t index) {
    int x;
    //TODO: Change lib hwio
    x=hwio_init(fn, ETHERNET_NAME/*, index*/);
    if(x) {
        fprintf(stderr,"ERROR: Unable to initialize axi-ethernet [%d]!\n",x);
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}


/**
 * \brief Deinitialize communication with axi-ethernet.
 */
void ethernet_deinit() {
    hwio_deinit();
}

/**
 * \brief Print status and configuration of axi-ethernet.
 * \param stat Status structure.
 */
void print_status(stat_t *stat)
{
    printf("========== AXI ETHERNET %d ==========\n", stat->id);
    printf("Core version: %s %u.%u.%u\n", (stat->version.type > 0) ? "SOFT" : "HARD", stat->version.major, stat->version.minor, stat->version.patch);
    printf("--------- Core capabilities ---------\n");
    printf("%s %s %s %s %s\n", ((stat->ability & MAC_MASK) > 0) ? "MAC FILTERS," : "", ((stat->ability & STAT_MASK) > 0) ? "STATISTICS," : "", ((stat->ability & MASK_1000) > 0) ? "1000Mbps," : "",((stat->ability & MASK_100) > 0) ? "100Mbps," : "",((stat->ability & MASK_10) > 0) ? "10Mbps," : "");
    printf("----------- Core settings -----------\n");
    printf("Link speed: %d Mbps\n", ((stat->emmc & SPEED_MASK) >> 30 == SEL_1000) ? 1000 : ((stat->emmc & SPEED_MASK) >> 30 == SEL_100) ? 100 : 10);
    printf("--------- Core RX settings ----------\n");
    printf("Enabled: %s\n", ((stat->rx_cfg & ENABLE_MASK) > 0) ? "Yes" : "No");
    if ((stat->rx_cfg & JUMBO_MASK) > 0)
        printf("MTU: Jumbo\n");
    else if ((stat->rx_mtu & MTU_DEFAULT_MASK) > 0)
        printf("MTU: %dB\n", stat->rx_mtu & MTU_MASK);
    else if ((stat->rx_cfg & VLAN_MASK) > 0)
        printf("MTU: 1522B\n");
    else
        printf("MTU: 1518B\n");
    printf("VLAN: %s\n", ((stat->rx_cfg & VLAN_MASK) > 0) ? "Yes" : "No");
    printf("FCS: %s\n", ((stat->rx_cfg & FCS_MASK) > 0) ? "Pass" : "Strip");
    printf("Length/Type check: %s\n", ((stat->rx_cfg & LTC_MASK) == 0) ? "Yes" : "No");
    printf("Promiscuous mode: %s\n", ((stat->fmi & PROM_MASK) > 0) ? "Yes" : "No");
    printf("Bad frames: %s\n", ((stat->raf & BAD_MASK) > 0) ? "Receive" : "Drop");
    if ((stat->fmi & PROM_MASK) == 0)
    {   
        printf("Filter MAC addresses:\n");
        for (uint32_t i = 0; i < 4; i++)
        {
            printf("%X:%X:%X:%X:%X:%X", stat->mac[i][0], stat->mac[i][1], stat->mac[i][2], stat->mac[i][3], stat->mac[i][4], stat->mac[i][5]);
// TODO: tento kod asi neni potreba
//             if ((stat->fmi & INDEX_MASK) == i)
//                 printf(" <--- used");
            printf("\n");
        }
    }
    printf("--------- Core TX settings ----------\n");
    printf("Enabled: %s\n", ((stat->tx_cfg & ENABLE_MASK) > 0) ? "Yes" : "No");
    if ((stat->tx_cfg & JUMBO_MASK) > 0)
        printf("MTU: Jumbo\n");
    else if ((stat->tx_mtu & MTU_DEFAULT_MASK) > 0)
        printf("MTU: %dB\n", stat->tx_mtu & MTU_MASK);
    else if ((stat->tx_cfg & VLAN_MASK) > 0)
        printf("MTU: 1522B\n");
    else
        printf("MTU: 1518B\n");
    printf("VLAN: %s\n", ((stat->tx_cfg & VLAN_MASK) > 0) ? "Yes" : "No");
    printf("FCS: %s\n", ((stat->tx_cfg & FCS_MASK) > 0) ? "Pass" : "Generate");
    if ((stat->ability & STAT_MASK) > 0)
    {
        printf("--------- Basic statistics ----------\n");
        printf("RX Frames OK: %"PRIu64"\n", stat->rx_frames_ok);
        printf("RX Frames FCS error: %"PRIu64"\n", stat->rx_frames_bad);
        printf("RX Bytes: %"PRIu64"\n", stat->rx_bytes);
        printf("TX Frames OK: %"PRIu64"\n", stat->tx_frames_ok);
        printf("TX Bytes: %"PRIu64"\n", stat->tx_bytes);
    }
    printf("=====================================\n");
}

/**
 * \brief Read status and configuration from axi-ethernet.
 * \param stat Status structure.
 */
void read_status(stat_t *stat)
{
    // Read values from axi ethernet
    stat->raf = hwio_read32(RAF);
    stat->ability = hwio_read32(AR);
    stat->rx_cfg = hwio_read32(RCW1);
    stat->tx_cfg = hwio_read32(TC);
    stat->emmc = hwio_read32(EMMC);
    stat->rx_mtu = hwio_read32(RXFC);
    stat->tx_mtu = hwio_read32(TXFC);
    stat->fmi = hwio_read32(FMI);
    uint32_t id = hwio_read32(ID);
    stat->version.patch = id & PATCH_MASK;
    stat->version.type = (id & TYPE_MASK) >> 8;
    stat->version.minor = (id & MINOR_MASK) >> 16;
    stat->version.major = (id & MAJOR_MASK) >> 24;
    if ((stat->ability & STAT_MASK) > 0)
    {
        stat->rx_bytes = ((uint64_t)hwio_read32(RXBU) << 32) | (uint64_t)hwio_read32(RXBL);
        stat->tx_bytes = ((uint64_t)hwio_read32(TXBU) << 32) | (uint64_t)hwio_read32(TXBL);
        stat->rx_frames_ok = ((uint64_t)hwio_read32(RXFU) << 32) | (uint64_t)hwio_read32(RXFL);
        stat->tx_frames_ok = ((uint64_t)hwio_read32(TXFU) << 32) | (uint64_t)hwio_read32(TXFL);
        stat->rx_frames_bad = ((uint64_t)hwio_read32(RXFCSERU) << 32) | (uint64_t)hwio_read32(RXFCSERL);
    }
    if ((stat->fmi & PROM_MASK) == 0)
    {
        uint32_t val;
        for (uint32_t i = 0; i < 4; i++)
        {
            // select MAC register
            val = hwio_read32(FMI);
            val &= ~INDEX_MASK;
            hwio_write32(FMI, val | i);
            // Read lower bits (31:0)
            val = hwio_read32(MAC0);
            stat->mac[i][0] = val & 0xFF;
            stat->mac[i][1] = (val >> 8) & 0xFF;
            stat->mac[i][2] = (val >> 16) & 0xFF;
            stat->mac[i][3] = (val >> 24) & 0xFF;
            // Read upper bits (47:32)
            val = hwio_read32(MAC1);
            stat->mac[i][4] = val & 0xFF;
            stat->mac[i][5] = (val >> 8) & 0xFF;
        }
    }
}

/**
 * \brief Dump all statistics to the output.
 */
void dump_all_statistics()
{
    for (uint32_t i = STAT_START; i <= STAT_END; i+=4)
    {
        printf("%08X\n", hwio_read32(i));
    }
}



/**
 * Enable or disable axi-ethernet.
 * \param enable Specifies if device is to be disabled (0) or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void enable_axi_ethernet(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RCW1);
        if (enable != 0)
            val |= ENABLE_MASK;
        else
            val &= ~ENABLE_MASK;
        hwio_write32(RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = hwio_read32(TC);
        if (enable != 0)
            val |= ENABLE_MASK;
        else
            val &= ~ENABLE_MASK;
        hwio_write32(TC, val);
    }
}

/**
 * Enable or disable jumbo frames.
 * \param enable Specifies if the support for jumbo frames is to be 
 *               disabled (0) or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_jumbo(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RCW1);
        if (enable != 0)
            val |= JUMBO_MASK;
        else
            val &= ~JUMBO_MASK;
        hwio_write32(RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = hwio_read32(TC);
        if (enable != 0)
            val |= JUMBO_MASK;
        else
            val &= ~JUMBO_MASK;
        hwio_write32(TC, val);
    }
}

/**
 * Enable or disable basic VLAN support.
 * \param enable Specifies if the basic VLAN support is to be disabled (0) 
 *               or enabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_vlan(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RCW1);
        if (enable != 0)
            val |= VLAN_MASK;
        else
            val &= ~VLAN_MASK;
        hwio_write32(RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = hwio_read32(TC);
        if (enable != 0)
            val |= VLAN_MASK;
        else
            val &= ~VLAN_MASK;
        hwio_write32(TC, val);
    }
}

/**
 * Enable or disable FCS passing or striping.
 * \param enable Specifies if device strips the FCS from frames (0) or if 
 *               the FCS is passed to the processing blocks (1)[RX interface]. 
 *               Or is the FCS generated by the axi-ethernet (0) or the FCS 
 *               from the processing blocks is used (1)[TX interface].
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_fcs(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RCW1);
        if (enable != 0)
            val |= FCS_MASK;
        else
            val &= ~FCS_MASK;
        hwio_write32(RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = hwio_read32(TC);
        if (enable != 0)
            val |= FCS_MASK;
        else
            val &= ~FCS_MASK;
        hwio_write32(TC, val);
    }
}

/**
 * Enable or disable type/len check on RX frames.
 * \param enable Specifies if the check is enabled (0) or disabled (1).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_type_check(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RCW1);
        if (enable != 0)
            val |= LTC_MASK;
        else
            val &= ~LTC_MASK;
        hwio_write32(RCW1, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        fprintf(stderr, "WARNING: Type/Len check is not supported by TX interface.\n");
    }
}

/**
 * Enable or disable receive of bad RX frames.
 * \param enable Specifies if the check is enabled (1) or disabled (0).
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void enable_bad_frames(uint32_t enable, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RAF);
        if (enable != 0)
            val |= BAD_MASK;
        else
            val &= ~BAD_MASK;
        hwio_write32(RAF, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        fprintf(stderr, "WARNING: Bad frames receive is not supported by TX interface.\n");
    }
}

/**
 * Set speed of axi-ethernet (10/100/1000Mbps). NOTE: This parameter does not 
 * set the phyter.
 * \param speed Selected speed.
 */
void set_speed(uint32_t speed)
{
    uint32_t val;

    val = hwio_read32(EMMC);
    val &= ~SPEED_MASK;
    val |= speed << 30;
    hwio_write32(EMMC, val);
}

/**
 * Set MTU.
 * \param mtu MTU value. If 0, default MTU is used.
 * \param direction Specifies if this operation is used on RX, TX or both 
 *                  interfaces.
 */
void set_mtu(uint32_t mtu, uint32_t direction)
{
    uint32_t val;
    if ((direction & ENABLE_RX) != 0)
    {
        val = hwio_read32(RXFC);
        if (mtu != 0)
        {
            val |= MTU_DEFAULT_MASK;
            val &= ~MTU_MASK;
            val |= mtu;
        }
        else
        {
            val |= ~MTU_DEFAULT_MASK;
        }
        hwio_write32(RXFC, val);
    }
    if ((direction & ENABLE_TX) != 0)
    {
        val = hwio_read32(TXFC);
        if (mtu != 0)
        {
            val |= MTU_DEFAULT_MASK;
            val &= ~MTU_MASK;
            val |= mtu;
        }
        else
        {
            val |= ~MTU_DEFAULT_MASK;
        }
        hwio_write32(TXFC, val);
    }
}

/**
 * Set promiscuous mode.
 * \param enable Specifies if the promiscuous mode is to be disabled (0) 
 *               or enabled (1).
 */
void set_promiscuite_mode(uint32_t enable)
{
    uint32_t val;

    val = hwio_read32(FMI);
    if (enable != 0)
        val |= PROM_MASK;
    else
        val &= ~PROM_MASK;
    hwio_write32(FMI, val);
}

/**
 * Set MAC for MAC filter at specified index.
 * \param index Index to the MAC addresses table (0-3).
 * \param mac MAC address.
 */
void set_mac(uint32_t index, mac_t mac)
{
    uint32_t val;
    
    // set the index
    val = hwio_read32(FMI);
    val &= ~INDEX_MASK;
    val |= index;
    hwio_write32(FMI, val);
    
    // set the MAC address
    val = mac[3];
    val <<= 8;
    val |= mac[2];
    val <<= 8;
    val |= mac[1];
    val <<= 8;
    val |= mac[0];
    hwio_write32(MAC0, val);
    val = mac[5];
    val <<= 8;
    val |= mac[4];
    hwio_write32(MAC1, val);
}

/**
 * \brief Perform commands on single interface.
 * \param params Command line parameters
 * \param ifc Inteface number
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int perform_cfg_update(params_t *params, int ifc)
{
    char *path;
    int ret;
    
    // Use either user provided path to device tree or the default one
    path = (params->sel_dtree_path != 0) ? params->dtree_path : DTREE_PATH;
    // Map axi-ethernet memory space to userspace and do any other init
    if((ret = ethernet_init(path, ifc)) == EXIT_FAILURE)
        return ret;
    
    // Perform requested commands on axi-ethernet
    // print full statistics dump
    if (params->sel_full_dump != 0)
        dump_all_statistics();
    // List all interfaces
    if (params->sel_list != 0)
    {
        //TODO: implement
        ;
    }
    // Enable or disable support of jumbo frames
    if (params->sel_jumbo != 0)
        set_jumbo(params->jumbo, params->direction);
    // Enable or disable basic support for VLAN frames
    if (params->sel_vlan != 0)
        set_vlan(params->vlan, params->direction);
    // Enable or disable FCS passing
    if (params->sel_fcs != 0)
        set_fcs(params->fcs, params->direction);
    // Enable or disable type/length check of RX frames
    if (params->sel_type != 0)
        set_type_check(params->type, params->direction);
    // Select speed of interface
    if (params->sel_speed != 0)
        set_speed(params->speed);
    // Select MTU
    if (params->sel_mtu != 0)
        set_mtu(params->mtu, params->direction);
    // Enable or disable promiscuous mode
    if (params->sel_promiscuite != 0)
        set_promiscuite_mode(params->promiscuite);
    // Set MAC for filter at index
    if (params->sel_index != 0 && params->sel_mac != 0)
        set_mac(params->index, params->mac);
    // List status and configuration of interface
    if (params->list_only != 0)
    {
        stat_t stat;
        memset(&stat, 0, sizeof(stat_t));
        stat.id = ifc;
        read_status(&stat);
        print_status(&stat);
    }
    // Enable or disable reception of bad frames
    if (params->sel_bad != 0)
        enable_bad_frames(params->bad, params->direction);
    // Enable or disable interface of axi-ethernet
    if (params->sel_enable != 0)
        enable_axi_ethernet(params->enable, params->direction);
    // Relese the address space and do any other deinit
    ethernet_deinit();
    // Report success
    return EXIT_SUCCESS;
}

/**
 * \brief Perform commands
 * \param params Command line parameters
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int perform_commands(params_t *params)
{
    int ret;
    
    // If interface number is set work with this axi_ethernet
    if (params->sel_ifc_num != 0)
    {
        ret = perform_cfg_update(params, params->ifc_num);
        if (ret != EXIT_SUCCESS)
            return ret;
    }
    // Otherwise get number of axi-ethernets
    else
    {
        // TODO: implement function (in lib hwio?) which returns number of components
        int num = 4; // hwio_list("axi-ethernet")
        for (int i = 0; i < num; i++)
        {
            ret = perform_cfg_update(params, i);
            if (ret != EXIT_SUCCESS)
                return ret;
        }        
    }
    return EXIT_SUCCESS;
}
