/**
 * @file axi_tap_ctl.c
 * @brief Tool to control axi_tap unit.
 * @author Tomas Fukac <xfukac00@stud.fit.vutbr.cz>
 * @copyright Brno University of Technology
 * @date 2013
 *
 * @section license LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * @mainpage
 *
 * The library libhwio is a part of the sec6net project
 * https://merlin.fit.vutbr.cz/wiki-sec6net/.
 */

#include "hwio_tool.h"
#include "hwio_comp_spec.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

HWIO_TOOL_CUSTOM_NOT_USED

#define CTL_OPTS "SCT"

// registers

#define REG_NAME   0x000 // unit name
#define REG_FLAGS   0x004


/*
    FLAFS REGISTER
    --------------
             3        2        1        0
  -------+--------+--------+--------+--------+
    ...  |  TAP   |  TAP1  |  TAP0  | DO_TAP |
  -------+--------+--------+--------+--------+

    DO_TAP - we want change state to 0 -> TAP, 1 -> DONT TAP (R/W)
    TAP0 - information about tapping in_data0 and in_ctrl0 to out_data0 and out_ctrl0 (R)
    TAP1 - information about tapping in_data1 and in_ctrl1 to out_data1 and out_ctrl1 (R)
    TAP - information about tap (is set if reg_tap0 and reg_tap1) (R)

*/
struct hwio_comp_spec ctl_compat[] = {
    {"debug", "AXI_TAP", HWIO_VER2(1, 0)},
    {"ug4-150", "axi_tap", HWIO_VER3(2, 0, HWIO_VERSION_NA)},
    HWIO_COMPAT_END
};

int ctl_custom_parse(int opt, const char *optarg,
		struct hwio_tool_params *params)
{
    switch(opt) {
    case 'S':
    case 'C':
    case 'T':
        if(hwio_tool_params_has_cmd(params))
            return hwio_tool_error_another_cmd(opt, params);

        params->cmd.code = opt;
        params->cmd.arg = NULL;

        return HWIO_TOOL_CUSTOM_OK;

    default:
        return HWIO_TOOL_CUSTOM_UNKNOWN;
    }
}

void hwio_tool_help(const char *toolname,
		const struct hwio_tool_params *params)
{
    printf("Usage: %s [options | command]\n", toolname);
    printf("  -L            List of units\n");
    printf("  -I            Unit information\n");
    printf("  -S            Start tapping\n");
    printf("  -C            Stop tapping\n");
    printf("  -T            Show information about tap\n");
    printf("selectors:\n");
    printf("  -A            Select all available devices\n");
    printf("  device...     Name or names of devices to work with\n");
}

int start_tap(struct hwio_comp *c,
		hwio_tool_comp_err_t err, const struct hwio_tool_params *params)
{
    uint32_t flags_reg;
    int tap_curr, tap_next;

    hwio_tool_verbose(params, 1, "Starting...\n");

    if(hwio_comp_read(c, REG_FLAGS, &flags_reg))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read");

    tap_curr = (flags_reg & 0x08) >> 3;
    tap_next = flags_reg & 0x01;

    if(tap_curr != tap_next)
    {
        hwio_tool_verbose(params, 0, "Unit is changing state now!\n");
        return EXIT_FAILURE;
    }

    if(hwio_comp_write(c, REG_FLAGS, 0))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write in register");
    else
        hwio_tool_verbose(params, 1, "Registr set\n");

	return HWIO_TOOL_OP_OK;
}

int stop_tap(struct hwio_comp *c,
		hwio_tool_comp_err_t err, const struct hwio_tool_params *params)
{
    uint32_t flags_reg;
    int tap_curr, tap_next;

    hwio_tool_verbose(params, 1, "Stopping...\n");

    if(hwio_comp_read(c, REG_FLAGS, &flags_reg))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read");

    tap_curr = (flags_reg & 0x08) >> 3;
    tap_next = flags_reg & 0x01;

    if(tap_curr != tap_next)
    {
        hwio_tool_verbose(params, 0, "Unit is changing state now!\n");
        return EXIT_FAILURE;
    }


    if(hwio_comp_write(c, REG_FLAGS, 1))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write in register");
    else
        hwio_tool_verbose(params, 1, "Registr set\n");

	return HWIO_TOOL_OP_OK;
}

int tap_state(struct hwio_comp *c,
		hwio_tool_comp_err_t err, const struct hwio_tool_params *params)
{
    uint32_t flags_reg;
    int tap_curr, tap_next;

    if(hwio_comp_read(c, REG_FLAGS, &flags_reg))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read");

    tap_curr = (flags_reg & 0x08) >> 3;
    tap_next = flags_reg & 0x01;

    printf("Unit state:\n");
    if(tap_curr != tap_next)
    {
        if(tap_curr)
            printf("  tap starting...\n");
        else
            printf("  stopping tapping...\n");
    }
    else
    {
        if(!tap_curr)
            printf("  tapping\n");
        else
            printf("  dont tapping\n");
    }
    printf("Registrs (0 - tap, 1 - dont tap):\n");

    printf("  TAP0:  %d\n",(flags_reg & 0x02) >> 1);
    printf("  TAP1:  %d\n",(flags_reg & 0x04) >> 2);
    printf("  TAP:   %d\n",(flags_reg & 0x08) >> 3);



	return HWIO_TOOL_OP_OK;
}

int main(int argc, char **argv)
{
    struct hwio_tool_params params = hwio_tool_params_new(
        HWIO_TOOL_CUSTOM_NULL, argc, argv);

    switch(hwio_tool_params_parse(&params, &ctl_custom_parse, CTL_OPTS)) {
    case HWIO_TOOL_PARAMS_FAILED:
        return HWIO_TOOL_EARGS;

    case HWIO_TOOL_PARAMS_EXIT:
        return 0;

    default:
        break;
    }

    hwio_tool_verbose(&params, 0, "Arguments parsed successfully");
    hwio_tool_verbose(&params, 1, "Command: -%c", params.cmd.code);
    hwio_tool_verbose(&params, 2, "Argument: %s", params.cmd.arg);

    switch(params.cmd.code) {
    case 'I':
        return hwio_tool_exec_generic(&params, ctl_compat,
            &hwio_tool_comp_info, &hwio_tool_error_handler);

    case 'L':
        return hwio_tool_exec_generic(&params, ctl_compat,
            &hwio_tool_comp_name, &hwio_tool_error_handler);

    case 'S':
        return hwio_tool_exec_generic(&params, ctl_compat,
            &start_tap, &hwio_tool_error_handler);

    case 'C':
        return hwio_tool_exec_generic(&params, ctl_compat,
            &stop_tap, &hwio_tool_error_handler);

    case 'T':
        return hwio_tool_exec_generic(&params, ctl_compat,
            &tap_state, &hwio_tool_error_handler);

    case '\0':
        return hwio_tool_error(HWIO_TOOL_EARGS, "No command was specified");

    default:
        return hwio_tool_error(HWIO_TOOL_EARGS, "Command -%c is not implemented", params.cmd.code);
    }
}
