/**
 * aidctl.c
 * Copyright (C) 2013 Jan Viktorin
 */

#include "hwio_tool.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>

#define REG_NEG     0x00000000
#define REG_COMMAND 0x00000004
#define REG_STATUS  0x00000004
#define REG_NAME(i) (0x00000008 + (i) * 4)
#define REG_VERSION 0x00000018
#define REG_TIME(i) (0x0000001C + (i) * 4)
#define REG_CUSTOM  0x0000002C

#define AIDCRL_OPTS "DC:Sf"

struct hwio_comp_spec aidctl_compat[] = {
	{"xlnx", "axi_id", HWIO_VER3(1, 0, 'a')},
	HWIO_COMPAT_END
};

struct hwio_tool_custom {
	uint32_t command;
	int valid;
	int force;
};

static
int parse_hex(const char *s, uint32_t *value)
{
	uint32_t result = 0;

	for(int i = 0; s[i] != '\0'; ++i) {
		if(isdigit(s[i]))
			result += s[i] - '0';

		else if(isxdigit(s[i]))
			result += tolower(s[i]) - 'a' + 10;

		else {
			return hwio_tool_error(0,
				"Invalid character, expect a hex number: %02x (%c)\n", s[i], s[i]);
		}

		if(s[i + 1] != '\0')
			result *= 16;
	}

	*value = result;
	return 1;
}

static
int aidctl_custom_parse(int opt, const char *optarg,
		struct hwio_tool_params *p)
{
	switch(opt) {
	case 'D':
	case 'S':
		if(hwio_tool_params_has_cmd(p))
			return hwio_tool_error_another_cmd(opt, p);

		p->cmd.code = opt;
		p->cmd.arg  = NULL;
		return HWIO_TOOL_CUSTOM_OK;

	case 'C':
		if(hwio_tool_params_has_cmd(p))
			return hwio_tool_error_another_cmd(opt, p);

		p->cmd.code = opt;
		p->cmd.arg  = optarg;

		if(optarg[0] == '0' && tolower(optarg[1]) == 'x')
			p->custom->valid = parse_hex(optarg + 2, &p->custom->command);
		else
			p->custom->valid = parse_hex(optarg, &p->custom->command);

		return HWIO_TOOL_CUSTOM_OK;

	case 'f':
		p->custom->force = 1;
		break;

	default:
		return HWIO_TOOL_CUSTOM_UNKNOWN;
	}

	return HWIO_TOOL_CUSTOM_OK;
}

void hwio_tool_help(const char *toolname,
		const struct hwio_tool_params *params)
{
	printf("Usage: %s [options | command]\n", toolname);
	printf("Options:\n");
	printf("  -f         skip testing of the negation register\n");
	printf("  -h         help\n");
	printf("  -v         verbosity level\n");
	printf("  -q         quiet mode\n");
	printf("  -x <path>  hardware description\n");
	printf("  -d <path>  physical device access point\n");
	printf("  -A         select all devices\n");
	printf("Commands:\n");
	printf("  -L        list all available ID components\n");
	printf("  -I        print info about selected ID components\n");
	printf("  -D        print design information (default command)\n");
	printf("  -C <hex>  write the command register\n");
	printf("  -S        print the status register contents\n");
}

static
int test_is_aid(const struct hwio_tool_params *params, struct hwio_comp *c, hwio_tool_comp_err_t err)
{
	if(params->custom->force) {
		hwio_tool_verbose(params, 1, "Force enabled, skipping the negation test");
		return 1;
	}

	if(hwio_comp_write(c, REG_NEG, 0xF0F0F0F0)) {
		err(c, "Failed to write register NEG");
		return 0;
	}

	uint32_t neg;
	if(hwio_comp_read(c, REG_NEG, &neg)) {
		err(c, "Failed to read register NEG");
		return 0;
	}

	if(neg != 0x0F0F0F0F) {
		err(c, "The component is not a compatible ID component (negation failed)");
		return 0;
	}

	return 1;
}

static
int aidctl_design_info(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	uint32_t name[4];
	uint32_t version;
	uint32_t time[4];
	uint32_t custom;

	if(!test_is_aid(params, c, err))
		return HWIO_TOOL_OP_FAILED;

	if(hwio_comp_read(c, REG_VERSION, &version)) {
		err(c, "Failed to read register VERSION");
		return HWIO_TOOL_OP_FAILED;
	}

	if(hwio_comp_read(c, REG_CUSTOM, &custom)) {
		err(c, "Failed to read register CUSTOM");
		return HWIO_TOOL_OP_FAILED;
	}

	for(int i = 0; i < 4; ++i) {
		if(hwio_comp_read(c, REG_NAME(i), name + i)) {
			err(c, "Failed to read register NAME(%d)", i);
			return HWIO_TOOL_OP_FAILED;
		}

		if(hwio_comp_read(c, REG_TIME(i), time + i)) {
			err(c, "Failed to read register TIME(%d)", i);
			return HWIO_TOOL_OP_FAILED;
		}
	}

	const char *sname = (char *) name;
	printf("Design : ");
	for(int i = 0; sname[i] != '\0' && i < 4 * 4; ++i)
		printf("%c", sname[i]);
	printf("\n");

	printf("Version: 0x%08X\n", version);

	const char *stime = (char *) time;
	printf("Built  : ");
	for(int i = 0; stime[i] != '\0' && i < 4 * 4; ++i)
		printf("%c", stime[i]);
	printf("\n");

	printf("Custom : 0x%08X\n", custom);

	return HWIO_TOOL_OP_OK;
}

static
int aidctl_command(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(!test_is_aid(params, c, err))
		return HWIO_TOOL_OP_FAILED;

	if(hwio_comp_write(c, REG_COMMAND, params->custom->command)) {
		err(c, "Failed to write register COMMAND");
		return HWIO_TOOL_OP_FAILED;
	}

	return HWIO_TOOL_OP_OK;
}

static
int aidctl_status(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(!test_is_aid(params, c, err))
		return HWIO_TOOL_OP_FAILED;

	uint32_t status;

	if(hwio_comp_read(c, REG_STATUS, &status)) {
		err(c, "Failed to read register STATUS");
		return HWIO_TOOL_OP_FAILED;
	}

	printf("0x%08X\n", status);
	return HWIO_TOOL_OP_OK;
}

int main(int argc, char **argv)
{
	struct hwio_tool_custom custom = {0, 0, 0};
	struct hwio_tool_params params = hwio_tool_params_new(
			&custom, argc, argv);

	switch(hwio_tool_params_parse(&params, &aidctl_custom_parse, AIDCRL_OPTS)) {
	case HWIO_TOOL_PARAMS_FAILED:
		return HWIO_TOOL_EARGS;

	case HWIO_TOOL_PARAMS_EXIT:
		return 0;

	default:
		break;
	}

	switch(params.cmd.code) {
	case 'I':
		return hwio_tool_exec_generic(&params, aidctl_compat,
				&hwio_tool_comp_info, &hwio_tool_error_handler);

	case 'L':
		return hwio_tool_exec_generic(&params, aidctl_compat,
				&hwio_tool_comp_name, &hwio_tool_error_handler);

	case 'C':
		if(!params.custom->valid)
			return HWIO_TOOL_EARGS;

		return hwio_tool_exec_generic(&params, aidctl_compat,
				&aidctl_command, &hwio_tool_error_handler);

	case 'S':
		return hwio_tool_exec_generic(&params, aidctl_compat,
				&aidctl_status, &hwio_tool_error_handler);

	case 'D':
	case '\0':			
		return hwio_tool_exec_generic(&params, aidctl_compat, &aidctl_design_info,
				&hwio_tool_error_handler);

	default:
		return hwio_tool_error(HWIO_TOOL_EARGS, "Command -%c is not implemented", params.cmd.code);
	}
}
