#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <stdbool.h>
#include <stdint.h>
#include "hwio.h"
#include "hwio_tool.h"
#include "hwio_comp.h"
#include "adectl.h"
#include "adectl_param.h"
#include "adectl_cmd.h"

/* Print help message. */
void hwio_tool_help(const char *toolname, const struct hwio_tool_params *params) {
	printf(
	"Usage: dataexportctl [options/cmds]\n"
	"Tool parameters:\n"
	"-h         Writes this help message\n"
	"-v         Raise verbosity.\n"
	"-q         Mute.\n"
	"-A         Exexute commands on every compatible component.\n"
	"-x <path>  Path to description of component (src).\n"
	"-d <path>  Path to file of component (phys).\n\n"
	"Default tools commands:\n"
	"-L         Lists all reacheble devices\n"
	"-I         Shows information about device(s)\n\n"
	"Commands specific to this tool:\n"
	"-C[val]    Set/get ECN value (0-3)\n"
	"-G[val]    Set/get flags/next header value (0-255)\n"
	"           Note: for flags only lower 2 bits matter (so using val>3 is pointless)\n"
	"-N[val]    Set/get identification/flow label value (0-1048575)\n"
	"           Note: for identification only lower 16 bits matter (so using val>65535 is pointless)\n"
	"-O[ip]     Set/get source ip address (x.x.x.x/x:x:x:x:x:x:x:x)\n"
	"-B[ip]     Set/get destination ip address (x.x.x.x/x:x:x:x:x:x:x:x)\n"
	"           Note: IPv4 format will be padded with 0 if used while IPv6 is enabled\n"
	"           Note: only lower 32 bits will be used out of IPv6 format if used while IPv4 is enabled\n"
	"-S[val]    Set/get DSCP value (0-63)\n"
	"-T[val]    Set/get time to live/hop limit value (0-255)\n"
	"-U[port]   Set/get source UDP port(0-65535)\n"
	"-P[port]   Set/get destination UDP port (0-65535)\n"
	"-M[mac]    Set/get destination MAC (xx:xx:xx:xx:xx:xx)\n"
	"-J[mac]    Set/get source MAC (xx:xx:xx:xx:xx:xx)\n"
	"-V         Get version of IP used\n"
	);
}

/* Check if value is in specified range. */
int check_value(uint64_t val, uint64_t a, uint64_t b)
{
	return (val < a || val > b) ? 1 : 0;
}

/* Print error message if parameter is out of range. */
int print_param_error(char* param, uint64_t val, uint64_t a, uint64_t b)
{
	return hwio_tool_error(1, "Value of parameter %s is out of range. Invalid \
value is %lu and the range is <%lu,%lu>\n", param, val, a, b);
}

int getIP(uint32_t *ip_val1, uint32_t *ip_val2, uint32_t *ip_val3, uint32_t *ip_val4, const char *optarg)
{
	uint32_t ip[8]={0, 0, 0, 0, 0, 0, 0, 0};
	if ((sscanf(optarg, "%u.%u.%u.%u", &ip[0], &ip[1], &ip[2], &ip[3]) != 4 ||
		ip[0]>255 || ip[1]>255 || ip[2]>255 || ip[3]>255) &&
		sscanf(optarg, "%4X:%4X:%4X:%4X:%4X:%4X:%4X:%4X", &ip[0], &ip[1], &ip[2], &ip[3], &ip[4], &ip[5], &ip[6], &ip[7]) != 8)
		return hwio_tool_error(1,"Invalid IP address specified: %s!\n", optarg);
	if (optarg[4]!=':') {
		*ip_val4 = ip[0];
		*ip_val4 <<= 8;
		*ip_val4 |= ip[1];
		*ip_val4 <<= 8;
		*ip_val4 |= ip[2];
		*ip_val4 <<= 8;
		*ip_val4 |= ip[3];
		*ip_val1 = 0;
		*ip_val2 = 0;
		*ip_val3 = 0;
	} else {
		*ip_val1 = ip[0];
		*ip_val1 <<= 16;
		*ip_val1 |= ip[1];
		*ip_val2 = ip[2];
		*ip_val2 <<= 16;
		*ip_val2 |= ip[3];
		*ip_val3 = ip[4];
		*ip_val3 <<= 16;
		*ip_val3 |= ip[5];
		*ip_val4 = ip[6];
		*ip_val4 <<= 16;
		*ip_val4 |= ip[7];
	}
	return 0;
}

int getMAC(uint64_t *MAC_val, const char *optarg)
{
	uint32_t mac[6];
	if (sscanf(optarg, "%2X:%2X:%2X:%2X:%2X:%2X", &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]) != 6)
		return hwio_tool_error(1,"Invalid MAC address specified: %s!\n", optarg);
	*MAC_val = mac[0];
	*MAC_val <<= 8;
	*MAC_val |= mac[1];
	*MAC_val <<= 8;
	*MAC_val |= mac[2];
	*MAC_val <<= 8;
	*MAC_val |= mac[3];
	*MAC_val <<= 8;
	*MAC_val |= mac[4];
	*MAC_val <<= 8;
	*MAC_val |= mac[5];
	if (check_value(*MAC_val, 0, BITM48))
		return print_param_error("-j", *MAC_val, 0, BITM48);
	return 0;
}

/* Custom parser. */
int custom_parser(int opt,const char *optarg, struct hwio_tool_params *params)
{
	if(hwio_tool_params_has_cmd(params))
		return hwio_tool_error_another_cmd(opt,params);
	params->cmd.code = opt;
	params->cmd.arg = optarg;

	if(strchr("SCNGTUPOBJMV", opt) && !optarg)
		return 0;

	switch (opt) {
		case 'S':
			params->custom->dscp_val = atoi(optarg);
			if (check_value(params->custom->dscp_val, 0, BITM6)) {
				print_param_error("-S", params->custom->dscp_val, 0, BITM6);
				return 1;
			}
			break;
		case 'C':
			params->custom->ecn_val = atoi(optarg);
			if (check_value(params->custom->ecn_val, 0, BITM2)) {
				print_param_error("-C", params->custom->ecn_val, 0, BITM2);
				return 1;
			}
			break;
		case 'N':
			params->custom->identification_val = atoi(optarg);
			if (check_value(params->custom->identification_val, 0, BITM20)) {
				print_param_error("-N", params->custom->identification_val, 0, BITM20);
				return 1;
			}
			break;
		case 'G':
			params->custom->flags_val = atoi(optarg);
			if (check_value(params->custom->flags_val, 0, BITM8)) {
				print_param_error("-G", params->custom->flags_val, 0, BITM8);
				return 1;
			}
			break;
		case 'T':
			params->custom->time_to_live_val = atoi(optarg);
			if (check_value(params->custom->time_to_live_val, 0, BITM8)) {
				print_param_error("-T", params->custom->time_to_live_val, 0, BITM8);
				return 1;
			}
			break;
		case 'U':
			params->custom->source_port_val = atoi(optarg);
			if (check_value(params->custom->source_port_val, 0, BITM16)) {
				print_param_error("-U", params->custom->source_port_val, 0, BITM16);
				return 1;
			}
			break;
		case 'P':
			params->custom->destination_port_val = atoi(optarg);
			if (check_value(params->custom->destination_port_val, 0, BITM16)) {
				print_param_error("-P", params->custom->destination_port_val, 0, BITM16);
				return 1;
			}
			break;
		case 'O':
			if (getIP(&params->custom->source_ip_val1, &params->custom->source_ip_val2, &params->custom->source_ip_val3, &params->custom->source_ip_val4, optarg))
				return 1;
			break; 
		case 'B':
			if (getIP(&params->custom->destination_ip_val1,&params->custom->destination_ip_val2, &params->custom->destination_ip_val3, &params->custom->destination_ip_val4, optarg))
				return 1;
			break;
		case 'J':
			if (getMAC(&params->custom->source_MAC_val, optarg))
				return 1;
			break;
		case 'M':
			if (getMAC(&params->custom->destination_MAC_val, optarg))
				return 1;
			break;
		default:
			return HWIO_TOOL_CUSTOM_UNKNOWN;
			break;
	}
    
	return HWIO_TOOL_CUSTOM_OK;
} 
