#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <stdbool.h>
#include <stdint.h>
#include "hwio.h"
#include "hwio_tool.h"
#include "hwio_comp.h"
#include "adectl.h"
#include "adectl_param.h"
#include "adectl_cmd.h"

static int read_value32(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const char *label, const uint32_t addr)
{
	uint32_t data = 0;
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) addr);
	
	if (hwio_comp_read(c, addr, &data) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addr);
	
	printf("%s: %lu\n", label, (unsigned long) data);
	hwio_tool_verbose(params, 1, "Read successful");
	return EXIT_SUCCESS;
}

static int write_value32(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const uint32_t addr, const uint32_t value)
{
	hwio_tool_verbose(params, 1, "Writing address 0x%lx", (unsigned long) addr);

	if(hwio_comp_write(c, addr, value) != HWIO_EOK)
		return hwio_tool_error(1,"Writing failed", optarg);

	hwio_tool_verbose(params, 1, "Write successful");
	return 0;
}

// just helpfull function for reading version
static int read_save_version(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const uint32_t addr)
{
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) ADD_VER);
	
	if (hwio_comp_read(c, addr, &params->custom->version_val) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) ADD_VER);
	
	hwio_tool_verbose(params, 1, "Read successful");
	return EXIT_SUCCESS;
}

static int read_value_ip(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const char *label, const uint32_t addr)
{
	uint32_t data[4] = {0, 0, 0, 0};
	
	// read version first so we know if we want IPv4 or IPv6
	read_save_version(c, err, params, ADD_VER);
	
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) addr+12);
	
	if (hwio_comp_read(c, addr+12, &data[3]) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addr+12);
	
	// we already have IPv4
	if (params->custom->version_val == 4) {
		printf("%s: %u.%u.%u.%u\n", label, data[3]>>24, (data[3]>>16) & 0x0FF,
				(data[3]>>8) & 0x0FF, data[3] & 0x0FF);
		hwio_tool_verbose(params, 1, "Read successful");
		return EXIT_SUCCESS;
	}
	
	// else we need another 3 words
	hwio_tool_verbose(params, 1, "Read successful");
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) addr);
	
	if (hwio_comp_read(c, addr, &data[0]) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addr);
		
	hwio_tool_verbose(params, 1, "Read successful");
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) addr+4);
	
	if (hwio_comp_read(c, addr+4, &data[1]) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addr+4);
		
	hwio_tool_verbose(params, 1, "Read successful");
	hwio_tool_verbose(params, 1, "Reading address 0x%lx", (unsigned long) addr+8);
	
	if (hwio_comp_read(c, addr+8, &data[2]) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addr+8);
	
	printf("%s: %X:%X:%X:%X:%X:%X:%X:%X\n", label, data[0]>>16,
			data[0] & 0x0FFFF, data[1]>>16, data[1] & 0x0FFFF,
			data[2]>>16, data[2] & 0x0FFFF, data[3]>>16,
			data[3] & 0x0FFFF);
	hwio_tool_verbose(params, 1, "Read successful");
	return EXIT_SUCCESS;
}

static int read_value_mac(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const char *label, const uint32_t addrlow, const uint32_t addrhigh)
{
	uint32_t datal = 0;
	uint32_t datah = 0;

	hwio_tool_verbose(params, 1, "Reading addresses 0x%lx, 0x%lx",
			(unsigned long) addrlow, (unsigned long) addrhigh);
	
	if (hwio_comp_read(c, addrhigh, &datah) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addrhigh);

	if (hwio_comp_read(c, addrlow, &datal) != HWIO_EOK)
		hwio_tool_error(1, "Reading of 0x%lx failed", (unsigned long) addrlow);
	
	printf("%s: %X:%X:%X:%X:%X:%X\n", label, datah>>24,
			(datah>>16) & 0x0FF, (datah>>8) & 0x0FF, datah & 0x0FF,
			(datal>>8) & 0x0FF, datal & 0x0FF);
	hwio_tool_verbose(params, 1, "Read successful");
	return EXIT_SUCCESS;
}

static int write_value_mac(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params,
		const uint32_t addrlow, const uint32_t addrhigh, const uint64_t value)
{
	hwio_tool_verbose(params, 1, "Writing addresses 0x%lx, 0x%lx",
			(unsigned long) addrlow, (unsigned long) addrhigh);

	uint32_t valueh = value / (BITM16 + 1);
	uint32_t valuel = value % (BITM16 + 1);

	if(hwio_comp_write(c, addrhigh, valueh) != HWIO_EOK)
		return hwio_tool_error(1,"Writing of 0x%lx failed", addrhigh);

	if(hwio_comp_write(c, addrlow, valuel) != HWIO_EOK)
		return hwio_tool_error(1,"Writing of 0x%lx failed", addrlow);

	hwio_tool_verbose(params, 1, "Write successful");
	return 0;
}

int rw_srcport(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->source_port_val))
		return read_value32(c, err, params, "Source port", ADD_S_UDP);

	return write_value32(c, err, params, ADD_S_UDP,
			params->custom->source_port_val);
}

int rw_dstport(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->destination_port_val))
		return read_value32(c, err, params, "Destination port", ADD_D_UDP);

	return write_value32(c, err, params, ADD_D_UDP,
			params->custom->destination_port_val);
}

int rw_dscp(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->dscp_val))
		return read_value32(c, err, params, "DSCP", ADD_DSCP);

	return write_value32(c, err, params, ADD_DSCP,
			params->custom->dscp_val);
}

int rw_ecn(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->ecn_val))
		return read_value32(c, err, params, "ECN", ADD_ECN);

	return write_value32(c, err, params, ADD_ECN,
			params->custom->ecn_val);
}

int rw_identification(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->identification_val))
		return read_value32(c, err, params, "Identification/Flow Label", ADD_IDENT_FL);

	return write_value32(c, err, params, ADD_IDENT_FL,
			params->custom->identification_val);
}

int rw_flags(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->flags_val))
		return read_value32(c, err, params, "Flags/Next Header", ADD_FLAG_NH);

	return write_value32(c, err, params, ADD_FLAG_NH,
			params->custom->flags_val);
}

int rw_ttl(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->time_to_live_val))
		return read_value32(c, err, params, "TTL/Hop Limit", ADD_TTL_HL);

	return write_value32(c, err, params, ADD_TTL_HL,
			params->custom->time_to_live_val);
}

int rw_srcip(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->source_ip_val1))
		return read_value_ip(c, err, params, "Source IP", ADD_S_IP1);

	int ret = write_value32(c, err, params, ADD_S_IP1,
			params->custom->source_ip_val1);
	if (ret) return ret;
	ret = write_value32(c, err, params, ADD_S_IP2,
			params->custom->source_ip_val2);
	if (ret) return ret;
	ret = write_value32(c, err, params, ADD_S_IP3,
			params->custom->source_ip_val3);
	if (ret) return ret;
	return write_value32(c, err, params, ADD_S_IP4,
			params->custom->source_ip_val4);
}

int rw_dstip(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param32_invalid(params->custom->destination_ip_val1))
		return read_value_ip(c, err, params, "Destination IP", ADD_D_IP1);

	int ret = write_value32(c, err, params, ADD_D_IP1,
			params->custom->destination_ip_val1);
	if (ret) return ret;
	ret = write_value32(c, err, params, ADD_D_IP2,
			params->custom->destination_ip_val2);
	if (ret) return ret;
	ret = write_value32(c, err, params, ADD_D_IP3,
			params->custom->destination_ip_val3);
	if (ret) return ret;
	return write_value32(c, err, params, ADD_D_IP4,
			params->custom->destination_ip_val4);
}

int rw_srcmac(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param64_invalid(params->custom->source_MAC_val))
		return read_value_mac(c, err, params, "Source MAC", ADD_S_MACL, ADD_S_MACU);

	return write_value_mac(c, err, params, ADD_S_MACL, ADD_S_MACU,
			params->custom->source_MAC_val);
}

int rw_dstmac(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	if(param64_invalid(params->custom->destination_MAC_val))
		return read_value_mac(c, err, params, "Destination MAC", ADD_D_MACL, ADD_D_MACU);

	return write_value_mac(c, err, params, ADD_D_MACL, ADD_D_MACU,
			params->custom->destination_MAC_val);
}

int r_version(struct hwio_comp *c, hwio_tool_comp_err_t err,
		const struct hwio_tool_params *params)
{
	return read_value32(c, err, params, "Version", ADD_VER);
}