--[[
  File: RSA.lua
  Desc: RSA encoding/decoding algorithm
  Author: Jan Drazil
]]

function gcd(a, b)
  --Euclidean Algorithm for greatest common divisor
  while(b~=0) do
    tmp = a % b
    a = b
    b = tmp
  end
  return a;
end

function EEA(a,b)
  --Extended Euclidean Algorithm--
  if(a<b) then
    tmp=a
    a=b
    b=tmp
  end

  alfa1,alfa2,beta1,beta2 = 0,1,1,0
  q,r=0,0
  while(b>0) do
    q=math.floor(a/b)
    r=a-q*b
    a,b=b,r
    tmp=alfa2
    alfa2=alfa1
    alfa1=tmp-q*alfa1
    tmp=beta2
    beta2=beta1
    beta1=tmp-q*beta1
  end
  return {a, alfa2, beta2}
end



function RSAcreateKey(prime1, prime2)
  --Create public and private key--
  n=prime1*prime2
  eulerTotient=(prime1-1)*(prime2-1)
  --for i in reversed(range(2,eulerTotient-1)):
  for i=eulerTotient-2,2,-1 do
    if(gcd(i, eulerTotient) == 1) then
      exponent=i
      break;
    end
  end
  d=EEA(exponent,eulerTotient)
  d=d[3] --multiplicative inverse
  if(d<0) then
    d=d+eulerTotient --fix negative value in Z_eulerTotient
  end
  return {['privateKey']={['exponent']=exponent, ['modulus']=n}, ['publicKey']={['exponent']=d, ['modulus']=n}}
end

function RSAencode(publicKey, str)
  letters={}
  for i=1,str:len() do
    table.insert(letters, string.byte(str, i))
  end
  for i=1,#letters do
    letters[i]=RSAUseKey(publicKey, letters[i])
  end
  return letters;
end

function RSAdecode(privateKey, intList)
  for i=1,#intList do
    intList[i]=string.char(RSAUseKey(privateKey, intList[i]))
  end
  return table.concat(letters, '')
end

function RSAUseKey(key, number)
  --Apply key to number--
  modulus,exponent=key['modulus'],key['exponent']
  -- (a * b) mod c = ((a mod c)(b mod c)) mod c
  -- x^y mod z = (x*x*x*x*...*x) mod z
  parcMod = number % modulus
  result = parcMod

  for i=0,exponent-2 do
    result = (result * parcMod) % modulus
  end
  return result
end

--Example of use
--[[keys=RSAcreateKey(907, 997)
cypher=RSAencode(keys['publicKey'],"Hello World!")
print(cypher)
print(RSAdecode(keys['privateKey'],cypher))
]]



---Generated---
keys=RSAcreateKey(73, 179)
print(RSAdecode(keys['publicKey'], RSAencode(keys['privateKey'], "Lorem ipsum!")))