#ifndef VIDEOSELECTOR_H
#define VIDEOSELECTOR_H

#include <QWidget>
#include <QFileDialog>
#include <QThread>
#include <QDebug>
#include <QMessageBox>
#include <QMutex>
#include <vector>
#include <VideoMatcher.h>
#include <VideoSequence.h>
#include <FrameDescriptorExtractor.h>

namespace Ui {
    class VideoSelector;
    class VideoSelectorItem;
    class VideoSelectorItemData;
    class VideoSelectorItemPc;
    class VideoSelectorWatcherThread;
    class VideoSelectorWorkerThread;
}

class VideoSelectorItemData
{
private:
    QString filename;
    vmatch::VideoSequenceDataPtr data;

public:
    VideoSelectorItemData(QString filename, vmatch::VideoSequenceDataPtr data);
    QString getFilename();
    vmatch::VideoSequenceDataPtr getData();
};

class VideoSelectorItem
{
public:
    typedef enum
    {
        ST_READY,
        ST_LOADING,
        ST_ERROR_RECOVERABLE,
        ST_ERROR_FINAL
    } Status;

protected:
    QString label;
    Status status;
    VideoSelectorItemData *data;

public:
    VideoSelectorItem(QString label);
    ~VideoSelectorItem();
    QString getLabel();
    Status getStatus();
    VideoSelectorItemData *getData();
    bool isReady();
    bool isLoading();
    bool isError();
    virtual void load() = 0;
    virtual int getProgress() = 0;

friend class VideoSelectorWorkerThread;
};

class VideoSelectorItemPc : public VideoSelectorItem
{
private:
    QString filename;
    vmatch::VideoSequenceDataSourcePtr dataSource;

    cv::Ptr<vmatch::CvVideoSequence> sequence;
    QMutex mutexSequenceModification;

    void initSequence(bool lock = true);
    void deleteSequence(bool lock = true);

public:
    VideoSelectorItemPc(QString filename, vmatch::VideoSequenceDataSource::Config config);
    virtual void load();
    virtual int getProgress();
};

#ifdef VTAPI
class VideoSelectorItemVt : public VideoSelectorItem
{
private:
    QString filename;
    vmatch::VideoSequenceDataSourcePtr dataSource;

public:
    VideoSelectorItemVt(QString filename, vmatch::VideoSequenceDataSource::Config config);
    virtual void load();
    virtual int getProgress();
};
#endif

class VideoSelectorWatcherThread : public QThread
{
    Q_OBJECT

private:
    VideoSelectorItem *item;
    int interval;
    int progress;

signals:
    void progressChanged(int progress);

public:
    VideoSelectorWatcherThread(VideoSelectorItem *item, int interval = 100);
    void run();
};

class VideoSelectorWorkerThread : public QThread
{
    Q_OBJECT

private:
    VideoSelectorItem *item;
    VideoSelectorWatcherThread *watcherThread;

signals:
    void itemStatusChanged(VideoSelectorItem *item);
    void itemProgressChanged(VideoSelectorItem *item, int progress);

private slots:
    void on_watcher_progressChanged(int progress);

public:
    VideoSelectorWorkerThread(VideoSelectorItem *item);
    void run();
    void terminateWatcherThread();
};

typedef std::vector<VideoSelectorItem *> VideoSelectorItems;

class VideoSelector : public QWidget
{
    Q_OBJECT
    
private:
    static const QString DEFAULT_TOOLTIP_IMPORT_VT;
    static const QString DEFAULT_TOOLTIP_IMPORT_PC;
    static const QString DEFAULT_IMPORT_DIALOG_TITLE;
    static const vmatch::VideoSequenceDataSource::Config DEFAULT_CONFIG = vmatch::VideoSequenceDataSource::CONF_COLHIST;

public:
    explicit VideoSelector(QWidget *parent = 0);
    ~VideoSelector();
    VideoSelectorItemData *getCurrentSequenceData();
    void setImportPcDialogTitle(QString title);
    void setImportPcButtonTooltip(QString title);
#ifdef VTAPI
    void setImportVtButtonTooltip(QString title);
#endif

signals:
    void selectionChanged(VideoSelectorItemData *data);

private slots:
    void on_btnImportVt_clicked();
    void on_btnImportPc_clicked();
    void on_btnReload_clicked();
    void on_btnClose_clicked();
    void on_cmbVideos_currentIndexChanged(int index);
    void on_worker_itemStatusChanged(VideoSelectorItem *item);
    void on_worker_itemProgressChanged(VideoSelectorItem *item, int progress);

private:
    Ui::VideoSelector *ui;
    VideoSelectorItems items;
    vmatch::VideoSequenceDataSource::Config config;
    QString titleImportPcDialog;

    void showPanelInfo(VideoSelectorItem *item);
    void showPanelProgress(VideoSelectorItem *item);
    void showPanelError(VideoSelectorItem *item);
    void addItem(VideoSelectorItem *item, bool showImmediately = true);
    void removeItem(int idx);
    void showItem(VideoSelectorItem *item);
    void invalidateItem(VideoSelectorItem *item);
    void selectItem(int idx);
    QString formatTime(int time) const;
};

#endif // VIDEOSELECTOR_H
