#include "videoselector.h"
#include "ui_videoselector.h"

const QString VideoSelector::DEFAULT_TOOLTIP_IMPORT_VT = "Načíst video ze serveru VideoTerror";
const QString VideoSelector::DEFAULT_TOOLTIP_IMPORT_PC = "Načíst video z tohoto počítače";
const QString VideoSelector::DEFAULT_IMPORT_DIALOG_TITLE = "Otevřít video";

VideoSelector::VideoSelector(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::VideoSelector)
{
    ui->setupUi(this);
    showPanelInfo(NULL);

    ui->btnImportVt->setToolTip(DEFAULT_TOOLTIP_IMPORT_VT);
    ui->btnImportPc->setToolTip(DEFAULT_TOOLTIP_IMPORT_PC);
    titleImportPcDialog = DEFAULT_IMPORT_DIALOG_TITLE;

    config = DEFAULT_CONFIG;

#ifndef VTAPI
    ui->btnImportVt->setVisible(false);
#endif
}

VideoSelector::~VideoSelector()
{
    delete ui;
}


void VideoSelector::on_btnImportVt_clicked()
{
#ifdef VTAPI
    QString filename = ""; // TODO ziskat odnekud

    VideoSelectorItemVt *item = new VideoSelectorItemVt(filename, config);

    VideoSelectorWorkerThread *worker = new VideoSelectorWorkerThread(item);
    connect(worker, SIGNAL(itemStatusChanged(VideoSelectorItem*)), SLOT(on_worker_itemStatusChanged(VideoSelectorItem*)));
    connect(worker, SIGNAL(itemProgressChanged(VideoSelectorItem*,int)), SLOT(on_worker_itemProgressChanged(VideoSelectorItem*,int)));

    addItem(item);
    worker->start();
#endif
}

void VideoSelector::on_btnImportPc_clicked()
{
    QString filename = QFileDialog::getOpenFileName(this, titleImportPcDialog);
    if(filename != "")
    {
        VideoSelectorItemPc *item = new VideoSelectorItemPc(filename, config);

        VideoSelectorWorkerThread *worker = new VideoSelectorWorkerThread(item);
        connect(worker, SIGNAL(itemStatusChanged(VideoSelectorItem*)), SLOT(on_worker_itemStatusChanged(VideoSelectorItem*)));
        connect(worker, SIGNAL(itemProgressChanged(VideoSelectorItem*,int)), SLOT(on_worker_itemProgressChanged(VideoSelectorItem*,int)));

        addItem(item);
        worker->start();
    }
}

void VideoSelector::showPanelInfo(VideoSelectorItem *item)
{
    ui->lstInfo->clear();

    if(item != NULL)
    {
        if(dynamic_cast<VideoSelectorItemPc *>(item) != NULL)
        {
               ui->lstInfo->addItem(QString("-z tohoto počítače"));
        }
        else
        {
            ui->lstInfo->addItem(QString("-ze serveru VideoTerror"));
        }

        double fps = item->getData()->getData()->getMetaData().fps;
        int length = item->getData()->getData()->getMetaData().length;

        ui->lstInfo->addItem(QString("-délka: ") + formatTime(length/fps) + " (" + QString::number(length) + " snímků)");
        ui->lstInfo->addItem(QString("-fps: ") + QString::number(fps));
    }

    if(!ui->lstInfo->isVisible())
    {
        ui->frmProgress->hide();
        ui->frmError->hide();
        ui->lstInfo->show();
    }
}

void VideoSelector::showPanelProgress(VideoSelectorItem *item)
{
    if(item != NULL)
    {
        ui->progressBar->setValue(item->getProgress());
    }
    else
    {
        ui->progressBar->setValue(0);
    }

    if(!ui->frmProgress->isVisible())
    {
        ui->frmProgress->show();
        ui->frmError->hide();
        ui->lstInfo->hide();
    }
}

void VideoSelector::showPanelError(VideoSelectorItem *item)
{
    if(item != NULL)
    {
        ui->btnReload->setEnabled(item->getStatus() != VideoSelectorItem::ST_ERROR_FINAL);
        ui->btnClose->setEnabled(true);
    }
    else
    {
        ui->btnReload->setEnabled(false);
        ui->btnClose->setEnabled(false);
    }

    if(!ui->frmError->isVisible())
    {
        ui->frmProgress->hide();
        ui->frmError->show();
        ui->lstInfo->hide();
    }
}

void VideoSelector::on_cmbVideos_currentIndexChanged(int index)
{
    if(index >= 0)
    {
        showItem(items[index]);

        if(items[index]->isReady())
        {
            emit selectionChanged(items[index]->getData());
        }
    }
    else {
        emit selectionChanged(NULL);
    }
}

void VideoSelector::addItem(VideoSelectorItem *item, bool showImmediately)
{
    items.push_back(item);
    ui->cmbVideos->addItem(item->getLabel());

    if(showImmediately)
    {
        selectItem((int)items.size()-1);
    }
}

void VideoSelector::showItem(VideoSelectorItem *item)
{
    switch(item->getStatus())
    {
    case VideoSelectorItem::ST_READY:
        showPanelInfo(item);
        break;

    case VideoSelectorItem::ST_LOADING:
        showPanelProgress(item);
        break;

    case VideoSelectorItem::ST_ERROR_RECOVERABLE:
    case VideoSelectorItem::ST_ERROR_FINAL:
        showPanelError(item);
        break;
    }
}

void VideoSelector::selectItem(int idx)
{
    if(idx < items.size())
    {
        ui->cmbVideos->setCurrentIndex(idx);
    }
}

void VideoSelector::removeItem(int idx)
{
    if(idx == ui->cmbVideos->currentIndex())
    {
        int newIdx = -1;
        if(ui->cmbVideos->currentIndex() > 0)
        {
            newIdx = ui->cmbVideos->currentIndex() - 1;
        }
        else if(ui->cmbVideos->currentIndex() < ui->cmbVideos->count()-1)
        {
            newIdx = ui->cmbVideos->currentIndex() + 1;
        }

        if(newIdx != -1)
        {
            selectItem(newIdx);
        }
        else
        {
            showPanelInfo(NULL);
        }
    }

    if(idx < items.size())
    {
        items.erase(items.begin() + idx, items.begin() + idx);
        ui->cmbVideos->removeItem(idx);
    }
}

void VideoSelector::on_worker_itemStatusChanged(VideoSelectorItem *item)
{
    invalidateItem(item);
}

void VideoSelector::on_worker_itemProgressChanged(VideoSelectorItem *item, int progress)
{
    Q_UNUSED(progress);
    invalidateItem(item);
}

void VideoSelector::invalidateItem(VideoSelectorItem *item)
{
    if(this->items[ui->cmbVideos->currentIndex()] == item)
    {
        showItem(item);

        if(item->isReady())
        {
            emit selectionChanged(item->getData());
        }
    }
}

VideoSelectorItemData::VideoSelectorItemData(QString filename, vmatch::VideoSequenceDataPtr data)
{
    this->filename = filename;
    this->data = data;
}

QString VideoSelectorItemData::getFilename()
{
    return filename;
}

vmatch::VideoSequenceDataPtr VideoSelectorItemData::getData()
{
    return data;
}

VideoSelectorItem::VideoSelectorItem(QString label)
{
    this->label = label;
    this->status = ST_LOADING;
    this->data = NULL;
}

VideoSelectorItem::~VideoSelectorItem()
{
    if(data != NULL)
    {
        delete data;
    }
}

QString VideoSelectorItem::getLabel()
{
    return label;
}

VideoSelectorItem::Status VideoSelectorItem::getStatus()
{
    return status;
}

VideoSelectorItemData *VideoSelectorItem::getData()
{
    return data;
}

bool VideoSelectorItem::isReady()
{
    return status == ST_READY;
}

bool VideoSelectorItem::isLoading()
{
    return status == ST_LOADING;
}

bool VideoSelectorItem::isError()
{
    return status == ST_ERROR_RECOVERABLE || status == ST_ERROR_FINAL;
}

VideoSelectorItemPc::VideoSelectorItemPc(QString filename, vmatch::VideoSequenceDataSource::Config config) : VideoSelectorItem(filename)
{
    this->dataSource = vmatch::VideoSequenceDataSource::create(config);
    this->filename = filename;
    this->sequence = NULL;
}

void VideoSelectorItemPc::load()
{
    status = ST_LOADING;
    initSequence();

    try
    {      
        vmatch::KeyFramesExtractorPtr extractor = dataSource->getKeyFramesExtractor();
        vmatch::VideoSequencePtr s = sequence;
        vmatch::KeyFrames keyframes = extractor->extract(s);
        //vmatch::KeyFrames keyframes = extractor->extract(sequence);

        vmatch::VideoSequence::MetaData meta;
        meta.fps = sequence->getCapture()->get(CV_CAP_PROP_FPS);
        meta.length = (int)sequence->getCapture()->get(CV_CAP_PROP_FRAME_COUNT);

        data = new VideoSelectorItemData(filename, new vmatch::VideoSequenceData(keyframes, meta));

        status = ST_READY;
    }
    catch(...)
    {
        status = ST_ERROR_FINAL;
        deleteSequence();
        return;
    }

    deleteSequence();
}

void VideoSelectorItemPc::initSequence(bool lock)
{
    if(lock)
    {
        mutexSequenceModification.lock();
    }

    if(data != NULL)
    {
        delete data;
        data = NULL;
    }

    if(sequence != NULL)
    {
        delete sequence;
        sequence = NULL;
    }

    try
    {
        sequence = new vmatch::CvVideoSequence(filename.toStdString(), dataSource->getFrameDescriptorExtractor());
    }
    catch(...)
    {
        sequence = NULL;
        status = ST_ERROR_RECOVERABLE;

        if(lock)
        {
            mutexSequenceModification.unlock();
        }

        throw;
    }

    if(lock)
    {
        mutexSequenceModification.unlock();
    }
}

void VideoSelectorItemPc::deleteSequence(bool lock)
{
    if(lock)
    {
        mutexSequenceModification.lock();
    }

    if(sequence != NULL)
    {
        sequence->getCapture()->release();
    }

    if(lock)
    {
        mutexSequenceModification.unlock();
    }
}

int VideoSelectorItemPc::getProgress()
{
    mutexSequenceModification.lock();

    if(sequence == NULL || sequence->getCapture()->get(CV_CAP_PROP_FRAME_COUNT) == 0)
    {
        mutexSequenceModification.unlock();
        return -1;
    }

    int newProgress = ceil(100*sequence->getCapture()->get(CV_CAP_PROP_POS_FRAMES) / sequence->getCapture()->get(CV_CAP_PROP_FRAME_COUNT));

    mutexSequenceModification.unlock();
    return newProgress;
}

#ifdef VTAPI
VideoSelectorItemVt::VideoSelectorItemVt(QString filename, vmatch::VideoSequenceDataSource::Config config) : VideoSelectorItem(filename)
{
    this->dataSource = vmatch::VideoSequenceDataSource::create(config);
    this->filename = filename;
}

void VideoSelectorItemVt::load()
{
    status = ST_LOADING;

    try
    {
        vmatch::VideoSequenceDataPtr sequenceData = dataSource->getSequenceDataVt(filename.toStdString());

        data = new VideoSelectorItemData(filename, sequenceData);
        status = ST_READY;
    }
    catch(...)
    {
        status = ST_ERROR_FINAL;
    }
}

int VideoSelectorItemVt::getProgress()
{
    return 0;
}
#endif

VideoSelectorWatcherThread::VideoSelectorWatcherThread(VideoSelectorItem *item, int interval) : QThread()
{
    this->item = item;
    this->interval = interval;
}

void VideoSelectorWatcherThread::run()
{
    this->progress = -1;

    while(true)
    {
        int newProgress = item->getProgress();
        if(newProgress != progress)
        {
            emit progressChanged(newProgress);
            progress = newProgress;
        }

        msleep(interval);
    }
}

VideoSelectorWorkerThread::VideoSelectorWorkerThread(VideoSelectorItem *item) : QThread()
{
    this->item = item;
    watcherThread = NULL;
}

void VideoSelectorWorkerThread::run()
{
    terminateWatcherThread();

    watcherThread = new VideoSelectorWatcherThread(item);
    connect(watcherThread, SIGNAL(progressChanged(int)), SLOT(on_watcher_progressChanged(int)));
    watcherThread->start();

    try
    {
        item->load();
    }
    catch(...) // theoretically shouldn`t happen
    {
        item->status = VideoSelectorItem::ST_ERROR_FINAL;
    }

    emit itemStatusChanged(item);
    terminateWatcherThread();
}

void VideoSelectorWorkerThread::terminateWatcherThread()
{
    if(watcherThread != NULL)
    {
        watcherThread->terminate();
        while(!watcherThread->isFinished()) {}
        delete watcherThread;
        watcherThread = NULL;
    }
}

void VideoSelectorWorkerThread::on_watcher_progressChanged(int progress)
{
    emit itemProgressChanged(item, progress);
}

void VideoSelector::on_btnReload_clicked()
{
    VideoSelectorItem *item = items[ui->cmbVideos->currentIndex()];

    VideoSelectorWorkerThread *worker = new VideoSelectorWorkerThread(item);
    connect(worker, SIGNAL(itemStatusChanged(VideoSelectorItem*)), SLOT(on_worker_itemStatusChanged(VideoSelectorItem*)));
    connect(worker, SIGNAL(itemProgressChanged(VideoSelectorItem*,int)), SLOT(on_worker_itemProgressChanged(VideoSelectorItem*,int)));

    worker->start();
}

void VideoSelector::on_btnClose_clicked()
{
    removeItem(ui->cmbVideos->currentIndex());
}

VideoSelectorItemData *VideoSelector::getCurrentSequenceData()
{
    if(ui->cmbVideos->currentIndex() >= 0 && items.size())
    {
        return items[ui->cmbVideos->currentIndex()]->getData();
    }
    else
    {
        return NULL;
    }
}

QString VideoSelector::formatTime(int time) const
{
    int hours = (int)(time / 3600);
    int minutes = (int)((time / 60) % 60);
    int seconds = (int)(time % 60);

    std::ostringstream str;
    str.fill('0');
    str << std::setw(2) << hours << ":" << std::setw(2) << minutes << ":" << std::setw(2) << seconds;

    return QString::fromStdString(str.str());
}

void VideoSelector::setImportPcDialogTitle(QString title)
{
    titleImportPcDialog = title;
}

void VideoSelector::setImportPcButtonTooltip(QString tooltip)
{
    ui->btnImportPc->setToolTip(tooltip);
}

#ifdef VTAPI

void VideoSelector::setImportVtButtonTooltip(QString tooltip)
{
    ui->btnImportVt->setToolTip(tooltip);
}

#endif
