#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QMainWindow>
#include <QMessageBox>
#include <QResizeEvent>
#include <QKeyEvent>
#include <QThread>
#include <QDebug>
#include <map>
#include <../gui/videoselector.h>
#include <../gui/videocontrols.h>
#include <../gui/progressindicator.h>
#include <../gui/similaritytimeline.h>
#include <../gui/paramscontroller.h>

namespace Ui {
class ComparisonKey;
class ComparisonResult;
class ComparisonWorkerThread;
class MainWindowState;
class MainWindowPlayerThread;
class MainWindow;
}

class ComparisonKey {
private:
    VideoSelectorItemData *dataRef;
    VideoSelectorItemData *dataQry;

public:
    ComparisonKey();
    ComparisonKey(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry);
    bool operator==(const ComparisonKey & second) const;
    bool operator<(const ComparisonKey & second) const;
    VideoSelectorItemData *getDataRef() const;
    VideoSelectorItemData *getDataQry() const;
};

class ComparisonResult {
private:
    vmatch::Segments segments;

public:
    ComparisonResult();
    ComparisonResult(vmatch::Segments segments);
    vmatch::Segments getSegments() const;
    bool isEmpty() const;
};

typedef std::map<ComparisonKey, ComparisonResult> ComparisonCollection;

class ComparisonWorkerThread : public QThread
{
    Q_OBJECT;

private:
    ParamsController *params;
    VideoSelectorItemData *dataRef;
    VideoSelectorItemData *dataQry;

    ComparisonKey currentComparisonKey;
    ComparisonResult currentComparisonResult;

    ComparisonResult computeSimilarity(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry, ParamsController *params);

signals:
    void comparisonStarted(ComparisonWorkerThread *thread);
    void comparisonFinished(ComparisonWorkerThread *thread);

public:
    ComparisonWorkerThread(ParamsController *params);
    void run();
    void setDataRef(VideoSelectorItemData *data);
    void setDataQry(VideoSelectorItemData *data);
    ComparisonKey getComparisonKey() const;
    ComparisonResult getComparisonResult() const;
};

class MainWindowState
{
private:
    ComparisonKey key;
    ComparisonResult comparison;
    SimilarityTimeline::Segments segments;

    bool positionsLocked;

    int segmentRef;
    int segmentQry;
    int frameNumberRef;
    int frameNumberQry;

    int positionToFrameNumber(double position, int start, int end) const;
    double frameNumberToPosition(int frameNumber, int start, int end) const;

public:
    MainWindowState(ComparisonKey key, ComparisonResult comparison);
    ComparisonKey getComparisonKey() const;
    ComparisonResult getComparison() const;
    SimilarityTimeline::Segments getSegments() const;
    void setPositionsLocked(bool positionsLocked);
    bool isPositionsLocked() const;
    int getSegmentIdxRef() const;
    int getSegmentIdxQry() const;
    SimilarityTimeline::Segment getSegmentRef() const;
    SimilarityTimeline::Segment getSegmentQry() const;
    void setPositionRef(int segment, double position);
    void setPositionQry(int segment, double position);
    void setFrameNumberRef(int segment, int frameNumber);
    void setFrameNumberQry(int segment, int frameNumber);
    double getPositionRef() const;
    double getPositionQry() const;
    int getFrameNumberRef() const;
    int getFrameNumberQry() const;

    int stepRef(int step = 1);
    int stepQry(int step = 1);
};

class MainWindowPlayerThread : public QThread
{
    Q_OBJECT;

public:
    typedef enum
    {
        GLOBAL,
        LOCAL_REF,
        LOCAL_QRY
    } Mode;

private:
    Mode mode;
    double fps;
    bool stopRequest;

signals:
    void playbackStarted(MainWindowPlayerThread *thread);
    void playbackStopped(MainWindowPlayerThread *thread);
    void nextFrameRequested(MainWindowPlayerThread *thread, bool *requestStop);

public:
    MainWindowPlayerThread();
    void startPlayback(Mode mode, double fps);
    void stopPlayback();
    Mode getMode() const;
    double getFps() const;

private:
    void run();
};

class MainWindow : public QMainWindow
{
    Q_OBJECT

private:
    static const int STEP_SMALL = 1;
    static const int STEP_LARGE = 10;

public:
    explicit MainWindow(QWidget *parent = 0);
    ~MainWindow();
    ParamsController *getParamsController() const;


private slots:
    void on_selector_selectionChanged(VideoSelectorItemData *data);

    void on_worker_comparisonStarted(ComparisonWorkerThread *thread);
    void on_worker_comparisonFinished(ComparisonWorkerThread *thread);

    void on_player_playbackStarted(MainWindowPlayerThread *thread);
    void on_player_playbackStopped(MainWindowPlayerThread *thread);
    void on_player_nextFrameRequested(MainWindowPlayerThread *thread, bool *requestStop);

    void on_timelineRef_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal);
    void on_timelineQry_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal);
    void on_timelineLocalRef_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal);
    void on_timelineLocalQry_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal);

    void on_controlsMain_play();
    void on_controlsMain_stop();
    void on_controlsMain_goToStart();
    void on_controlsMain_goToEnd();
    void on_controlsMain_goBackwards();
    void on_controlsMain_goBackwardsFast();
    void on_controlsMain_goForwards();
    void on_controlsMain_goForwardsFast();
    void on_controlsRef_play();
    void on_controlsRef_stop();
    void on_controlsRef_goToStart();
    void on_controlsRef_goToEnd();
    void on_controlsRef_goBackwards();
    void on_controlsRef_goBackwardsFast();
    void on_controlsRef_goForwards();
    void on_controlsRef_goForwardsFast();
    void on_controlsQry_play();
    void on_controlsQry_stop();
    void on_controlsQry_goToStart();
    void on_controlsQry_goToEnd();
    void on_controlsQry_goBackwards();
    void on_controlsQry_goBackwardsFast();
    void on_controlsQry_goForwards();
    void on_controlsQry_goForwardsFast();

    void on_paramsController_paramValueChanged();

private:
    Ui::MainWindow *ui;
    MainWindowState *state;
    ProgressIndicator *progressIndicator;
    MainWindowPlayerThread *playerThread;
    ComparisonWorkerThread *workerThread;
    ComparisonCollection comparisons;

    void setComparison(ComparisonKey key);
    void startWorker(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry);
    void synchronizeState();

protected:
    virtual void resizeEvent(QResizeEvent *e);
    bool eventFilter(QObject *object, QEvent *e);
};

#endif // MAINWINDOW_H
