#include "mainwindow.h"
#include "ui_mainwindow.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    assert(ui != NULL);

    ui->setupUi(this);

    assert(ui->controlsMain != NULL);
    assert(ui->controlsRef != NULL);
    assert(ui->controlsQry != NULL);
    assert(ui->timelineLocalRef != NULL);
    assert(ui->timelineLocalQry != NULL);
    assert(ui->timelineRef != NULL);
    assert(ui->timelineQry != NULL);
    assert(ui->selectorRef != NULL);
    assert(ui->selectorQry != NULL);
    assert(ui->controlsRef != NULL);
    assert(ui->controlsQry != NULL);
    assert(ui->controlsMain != NULL);

    state = NULL;
    workerThread = NULL;

    ui->controlsMain->setEnabled(false);
    ui->controlsRef->setEnabled(false);
    ui->controlsQry->setEnabled(false);

    ui->timelineLocalRef->setHighlightSegments(false);
    ui->timelineLocalQry->setHighlightSegments(false);

    ui->timelineRef->setView(SimilarityTimeline::VIEW_REF);
    ui->timelineQry->setView(SimilarityTimeline::VIEW_QRY);
    ui->timelineLocalRef->setView(SimilarityTimeline::VIEW_REF);
    ui->timelineLocalQry->setView(SimilarityTimeline::VIEW_QRY);

    progressIndicator = new ProgressIndicator(centralWidget());
    assert(progressIndicator != NULL);

    ui->selectorRef->setImportPcDialogTitle("Otevřít kontrolní video");
    ui->selectorQry->setImportPcDialogTitle("Otevřít analyzované video");
    progressIndicator->setText("Porovnávám videa...");

    playerThread = new MainWindowPlayerThread();
    assert(playerThread != NULL);

    connect(playerThread, SIGNAL(playbackStarted(MainWindowPlayerThread*)), SLOT(on_player_playbackStarted(MainWindowPlayerThread*)));
    connect(playerThread, SIGNAL(playbackStopped(MainWindowPlayerThread*)), SLOT(on_player_playbackStopped(MainWindowPlayerThread*)));
    connect(playerThread, SIGNAL(nextFrameRequested(MainWindowPlayerThread*, bool*)), SLOT(on_player_nextFrameRequested(MainWindowPlayerThread*, bool*)));

    connect(ui->selectorRef, SIGNAL(selectionChanged(VideoSelectorItemData*)), SLOT(on_selector_selectionChanged(VideoSelectorItemData*)));
    connect(ui->selectorQry, SIGNAL(selectionChanged(VideoSelectorItemData*)), SLOT(on_selector_selectionChanged(VideoSelectorItemData*)));

    connect(ui->timelineRef, SIGNAL(segmentClicked(int,double,double)), SLOT(on_timelineRef_cursorMoved(int,double,double)));
    connect(ui->timelineQry, SIGNAL(segmentClicked(int,double,double)), SLOT(on_timelineQry_cursorMoved(int,double,double)));
    connect(ui->timelineLocalRef, SIGNAL(segmentClicked(int,double,double)), SLOT(on_timelineLocalRef_cursorMoved(int,double,double)));
    connect(ui->timelineLocalQry, SIGNAL(segmentClicked(int,double,double)), SLOT(on_timelineLocalQry_cursorMoved(int,double,double)));

    connect(ui->paramsController, SIGNAL(paramValueChanged()), SLOT(on_paramsController_paramValueChanged()));

    installEventFilter(this);

    ui->selectorRef->setFocusPolicy(Qt::NoFocus);
    ui->selectorQry->setFocusPolicy(Qt::NoFocus);
    ui->controlsRef->setFocusPolicy(Qt::NoFocus);
    ui->controlsQry->setFocusPolicy(Qt::NoFocus);
    ui->controlsMain->setFocusPolicy(Qt::NoFocus);
}

MainWindow::~MainWindow()
{
    assert(ui != NULL);

    if(playerThread != NULL)
    {
        playerThread->terminate();
        delete playerThread;
    }

    if(workerThread != NULL)
    {
        workerThread->terminate();
        delete workerThread;
    }

    if(state != NULL)
    {
        delete state;
    }

    delete ui;
}

void MainWindow::resizeEvent(QResizeEvent *e) {
    assert(progressIndicator != NULL);
    assert(e != NULL);

    progressIndicator->move(ui->layoutSelectors->geometry().x(), ui->layoutTimelines->geometry().y());
    progressIndicator->resize(ui->layoutPlayers->geometry().width(), ui->layoutPlayers->geometry().y()-ui->layoutTimelines->geometry().y()+ui->layoutPlayers->geometry().height());
    e->accept();
}

bool MainWindow::eventFilter(QObject *object, QEvent *e)
{
    Q_UNUSED(object);

    assert(e != NULL);
    assert(ui != NULL);
    assert(ui->controlsMain != NULL);

    QKeyEvent *keyEvent = dynamic_cast<QKeyEvent *>(e);
    if(keyEvent != NULL && keyEvent->type() == QEvent::Type::KeyRelease)
    {
        switch(keyEvent->key())
        {
            case Qt::Key_Space:
                switch(ui->controlsMain->getState())
                {
                    case VideoControls::STATE_IDLE:
                        on_controlsMain_play();
                        break;

                    case VideoControls::STATE_PLAYING:
                        on_controlsMain_stop();
                        break;
                }
                break;

            case Qt::Key_Left:
                on_controlsMain_goBackwards();
                break;

            case Qt::Key_Up:
                on_controlsMain_goForwardsFast();
                break;

            case Qt::Key_Down:
                on_controlsMain_goBackwardsFast();
                break;

            case Qt::Key_Right:
                on_controlsMain_goForwards();
                break;

           case Qt::Key_PageDown:
                on_controlsMain_goToStart();
                break;

           case Qt::Key_PageUp:
                on_controlsMain_goToEnd();
                break;
        }

        return true;
    }

    return false;
}

ParamsController *MainWindow::getParamsController() const
{
    //qDebug() << "MainWindow::getParamsController()";

    assert(ui != NULL);
    assert(ui->paramsController != NULL);

    return ui->paramsController;
}

void MainWindow::on_paramsController_paramValueChanged()
{
    static int i = 0;
    if(i++%2)
    {
        return;
    }

    //qDebug() << "MainWindow::on_paramsController_paramValueChanged";

    assert(progressIndicator != NULL);
    assert(ui != NULL);
    assert(ui->selectorRef != NULL);
    assert(ui->selectorQry != NULL);

    if(state != NULL)
    {
        progressIndicator->hide();
        comparisons.clear();

        VideoSelectorItemData *dataRef = ui->selectorRef->getCurrentSequenceData();
        VideoSelectorItemData *dataQry = ui->selectorQry->getCurrentSequenceData();

        if(dataRef != NULL && dataQry != NULL)
        {
            startWorker(dataRef, dataQry);
        }

        setFocus();
    }
}

void MainWindow::on_selector_selectionChanged(VideoSelectorItemData *data)
{
    qDebug() << "MainWindow::on_selector_selectionChanged";

    Q_UNUSED(data);

    assert(ui != NULL);
    assert(ui->selectorRef != NULL);
    assert(ui->selectorQry != NULL);

    qDebug() << "1" << "\n";
    VideoSelectorItemData *dataRef = ui->selectorRef->getCurrentSequenceData();
    qDebug() << "2" << "\n";
    VideoSelectorItemData *dataQry = ui->selectorQry->getCurrentSequenceData();
    qDebug() << "3" << "\n";

    if(dataRef != NULL && dataQry != NULL)
    {
        qDebug() << "4" << "\n";
        ComparisonKey key(dataRef, dataQry);
qDebug() << "5" << "\n";
        if(comparisons.find(key) != comparisons.end())
        {
            qDebug() << "6" << "\n";
            if(workerThread != NULL && workerThread->isRunning())
            {
                qDebug() << "7" << "\n";
                workerThread->terminate();
                qDebug() << "8" << "\n";
            }
qDebug() << "9" << "\n";
            setComparison(key);
            qDebug() << "10" << "\n";
        }
        else
        {
            qDebug() << "11" << "\n";
            startWorker(dataRef, dataQry);
            qDebug() << "12" << "\n";
        }
    }
qDebug() << "13" << "\n";
    setFocus();
    qDebug() << "14" << "\n";
}

void MainWindow::on_timelineRef_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal)
{
    //qDebug() << "MainWindow::on_timelineRef_cursorMoved";

    Q_UNUSED(positionGlobal);

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->setPositionRef(segmentIdx, positionSegment);
        synchronizeState();
    }
}

void MainWindow::on_timelineQry_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal)
{
    //qDebug() << "MainWindow::on_timelineQry_cursorMoved";

    Q_UNUSED(positionGlobal);

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->setPositionQry(segmentIdx, positionSegment);
        synchronizeState();
    }
}

void MainWindow::on_timelineLocalRef_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal)
{
    //qDebug() << "MainWindow::on_timelineLocalRef_cursorMoved";

    Q_UNUSED(positionGlobal);

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionRef(segmentIdx, positionSegment);
        synchronizeState();
    }
}

void MainWindow::on_timelineLocalQry_cursorMoved(int segmentIdx, double positionSegment, double positionGlobal)
{
    //qDebug() << "MainWindow::on_timelineLocalQry_cursorMoved";

    Q_UNUSED(positionGlobal);

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionQry(segmentIdx, positionSegment);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_play()
{
    //qDebug() << "MainWindow::on_controlsMain_play";

    if(state != NULL)
    {
        playerThread->startPlayback(MainWindowPlayerThread::GLOBAL, (state->getComparisonKey().getDataRef()->getData()->getMetaData().fps + state->getComparisonKey().getDataQry()->getData()->getMetaData().fps)/2);
    }
}

void MainWindow::on_controlsMain_stop()
{
    //qDebug() << "MainWindow::on_controlsMain_stop";

    if(state != NULL)
    {
        playerThread->stopPlayback();
    }
}

void MainWindow::on_controlsMain_goToStart()
{
    //qDebug() << "MainWindow::on_controlsMain_goToStart";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->setPositionRef(state->getSegmentIdxRef(), 0);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_goToEnd()
{
    //qDebug() << "MainWindow::on_controlsMain_goToEnd";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->setPositionRef(state->getSegmentIdxRef(), 1);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_goBackwards()
{
    //qDebug() << "MainWindow::on_controlsMain_goBackwards";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->stepRef(-STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_goBackwardsFast()
{
    //qDebug() << "MainWindow::on_controlsMain_goBackwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->stepRef(-STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_goForwards()
{
    //qDebug() << "MainWindow::on_controlsMain_goForwards";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->stepRef(STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsMain_goForwardsFast()
{
    //qDebug() << "MainWindow::on_controlsMain_goForwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(true);
        state->stepRef(STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_play()
{
    //qDebug() << "MainWindow::on_controlsRef_play";

    assert(playerThread != NULL);

    if(state != NULL)
    {
        playerThread->startPlayback(MainWindowPlayerThread::LOCAL_REF, state->getComparisonKey().getDataRef()->getData()->getMetaData().fps);
    }
}

void MainWindow::on_controlsRef_stop()
{
    //qDebug() << "MainWindow::on_controlsRef_stop";

    assert(playerThread != NULL);

    if(state != NULL)
    {
        playerThread->stopPlayback();
    }
}

void MainWindow::on_controlsRef_goToStart()
{
    //qDebug() << "MainWindow::on_controlsRef_goToStart";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionRef(state->getSegmentIdxRef(), 0);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_goToEnd()
{
    //qDebug() << "MainWindow::on_controlsRef_goToEnd";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionRef(state->getSegmentIdxRef(), 1);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_goBackwards()
{
    //qDebug() << "MainWindow::on_controlsRef_goBackwards";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepRef(-STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_goBackwardsFast()
{
    //qDebug() << "MainWindow::on_controlsRef_goBackwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepRef(-STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_goForwards()
{
    //qDebug() << "MainWindow::on_controlsRef_goForwards";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepRef(STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsRef_goForwardsFast()
{
    //qDebug() << "MainWindow::on_controlsRef_goForwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepRef(STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_play()
{
    //qDebug() << "MainWindow::on_controlsQry_play";

    assert(playerThread != NULL);

    if(state != NULL)
    {
        playerThread->startPlayback(MainWindowPlayerThread::LOCAL_QRY, state->getComparisonKey().getDataQry()->getData()->getMetaData().fps);
    }
}

void MainWindow::on_controlsQry_stop()
{
    //qDebug() << "MainWindow::on_controlsQry_stop";

    assert(playerThread != NULL);

    if(state != NULL)
    {
        playerThread->stopPlayback();
    }
}

void MainWindow::on_controlsQry_goToStart()
{
    //qDebug() << "MainWindow::on_controlsQry_goToStart";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionQry(state->getSegmentIdxQry(), 0);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_goToEnd()
{
    //qDebug() << "MainWindow::on_controlsQry_goToEnd";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->setPositionQry(state->getSegmentIdxQry(), 1);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_goBackwards()
{
    //qDebug() << "MainWindow::on_controlsQry_goBackwards";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepQry(-STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_goBackwardsFast()
{
    //qDebug() << "MainWindow::on_controlsQry_goBackwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepQry(-STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_goForwards()
{
    //qDebug() << "MainWindow::on_controlsQry_goForwards";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepQry(STEP_SMALL);
        synchronizeState();
    }
}

void MainWindow::on_controlsQry_goForwardsFast()
{
    //qDebug() << "MainWindow::on_controlsQry_goForwardsFast";

    if(state != NULL)
    {
        state->setPositionsLocked(false);
        state->stepQry(STEP_LARGE);
        synchronizeState();
    }
}

void MainWindow::on_player_playbackStarted(MainWindowPlayerThread *thread)
{
    //qDebug() << "MainWindow::on_player_playbackStarted";

    assert(thread != NULL);
    assert(ui != NULL);
    assert(ui->controlsMain != NULL);
    assert(ui->controlsRef != NULL);
    assert(ui->controlsQry != NULL);

    switch(thread->getMode())
    {
        case MainWindowPlayerThread::GLOBAL:
            ui->controlsMain->setState(VideoControls::STATE_PLAYING);
            ui->controlsMain->setEnabled(true);
            ui->controlsRef->setEnabled(false);
            ui->controlsQry->setEnabled(false);
            break;

        case MainWindowPlayerThread::LOCAL_REF:
            ui->controlsRef->setState(VideoControls::STATE_PLAYING);
            ui->controlsMain->setEnabled(false);
            ui->controlsRef->setEnabled(true);
            ui->controlsQry->setEnabled(false);
            break;

        case MainWindowPlayerThread::LOCAL_QRY:
            ui->controlsQry->setState(VideoControls::STATE_PLAYING);
            ui->controlsMain->setEnabled(false);
            ui->controlsRef->setEnabled(false);
            ui->controlsQry->setEnabled(true);
            break;
    }
}

void MainWindow::on_player_playbackStopped(MainWindowPlayerThread *thread)
{
    //qDebug() << "MainWindow::on_player_playbackStopped";

    Q_UNUSED(thread);

    assert(ui != NULL);
    assert(ui->controlsMain != NULL);
    assert(ui->controlsRef != NULL);
    assert(ui->controlsQry != NULL);

    ui->controlsMain->setEnabled(true);
    ui->controlsRef->setEnabled(true);
    ui->controlsQry->setEnabled(true);

    ui->controlsMain->setState(VideoControls::STATE_IDLE);
    ui->controlsRef->setState(VideoControls::STATE_IDLE);
    ui->controlsQry->setState(VideoControls::STATE_IDLE);
}

void MainWindow::on_player_nextFrameRequested(MainWindowPlayerThread *thread, bool *requestStop)
{
    //qDebug() << "MainWindow::on_player_nextFrameRequested";

    assert(state != NULL);
    assert(thread != NULL);
    assert(requestStop != NULL);

    switch(thread->getMode())
    {
        case MainWindowPlayerThread::GLOBAL:
            on_controlsMain_goForwards();
            *requestStop = state->getPositionRef() >= 1 && state->getSegmentIdxRef() >= (signed)state->getSegments().size()-1;
            break;

        case MainWindowPlayerThread::LOCAL_REF:
            on_controlsRef_goForwards();
            *requestStop = state->getPositionRef() >= 1 && state->getSegmentIdxRef() >= (signed)state->getSegments().size()-1;
            break;

        case MainWindowPlayerThread::LOCAL_QRY:
            on_controlsQry_goForwards();
            *requestStop = state->getPositionQry() >= 1 && state->getSegmentIdxQry() >= (signed)state->getSegments().size()-1;
            break;
    }
}

void MainWindow::synchronizeState()
{
    //qDebug() << "MainWindow::synchronizeState";

    assert(ui != NULL);
    assert(ui->timelineRef != NULL);
    assert(ui->timelineQry != NULL);
    assert(ui->timelineLocalRef != NULL);
    assert(ui->timelineLocalQry != NULL);
    assert(ui->playerRef != NULL);
    assert(ui->playerQry != NULL);

    if(state != NULL)
    {
        ui->timelineRef->setCursorPositionSegment(state->getSegmentIdxRef(), state->getPositionRef());
        ui->timelineQry->setCursorPositionSegment(state->getSegmentIdxQry(), state->getPositionQry());

        ui->timelineLocalRef->setCursorPositionSegment(state->getSegmentIdxRef(), state->getPositionRef());
        ui->timelineLocalQry->setCursorPositionSegment(state->getSegmentIdxQry(), state->getPositionQry());

        SimilarityTimeline::Segment segmentRef = state->getSegmentRef();
        SimilarityTimeline::Segment segmentQry = state->getSegmentQry();

        if(segmentRef.getDissimilarityType() != SimilarityTimeline::Segment::DIS_INJECTION)
        {
            int lengthRef = segmentRef.getEndRef() - segmentRef.getStartRef();
            int posRef = segmentRef.getStartRef() + ceil(state->getPositionRef()*lengthRef);
            ui->playerRef->setPosition(posRef);
        }

        if(segmentQry.getDissimilarityType() != SimilarityTimeline::Segment::DIS_REMOVAL)
        {
            int lengthQry = segmentQry.getEndQry() - segmentQry.getStartQry();
            int posQry = segmentQry.getStartQry() + ceil(state->getPositionQry()*lengthQry);
            ui->playerQry->setPosition(posQry);
        }
    }
}

void MainWindow::startWorker(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry)
{
    qDebug() << "MainWindow::startWorker";

    if(workerThread != NULL && workerThread->isRunning())
    {
        workerThread->terminate();
        while(workerThread->isRunning()) {}
        delete workerThread;
        workerThread = NULL;
    }

    workerThread = new ComparisonWorkerThread(ui->paramsController);
    connect(workerThread, SIGNAL(comparisonStarted(ComparisonWorkerThread*)), SLOT(on_worker_comparisonStarted(ComparisonWorkerThread*)));
    connect(workerThread, SIGNAL(comparisonFinished(ComparisonWorkerThread*)), SLOT(on_worker_comparisonFinished(ComparisonWorkerThread*)));

    assert(workerThread != NULL);

    workerThread->setDataRef(dataRef);
    workerThread->setDataQry(dataQry);
    workerThread->start();
}

void MainWindow::on_worker_comparisonStarted(ComparisonWorkerThread *thread)
{
    //qDebug() << "MainWindow::on_worker_comparisonStarted";

    Q_UNUSED(thread);

    assert(progressIndicator != NULL);
    assert(thread != NULL);

    progressIndicator->show();
}

void MainWindow::on_worker_comparisonFinished(ComparisonWorkerThread *thread)
{
    //qDebug() << "MainWindow::on_worker_comparisonFinished";

    assert(progressIndicator != NULL);
    assert(thread != NULL);

    comparisons[thread->getComparisonKey()] = thread->getComparisonResult();
    progressIndicator->hide();
    setComparison(thread->getComparisonKey());
}

void MainWindow::setComparison(ComparisonKey key)
{
    //qDebug() << "MainWindow::setComparison";

    if(comparisons.find(key) == comparisons.end())
    {
        return;
    }

    MainWindowState *oldState = state;

    state = new MainWindowState(key, comparisons[key]);
    assert(state != NULL);

    if(oldState != NULL)
    {
        delete oldState;
    }

    assert(ui != NULL);
    assert(ui->timelineRef != NULL);
    assert(ui->timelineQry != NULL);
    assert(ui->timelineLocalRef != NULL);
    assert(ui->timelineLocalQry != NULL);
    assert(ui->controlsMain != NULL);
    assert(ui->controlsRef != NULL);
    assert(ui->controlsQry != NULL);
    assert(progressIndicator != NULL);

    if(state->getSegments().size() > 0)
    {
        ui->timelineRef->setSegments(state->getSegments());
        ui->timelineRef->setCursorPositionGlobal(0);

        ui->timelineQry->setSegments(state->getSegments());
        ui->timelineQry->setCursorPositionGlobal(0);

        ui->timelineLocalRef->setSegments(state->getSegments());
        ui->timelineLocalRef->setCursorPositionGlobal(0);

        ui->timelineLocalQry->setSegments(state->getSegments());
        ui->timelineLocalQry->setCursorPositionGlobal(0);

        ui->playerRef->open(key.getDataRef()->getFilename());
        ui->playerQry->open(key.getDataQry()->getFilename());
    }

    ui->controlsMain->setEnabled(true);
    ui->controlsRef->setEnabled(true);
    ui->controlsQry->setEnabled(true);

    progressIndicator->hide();
}

ComparisonKey::ComparisonKey()
{
    //qDebug() << "ComparisonKey::ComparisonKey 1";

    dataRef = NULL;
    dataQry = NULL;
}

ComparisonKey::ComparisonKey(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry)
{
    //qDebug() << "ComparisonKey::ComparisonKey 2";

    assert(dataRef != NULL);
    assert(dataQry != NULL);

    this->dataRef = dataRef;
    this->dataQry = dataQry;
}

bool ComparisonKey::operator==(const ComparisonKey & second) const
{
    return this->dataRef == second.dataRef && this->dataQry == second.dataQry;
}

bool ComparisonKey::operator<(const ComparisonKey & second) const
{
    if(this->dataRef != second.dataRef)
    {
        return this->dataRef < second.dataRef;
    }

    return this->dataQry < second.dataQry;
}

VideoSelectorItemData *ComparisonKey::getDataRef() const
{
    return dataRef;
}

VideoSelectorItemData *ComparisonKey::getDataQry() const
{
    return dataQry;
}

ComparisonResult::ComparisonResult()
{
    segments = vmatch::Segments();
}

ComparisonResult::ComparisonResult(vmatch::Segments segments)
{
    this->segments = segments;
}

vmatch::Segments ComparisonResult::getSegments() const
{
    return segments;
}

bool ComparisonResult::isEmpty() const
{
    return segments.size() == 0;
}

ComparisonWorkerThread::ComparisonWorkerThread(ParamsController *params) : QThread()
{
    //qDebug() << "ComparisonWorkerThread::ComparisonWorkerThread";

    assert(params != NULL);

    this->dataRef = NULL;
    this->dataQry = NULL;
    this->params = params;
}

void ComparisonWorkerThread::run()
{
    //qDebug() << "ComparisonWorkerThread::run";

    currentComparisonKey = ComparisonKey(dataRef, dataQry);
    currentComparisonResult = ComparisonResult();

    emit comparisonStarted(this);

    if(dataRef != NULL && dataQry != NULL)
    {
        currentComparisonResult = computeSimilarity(dataRef, dataQry, params);
    }

    emit comparisonFinished(this);
}

ComparisonResult ComparisonWorkerThread::computeSimilarity(VideoSelectorItemData *dataRef, VideoSelectorItemData *dataQry, ParamsController *params)
{
    //qDebug() << "ComparisonWorkerThread::computeSimilarity";

    assert(dataRef != NULL);
    assert(dataQry != NULL);
    assert(params != NULL);
    assert(dataRef->getData() != NULL);
    assert(dataQry->getData() != NULL);

    vmatch::BruteForceVideoComparer videoComparer;
    vmatch::NeedlemanWunschSegmentsExtractor segmentsExtractor;

    segmentsExtractor.setGeometricSubdivisionThreshold(params->getGeometricSubdivisionThresholdValue());
    segmentsExtractor.setSimilaritySubdivisionThreshold(params->getSimilaritySubdivisionThresholdValue());
    segmentsExtractor.setMinSegmentLength(params->getMinSegmentLengthValue());

    vmatch::SimilarityMatrix similarity = videoComparer.compare(dataRef->getData()->getKeyFrames(), dataQry->getData()->getKeyFrames());
    vmatch::Segments segments = segmentsExtractor.extract(similarity, dataRef->getData(), dataQry->getData());

    return ComparisonResult(segments);
}

void ComparisonWorkerThread::setDataRef(VideoSelectorItemData *data)
{
    //qDebug() << "ComparisonWorkerThread::setDataRef";

    assert(data != NULL);
    dataRef = data;
}

void ComparisonWorkerThread::setDataQry(VideoSelectorItemData *data)
{
    //qDebug() << "ComparisonWorkerThread::setDataQry";

    assert(data != NULL);
    dataQry = data;
}

ComparisonKey ComparisonWorkerThread::getComparisonKey() const
{
    //qDebug() << "ComparisonWorkerThread::getComparisonKey()";

    return currentComparisonKey;
}

ComparisonResult ComparisonWorkerThread::getComparisonResult() const
{
    //qDebug() << "ComparisonWorkerThread::getComparisonResult";

    return currentComparisonResult;
}

MainWindowState::MainWindowState(ComparisonKey key, ComparisonResult comparison)
{
    //qDebug() << "MainWindowState::MainWindowState";

    this->key = key;
    this->comparison = comparison;
    this->segments = SimilarityTimeline::Segments(comparison.getSegments());

    this->segmentRef = 0;
    this->segmentQry = 0;
    this->frameNumberRef = 0;
    this->frameNumberQry = 0;

    this->positionsLocked = true;
}

int MainWindowState::positionToFrameNumber(double position, int start, int end) const
{
    return ceil(position*(end-start));
}

double MainWindowState::frameNumberToPosition(int frameNumber, int start, int end) const
{
    return ((double)frameNumber)/(end-start);
}

ComparisonKey MainWindowState::getComparisonKey() const
{
    return key;
}

ComparisonResult MainWindowState::getComparison() const
{
    return comparison;
}

SimilarityTimeline::Segments MainWindowState::getSegments() const
{
    return segments;
}

void MainWindowState::setPositionsLocked(bool positionsLocked)
{
    //qDebug() << "MainWindowState::setPositionsLocked";

    this->positionsLocked = positionsLocked;
}

bool MainWindowState::isPositionsLocked() const
{
    //qDebug() << "MainWindowState::isPositionsLocked";

    return positionsLocked;
}

void MainWindowState::setPositionRef(int segment, double position)
{
    //qDebug() << "MainWindowState::setPositionRef";

    this->segmentRef = segment;
    this->frameNumberRef = positionToFrameNumber(position, getSegmentRef().getStartRef(), getSegmentRef().getEndRef());

    if(positionsLocked)
    {
        this->segmentQry = segment;
        this->frameNumberQry = positionToFrameNumber(position, getSegmentQry().getStartQry(), getSegmentQry().getEndQry());
    }
}

void MainWindowState::setPositionQry(int segment, double position)
{
    //qDebug() << "MainWindowState::setPositionQry";

    this->segmentQry = segment;
    this->frameNumberQry = positionToFrameNumber(position, getSegmentQry().getStartQry(), getSegmentQry().getEndQry());

    if(positionsLocked)
    {
        this->segmentRef = segment;
        this->frameNumberRef = positionToFrameNumber(position, getSegmentRef().getStartRef(), getSegmentRef().getEndRef());
    }
}

void MainWindowState::setFrameNumberRef(int segment, int frameNumber)
{
    //qDebug() << "MainWindowState::setFrameNumberRef";

    this->segmentRef = segment;
    this->frameNumberRef = frameNumber;

    if(positionsLocked)
    {
        this->segmentQry = segment;
        this->frameNumberQry = positionToFrameNumber(getPositionRef(), getSegmentQry().getStartQry(), getSegmentQry().getEndQry());
    }
}

void MainWindowState::setFrameNumberQry(int segment, int frameNumber)
{
    //qDebug() << "MainWindowState::setFrameNumberQry";

    this->segmentQry = segment;
    this->frameNumberQry = frameNumber;

    if(positionsLocked)
    {
        this->segmentRef = segment;
        this->frameNumberRef = positionToFrameNumber(getPositionQry(), getSegmentRef().getStartRef(), getSegmentRef().getEndRef());
    }
}

int MainWindowState::getSegmentIdxRef() const
{
    return segmentRef;
}

int MainWindowState::getSegmentIdxQry() const
{
    return segmentQry;
}

double MainWindowState::getPositionRef() const
{
    return frameNumberToPosition(frameNumberRef, getSegmentRef().getStartRef(), getSegmentRef().getEndRef());
}

double MainWindowState::getPositionQry() const
{
    return frameNumberToPosition(frameNumberQry, getSegmentQry().getStartQry(), getSegmentQry().getEndQry());
}

int MainWindowState::getFrameNumberRef() const
{
    return frameNumberRef;
}

int MainWindowState::getFrameNumberQry() const
{
    return frameNumberQry;
}

SimilarityTimeline::Segment MainWindowState::getSegmentRef() const
{
    return segments[segmentRef];
}

SimilarityTimeline::Segment MainWindowState::getSegmentQry() const
{
    return segments[segmentQry];
}

int MainWindowState::stepRef(int step)
{
    //qDebug() << " MainWindowState::stepRef";

    int stepLeft = abs(step);
    int currentSegmentIdx = getSegmentIdxRef();
    int currentFrameNumber = getFrameNumberRef();

    while(stepLeft > 0 && currentSegmentIdx >= 0 && currentSegmentIdx < segments.size())
    {
        SimilarityTimeline::Segment currentSegment = segments[currentSegmentIdx];
        if(currentSegment.getDissimilarityType() != SimilarityTimeline::Segment::DIS_INJECTION)
        {
            int lengthSegmentFrames = currentSegment.getEndRef() - currentSegment.getStartRef();
            int maxStep = (step > 0)? lengthSegmentFrames-currentFrameNumber : currentFrameNumber;

            if(stepLeft > maxStep)
            {
                if(step > 0)
                {
                    stepLeft -= maxStep;
                    currentSegmentIdx++;
                    currentFrameNumber = 0;
                }
                else
                {
                    stepLeft -= maxStep;
                    currentSegmentIdx--;
                    if(currentSegmentIdx >= 0)
                    {
                        currentFrameNumber = segments[currentSegmentIdx].getEndRef() - segments[currentSegmentIdx].getStartRef();
                    }
                }
            }
            else
            {
                setFrameNumberRef(currentSegmentIdx, (step > 0)? currentFrameNumber+stepLeft : currentFrameNumber-stepLeft);
                return abs(step);
            }
        }
        else
        {
            if(step > 0)
            {
                currentSegmentIdx++;
                currentFrameNumber = 0;
            }
            else
            {
                currentSegmentIdx--;
                if(currentSegmentIdx >= 0)
                {
                    currentFrameNumber = segments[currentSegmentIdx].getEndRef() - segments[currentSegmentIdx].getStartRef();
                }
            }
        }
    }

    bool locked = isPositionsLocked();
    setPositionsLocked(false);
    if(step > 0)
    {
        setPositionRef((int)segments.size()-1, 1);
    }
    else
    {
        setPositionRef(0, 0);
    }
    setPositionsLocked(locked);

    return abs(step)-stepLeft;
}

int MainWindowState::stepQry(int step)
{
    //qDebug() << "MainWindowState::stepQry";

    int stepLeft = abs(step);
    int currentSegmentIdx = getSegmentIdxQry();
    int currentFrameNumber = getFrameNumberQry();

    while(stepLeft > 0 && currentSegmentIdx >= 0 && currentSegmentIdx < segments.size())
    {
        SimilarityTimeline::Segment currentSegment = segments[currentSegmentIdx];
        if(currentSegment.getDissimilarityType() != SimilarityTimeline::Segment::DIS_INJECTION)
        {
            int lengthSegmentFrames = currentSegment.getEndQry() - currentSegment.getStartQry();
            int maxStep = (step > 0)? lengthSegmentFrames-currentFrameNumber : currentFrameNumber;

            if(stepLeft > maxStep)
            {
                if(step > 0)
                {
                    stepLeft -= maxStep;
                    currentSegmentIdx++;
                    currentFrameNumber = 0;
                }
                else
                {
                    stepLeft -= maxStep;
                    currentSegmentIdx--;
                    if(currentSegmentIdx >= 0)
                    {
                        currentFrameNumber = segments[currentSegmentIdx].getEndQry() - segments[currentSegmentIdx].getStartQry();
                    }
                }
            }
            else
            {
                setFrameNumberQry(currentSegmentIdx, (step > 0)? currentFrameNumber+stepLeft : currentFrameNumber-stepLeft);
                return abs(step);
            }
        }
        else
        {
            if(step > 0)
            {
                currentSegmentIdx++;
                currentFrameNumber = 0;
            }
            else
            {
                currentSegmentIdx--;
                if(currentSegmentIdx >= 0)
                {
                    currentFrameNumber = segments[currentSegmentIdx].getEndQry() - segments[currentSegmentIdx].getStartQry();
                }
            }
        }
    }

    bool locked = isPositionsLocked();
    setPositionsLocked(false);
    if(step > 0)
    {
        setPositionQry((int)segments.size()-1, 1);
    }
    else
    {
        setPositionQry(0, 0);
    }
    setPositionsLocked(locked);

    return abs(step)-stepLeft;
}

MainWindowPlayerThread::MainWindowPlayerThread() : QThread()
{
}

void MainWindowPlayerThread::startPlayback(MainWindowPlayerThread::Mode mode, double fps)
{
    //qDebug() << "MainWindowPlayerThread::startPlayback";

    this->mode = mode;
    this->fps = fps;
    this->stopRequest = false;
    this->start();
}

void MainWindowPlayerThread::stopPlayback()
{
    //qDebug() << "MainWindowPlayerThread::stopPlayback";

    stopRequest = true;
}

MainWindowPlayerThread::Mode MainWindowPlayerThread::getMode() const
{
    return mode;
}

double MainWindowPlayerThread::getFps() const
{
    return fps;
}

void MainWindowPlayerThread::run()
{
    //qDebug() << "MainWindowPlayerThread::run";

    emit playbackStarted(this);

    while(!stopRequest)
    {
        emit nextFrameRequested(this, &stopRequest);
        msleep(1000.0/fps);
    }

    emit playbackStopped(this);
}
