#include "SimilarityMatrix.h"


namespace vmatch {


SimilarityMatrix::SimilarityMatrix() : cv::Mat() {

}


SimilarityMatrix::SimilarityMatrix(const Mat & data) : cv::Mat(data.rows, data.cols, CV_32F) {
	if(data.type() != CV_32F || data.rows == 0 || data.cols == 0) {
		CV_Error(CV_StsBadArg, "Similarity matrix has to be a CV_32F type, non-empty matrix.");
	}
	
	float *srcPtr = (float *)data.data;
	float *dstPtr = (float *)this->data;

	CV_Assert(srcPtr != NULL);
	CV_Assert(dstPtr != NULL);

	for(int y = 0; y < rows; y++) {
		for(int x = 0; x < cols; x++) {
			dstPtr[y*cols + x] = srcPtr[y*cols + x];
		}
	}
}


SimilarityMatrix SimilarityMatrix::normalize() const {
	float min = *std::min_element(this->begin<float>(),this->end<float>());
	float max = *std::max_element(this->begin<float>(),this->end<float>());

	Mat out = cv::Mat::zeros(rows, cols, CV_32F);

	float *srcData = (float *)data;
	float *dstData = (float *)out.data;

	CV_Assert(srcData != NULL);
	CV_Assert(dstData != NULL);

	for(int i = 0; i < rows*cols; i++) {
		dstData[i] = (srcData[i]-min)/max;
	}

	return SimilarityMatrix(out);
}


void SimilarityMatrix::threshold(Mat & out, double threshold, bool inverse) const {
	if(threshold <= 0) {
		CV_Error(CV_StsBadArg, "Invalid similarity matrix threshold.");
	}

	out = cv::Mat::zeros(rows, cols, CV_8UC1);
	
	float *srcData = (float *)data;
	uchar *dstData = (uchar *)out.data;

	CV_Assert(srcData != NULL);
	CV_Assert(dstData != NULL);

	uchar above = (inverse)? 0 : 255;
	uchar below = 255 - above;

	for(int i = 0; i < rows*cols; i++) {
		if(srcData[i] > threshold) {
			dstData[i] = above;
		}
		else {
			dstData[i] = below;
		}
	}
}


void SimilarityMatrix::render(Mat & frame) const {
	float min = *std::min_element(this->begin<float>(),this->end<float>());
	float max = *std::max_element(this->begin<float>(),this->end<float>());

	frame = cv::Mat::zeros(rows, cols, CV_8UC1);

	float *srcData = (float *)data;
	uchar *dstData = (uchar *)frame.data;

	CV_Assert(srcData != NULL);
	CV_Assert(dstData != NULL);

	for(int i = 0; i < rows*cols; i++) {
		dstData[i] = (int)(((srcData[i]-min)/max)*255);
	}
}


cv::Mat SimilarityMatrix::createDataMatrix(int rows, int cols) {	
	if(rows <= 0 || cols <= 0) {
		CV_Error(CV_StsBadArg, "Invalid similarity matrix dimensions.");
	}
	
	cv::Mat data(rows, cols, CV_32F);	
	float *dataPtr = (float *)data.data;

	CV_Assert(dataPtr != NULL);

	for(int y = 0; y < rows; y++) {
		for(int x = 0; x < cols; x++) {
			dataPtr[y*cols + x] = FLT_MAX;
		}
	}

	return data;
}


}
