#include "FrameDescriptorExtractor.h"


namespace vmatch {


FrameDescriptorExtractorFactory::LookupTable FrameDescriptorExtractorFactory::lookup;
bool FrameDescriptorExtractorFactory::initialized = false;


void FrameDescriptorExtractorFactory::add(int type, FrameDescriptorExtractorPtr extractor) {
	CV_Assert(extractor != NULL);
	lookup[type] = extractor;
	initialized = true;
}


void FrameDescriptorExtractorFactory::remove(int type, FrameDescriptorExtractorPtr extractor) {
	if(isAssigned(type)) {
		lookup.erase(type);
	}
}


void FrameDescriptorExtractorFactory::addDefaults() {
	add(FrameDescriptor::COLHIST, new ColHistFrameDescriptorExtractor());
	add(FrameDescriptor::GRADHIST, new GradHistFrameDescriptorExtractor());
	add(FrameDescriptor::GRADRGHIST, new GradRGHistFrameDescriptorExtractor());
	add(FrameDescriptor::QUICKHIST, new QuickHistogramFrameDescriptorExtractor());
	add(FrameDescriptor::LOCAL_SURF, new SurfFrameDescriptorExtractor());
}


bool FrameDescriptorExtractorFactory::isAssigned(int type) {
	return lookup.find(type) != lookup.end();
}


FrameDescriptorExtractorPtr FrameDescriptorExtractorFactory::create(int type) {	
	if(!initialized) {
		addDefaults();
	}
	
	if(!isAssigned(type)) {
		CV_Error(CV_StsObjectNotFound, "No descriptor extractor is assigned to handle the requested descriptor type.");
	}

	// TODO technicky neni tovarna - predava se jenom jeden poiner;
	//      v implementaci nevadi, tak to tak zatim nechavam,
	//      ale neni to spravne - mel by se vytvorit novy objekt
	return lookup[type];
}


FrameDescriptorExtractor::FrameDescriptorExtractor() {
	defaults();
}


void FrameDescriptorExtractor::defaults() {
}


void FrameDescriptorExtractor::rgb2IOO(const cv::Mat & src, cv::Mat & dst) {
	if(src.channels() != 3) return; // TODO vyjimka

	dst = cv::Mat(src.rows, src.cols, CV_8UC3);

	uchar *srcPtr = (uchar *)src.data;
	uchar *dstPtr = (uchar *)dst.data;

	long index,index2;
	float r, g, b;
	for(int y = 0; y < src.rows; y++) {
		for(int x = 0; x < src.cols; x++) {
			index = y*src.step + x*3;
			index2 = y*dst.cols*3 + x*3;
			r = srcPtr[index+2];
			g = srcPtr[index+1];
			b = srcPtr[index];

			dstPtr[index2] = (uchar)((r + g + b) / 3.0);
			dstPtr[index2+1] = (uchar)((r + g - 2*b) / 4.0 + 127.5);
			dstPtr[index2+2] = (uchar)((r - 2*g + b) / 4.0 + 127.5);
		}
	}
}


void FrameDescriptorExtractor::rgb2rg(const cv::Mat & src, cv::Mat & dst) {
	if(src.channels() != 3) return; // TODO vyjimka

	
	dst = cv::Mat(src.rows, src.cols, CV_8UC3);

	uchar *srcPtr = (uchar *)src.data;
	uchar *dstPtr = (uchar *)dst.data;
	
	long index,index2;
	float r, g, b;
	for(int y = 0; y < src.rows; y++) {
		for(int x = 0; x < src.cols; x++) {
			index = y*src.step + x*3;
			index2 = y*dst.cols*3 + x*3;
			r = srcPtr[index+2];
			g = srcPtr[index+1];
			b = srcPtr[index];

			if((r+g+b) > 0)
			{
				dstPtr[index2+2] = (uchar)((r / (r + g + b)) * 255.0);
				dstPtr[index2+1] = (uchar)((g / (r + g + b)) * 255.0);
				dstPtr[index2] = (uchar)((b / (r + g + b)) * 255.0);
			}
			else
			{
				dstPtr[index2+2] = 0;
				dstPtr[index2+1] = 0;
				dstPtr[index2] = 0;
			}
		}
	}
}


void FrameDescriptorExtractor::rgb2gradients(const cv::Mat & src, cv::Mat & dst) {
	cv::Mat srcGray;

	cv::cvtColor(src, srcGray, CV_RGB2GRAY);

	cv::Mat X, Y, tmpX, tmpY;

	cv::Sobel(srcGray, tmpX, CV_16S, 1, 0, 3, 1, 0, cv::BORDER_DEFAULT);
	cv::Sobel(srcGray, tmpY, CV_16S, 0, 1, 3, 1, 0, cv::BORDER_DEFAULT);

	cv::convertScaleAbs(tmpX, X);
	cv::convertScaleAbs(tmpY, Y);

	addWeighted(X, 0.5, Y, 0.5, 0, dst);

	cv::Mat tmp;
	cv::normalize(dst, dst, 255, 0, cv::NORM_MINMAX);
}


void FrameDescriptorExtractor::tileHistCompute3(const cv::Mat & img, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1, int histBinsCH2, int histBinsCH3) {
	int index;
	int a, b, c;

	double coefA = 1.* histBinsCH1 / 256;
	double coefB = 1.* histBinsCH2 / 256;
	double coefC = 1.* histBinsCH3 / 256;

	
	float *kernelPtr = (float *)kernel.data;
	float kernelValue;

	uchar *imgPtr = (uchar *)img.data;
	float *tileHistPtr = (float *)tileHist.data;

	// indexace gaussovky
	int kX = 0; 
	int kY = 0;
	//indexace pro aktualni dlazdici
	for(int y = tile.y; y < tile.height + tile.y; y++)
	{
		for(int x = tile.x; x < tile.width + tile.x; x++)
		{
			index = y*img.cols*3 + x*3;
			
			//vypocet binu pro kazdy kanal
			a = cvFloor(imgPtr[index]*coefA);
			b = cvFloor(imgPtr[index+1]*coefB) + histBinsCH1;
			c = cvFloor(imgPtr[index+2]*coefC) + histBinsCH1 + histBinsCH2;

			//prispevek do binu pro kazdy kanal
			kernelValue = kernelPtr[kY*tile.width + kX];
			tileHistPtr[a] += 1 * kernelValue;//vahovani prispevku podle gaussovy funkce
			tileHistPtr[b] += 1 * kernelValue;
			tileHistPtr[c] += 1 * kernelValue;

			kX++;
		}
		kY++;
		kX = 0;
	}
}


void FrameDescriptorExtractor::tileHistCompute3I2(const cv::Mat & img1, const cv::Mat & img2, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1, int histBinsCH2, int histBinsCH3) {
	int index, index2;
	int a, r, g;

	double coefA = 1.* histBinsCH1 / 256;
	double coefB = 1.* histBinsCH2 / 256;
	double coefC = 1.* histBinsCH3 / 256;

	
	float *kernelPtr = (float *)kernel.data;
	float kernelValue;

	uchar *imgPtr = (uchar *)img1.data;
	uchar *img2Ptr = (uchar *)img2.data;
	float *tileHistPtr = (float *)tileHist.data;

	// indexace gaussovky
	int kX = 0; 
	int kY = 0;

	//indexace pro aktualni dlazdici
	for(int y = tile.y; y < tile.height + tile.y; y++)
	{
		for(int x = tile.x; x < tile.width + tile.x; x++)
		{
			index = y*img1.cols + x;
			index2 = y*img2.cols*3 + x*3;
			
			//vypocet binu pro kazdy kanal
			a = cvFloor(imgPtr[index]*coefA);
			r = cvFloor(img2Ptr[index+2]*coefB) + histBinsCH1;
			g = cvFloor(img2Ptr[index+1]*coefC) + histBinsCH1 + histBinsCH2;

			//prispevek do binu pro kazdy kanal
			kernelValue = kernelPtr[kY*tile.width + kX];
			tileHistPtr[a] += 1 * kernelValue;//vahovani prispevku podle gaussovy funkce
			tileHistPtr[r] += 1 * kernelValue;
			tileHistPtr[g] += 1 * kernelValue;

			kX++;
		}
		kY++;
		kX = 0;
	}
}


void FrameDescriptorExtractor::tileHistCompute1(const cv::Mat & img, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1) {
	int index;
	int a;

	double coefA = 1.* histBinsCH1 / 256;
	

	
	float *kernelPtr = (float *)kernel.data;
	float kernelValue;

	uchar *imgPtr = (uchar *)img.data;
	float *tileHistPtr = (float *)tileHist.data;

	// indexace gaussovky
	int kX = 0; 
	int kY = 0;
	//indexace pro aktualni dlazdici
	for(int y = tile.y; y < tile.height + tile.y; y++)
	{
		for(int x = tile.x; x < tile.width + tile.x; x++)
		{
			index = y*img.cols + x;
			
			//vypocet binu pro kazdy kanal
			a = cvFloor(imgPtr[index]*coefA);
			

			//prispevek do binu pro kazdy kanal
			kernelValue = kernelPtr[kY*tile.width + kX];
			tileHistPtr[a] += 1 * kernelValue;//vahovani prispevku podle gaussovy funkce
			

			kX++;
		}
		kY++;
		kX = 0;
	}
}


void FrameDescriptorExtractor::get2DGaussian(cv::Mat & normKernel, int width, int height) {
	//The Gaussian standard deviation. If it is non-positive, it is computed from ksize as sigma = 0.3*(ksize/2 - 1) + 0.8
	int sigmaX = -1;//width/5;
	int sigmaY = -1;//height/5;

	cv::Mat gaussX = cv::getGaussianKernel(width,sigmaX,CV_32F);
	cv::Mat gaussY = cv::getGaussianKernel(height,sigmaY,CV_32F);
	cv::Mat kernel = cv::Mat::zeros(height,width,CV_32F);

	float *XPtr = (float *)gaussX.data;
	float *YPtr = (float *)gaussY.data;
	float *kernelPtr = (float *)kernel.data;

	for(int y = 0; y < height; y++)
	{
		for(int x = 0; x < width; x++)
		{
			kernelPtr[y*width + x] = XPtr[x]*YPtr[y];
		}
	}
	
	//normalizace gaussovky na 1.0
	float norm = 1.0f / (XPtr[width/2]*YPtr[height/2]);
	normKernel = cv::Mat::zeros(height,width,CV_32FC1);
	float *normKernelPtr = (float *)normKernel.data;

	for(int i = 0; i < kernel.cols*kernel.rows; i++)
	{
		normKernelPtr[i] = kernelPtr[i]*norm;
	}
}


ColHistFrameDescriptorExtractor::ColHistFrameDescriptorExtractor() : FrameDescriptorExtractor() {
	defaults();
}


FrameDescriptorPtr ColHistFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	cv::Mat frameTransformed;

	switch(mode) {
		case MODE_IOO:
			rgb2IOO(frame, frameTransformed);
			//frameTransformed = frame; // TODO proc to tu bylo?
			break;

		case MODE_RG:
			rgb2rg(frame, frameTransformed);
			break;

		case MODE_RGB:
			frameTransformed = frame;
			break;

		default:
			break;
	}

	cv::Mat descriptor;
	computeDescriptor(frameTransformed, descriptor);

	return new ColHistFrameDescriptor(descriptor);
}


void ColHistFrameDescriptorExtractor::defaults() {
	mode = MODE_IOO;

	xDivision = 4;
	yDivision = 4;

	tileOverlap = 0.5;

	histBinsCH1 = 32;
	histBinsCH2 = 16;
	histBinsCH3 = 16;
}


FrameDescriptorPtr ColHistFrameDescriptorExtractor::createEmptyDescriptor() {
	cv::Mat descriptor = cv::Mat::zeros(1, getDescriptorSize(), CV_8UC1);
	return new ColHistFrameDescriptor(descriptor);
}


int ColHistFrameDescriptorExtractor::getType() {
	return FrameDescriptor::COLHIST;
}


long ColHistFrameDescriptorExtractor::getDescriptorSize() {
	int tileCount = xDivision*yDivision;
	int histSize = histBinsCH1 + histBinsCH2 + histBinsCH3;
	long descrSize = tileCount * histSize;

	return descrSize;
}

void ColHistFrameDescriptorExtractor::computeDescriptor(const cv::Mat & frame, cv::Mat & descriptor) {
	int histSize = histBinsCH1 + histBinsCH2 + histBinsCH3;
	long descrSize = getDescriptorSize();

	descriptor = cv::Mat::zeros(1, descrSize, CV_8UC1);
	uchar *descrPtr = (uchar*)descriptor.data;
	int index = 0;

	//vypocet velikosti dlazdice 
	int tileWidth = cvFloor((frame.cols) / (1 + (xDivision - 1) * (1 - tileOverlap)));
	int tileHight = cvFloor(frame.rows / (1 + (yDivision - 1) * (1 - tileOverlap)));

	cv::Mat kernel; 
	get2DGaussian(kernel, tileWidth, tileHight); //gaussian

	// cyklus pres dlazdice
	// xT = xTile, yT = yTile
	for(int yT = 0; yT < yDivision; yT++)
	{
		for(int xT = 0; xT < xDivision; xT++)
		{
			cv::Mat tileHist = cv::Mat::zeros(1, histSize, CV_32FC1);
			CvRect tile = cvRect((int)(((1-tileOverlap)*xT)*tileWidth), (int)(((1-tileOverlap)*yT)*tileHight), tileWidth, tileHight); //aktualni dlazdice
			
			tileHistCompute3(frame, tileHist, tile, kernel, histBinsCH1, histBinsCH2, histBinsCH3); //vypocet histogramu jedne dlazdice
			
			
			float *tileHistPtr = (float*)tileHist.data;

			//normalizace na 1B
			double norm = 255.0 / (tileWidth*tileHight);
			for(int i = 0; i < tileHist.cols; i++)
			{
				tileHistPtr[i] = (float)cvFloor(tileHistPtr[i]*norm + 0.5);
			}



			//konkatenace histogramu dlazdic do jednoho vektoru
			for(int i = 0; i < tileHist.cols; i++)
			{
				descrPtr[index] = (uchar)tileHistPtr[i];
				index++;
			}


		}
	}
}


ColHistFrameDescriptorExtractor::ColHistFrameDescriptor::ColHistFrameDescriptor(cv::Mat data) : VectorFrameDescriptor(FrameDescriptor::COLHIST, data) {
}


const int GradHistFrameDescriptorExtractor::DEFAULT_X_DIVISON = 4;
const int GradHistFrameDescriptorExtractor::DEFAULT_Y_DIVISON = 4;
const float GradHistFrameDescriptorExtractor::DEFAULT_TILE_OVERLAP = 0.5;
const int GradHistFrameDescriptorExtractor::DEFAULT_HIST_BINS = 64;


GradHistFrameDescriptorExtractor::GradHistFrameDescriptorExtractor(int xDivision, int yDivision, float tileOverlap, int bins) : FrameDescriptorExtractor() {
	setXDivision(xDivision);
	setYDivision(yDivision);
	setTileOverlap(tileOverlap);
	setHistogramBinCount(bins);
}

FrameDescriptorPtr GradHistFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	cv::Mat frameTransformed;
	rgb2gradients(frame, frameTransformed);

	cv::Mat descriptor;
	computeDescriptor(frameTransformed, descriptor);

	return new GradHistFrameDescriptor(descriptor);
}


FrameDescriptorPtr GradHistFrameDescriptorExtractor::createEmptyDescriptor() {
	cv::Mat descriptor = cv::Mat::zeros(1, getDescriptorSize(), CV_8UC1);
	return new GradHistFrameDescriptor(descriptor);
}


void GradHistFrameDescriptorExtractor::defaults() {
	xDivision = DEFAULT_X_DIVISON;
	yDivision = DEFAULT_Y_DIVISON;
	tileOverlap = DEFAULT_TILE_OVERLAP;
	histBinsCH1 = DEFAULT_HIST_BINS;
}


int GradHistFrameDescriptorExtractor::getType() {
	return FrameDescriptor::GRADHIST;
}


long GradHistFrameDescriptorExtractor::getDescriptorSize() {
	int tileCount = xDivision*yDivision;
	long descrSize = tileCount * histBinsCH1;

	return descrSize;
}


void GradHistFrameDescriptorExtractor::computeDescriptor(const cv::Mat & frame, cv::Mat & descriptor) {
	long descrSize = getDescriptorSize();

	descriptor = cv::Mat::zeros(1, descrSize, CV_8UC1);
	uchar *descrPtr = (uchar*)descriptor.data;
	int index = 0;
	

	//vypocet velikosti dlazdice 
	int tileWidth = cvFloor((frame.cols) / (1 + (xDivision - 1) * (1 - tileOverlap)));
	int tileHight = cvFloor(frame.rows / (1 + (yDivision - 1) * (1 - tileOverlap)));

	cv::Mat kernel; 
	get2DGaussian(kernel, tileWidth, tileHight); //gaussian

	
	// cyklus pres dlazdice
	// xT = xTile, yT = yTile
	for(int yT = 0; yT < yDivision; yT++)
	{
		for(int xT = 0; xT < xDivision; xT++)
		{
			cv::Mat tileHist = cv::Mat::zeros(1, histBinsCH1, CV_32FC1);
			CvRect tile = cvRect((int)(((1-tileOverlap)*xT)*tileWidth), (int)(((1-tileOverlap)*yT)*tileHight), tileWidth, tileHight); //aktualni dlazdice
			
			tileHistCompute1(frame, tileHist, tile, kernel, histBinsCH1); //vypocet histogramu jedne dlazdice
			
			
			float *tileHistPtr = (float*)tileHist.data;

			//normalizace na 1B
			double norm = 255.0 / (tileWidth*tileHight);
			for(int i = 0; i < tileHist.cols; i++)
			{
				tileHistPtr[i] = (float)cvFloor(tileHistPtr[i]*norm + 0.5);
			}



			//konkatenace histogramu dlazdic do jednoho vektoru
			for(int i = 0; i < tileHist.cols; i++)
			{
				descrPtr[index] = (uchar)tileHistPtr[i];
				index++;
			}


		}
	}
}


GradHistFrameDescriptorExtractor::GradHistFrameDescriptor::GradHistFrameDescriptor(cv::Mat data) : VectorFrameDescriptor(FrameDescriptor::GRADHIST, data) {
}


void GradHistFrameDescriptorExtractor::setTileOverlap(float overlap) {
	if(overlap < 0 || overlap > 1) {
		CV_Error(CV_StsBadArg, "Invalid tile overlap factor value.");
	}
	this->tileOverlap = overlap;
}


void GradHistFrameDescriptorExtractor::setXDivision(int xDivision) {
	if(xDivision < 0) {
		CV_Error(CV_StsBadArg, "Invalid xDivision value.");
	}
	this->xDivision = xDivision;
}


void GradHistFrameDescriptorExtractor::setYDivision(int yDivision) {
	if(yDivision < 0) {
		CV_Error(CV_StsBadArg, "Invalid yDivision value.");
	}
	this->yDivision = yDivision;
}


void GradHistFrameDescriptorExtractor::setHistogramBinCount(int bins) {
	if(bins < 0) {
		CV_Error(CV_StsBadArg, "Invalid histogram bins count.");
	}
	this->histBinsCH1 = bins;
}


GradRGHistFrameDescriptorExtractor::GradRGHistFrameDescriptorExtractor() : FrameDescriptorExtractor() {
	defaults();
}


FrameDescriptorPtr GradRGHistFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	cv::Mat frameGradients;
	rgb2gradients(frame, frameGradients);

	cv::Mat frameRG;
	rgb2rg(frame, frameRG);

	cv::Mat descriptor;
	computeDescriptor(frameRG, frameGradients, descriptor);

	return new GradRGHistFrameDescriptor(descriptor);
}


void GradRGHistFrameDescriptorExtractor::defaults() {
	xDivision = 4;
	yDivision = 4;

	tileOverlap = 0.5;

	histBinsCH1 = 32;
	histBinsCH2 = 16;
	histBinsCH3 = 16;
}


int GradRGHistFrameDescriptorExtractor::getType() {
	return FrameDescriptor::GRADRGHIST;
}


FrameDescriptorPtr GradRGHistFrameDescriptorExtractor::createEmptyDescriptor() {
	cv::Mat descriptor = cv::Mat::zeros(1, getDescriptorSize(), CV_8UC1);
	return new GradRGHistFrameDescriptor(descriptor);
}


long GradRGHistFrameDescriptorExtractor::getDescriptorSize() {
	int tileCount = xDivision*yDivision;
	int histSize = histBinsCH1 + histBinsCH2 + histBinsCH3;
	long descrSize = tileCount * histSize;

	return descrSize;
}


void GradRGHistFrameDescriptorExtractor::computeDescriptor(const cv::Mat & frameRG, const cv::Mat & frameGradients, cv::Mat & descriptor) {
	int histSize = histBinsCH1 + histBinsCH2 + histBinsCH3;
	long descrSize = getDescriptorSize();

	descriptor = cv::Mat::zeros(1, descrSize, CV_8UC1);
	uchar *descrPtr = (uchar*)descriptor.data;
	int index = 0;	

	//vypocet velikosti dlazdice 
	int tileWidth = cvFloor((frameGradients.cols) / (1 + (xDivision - 1) * (1 - tileOverlap)));
	int tileHight = cvFloor(frameGradients.rows / (1 + (yDivision - 1) * (1 - tileOverlap)));

	cv::Mat kernel; 
	get2DGaussian(kernel, tileWidth, tileHight); //gaussian
	
	// cyklus pres dlazdice
	// xT = xTile, yT = yTile
	for(int yT = 0; yT < yDivision; yT++)
	{
		for(int xT = 0; xT < xDivision; xT++)
		{
			cv::Mat tileHist = cv::Mat::zeros(1, histSize, CV_32FC1);
			CvRect tile = cvRect((int)(((1-tileOverlap)*xT)*tileWidth), (int)(((1-tileOverlap)*yT)*tileHight), tileWidth, tileHight); //aktualni dlazdice
			
			tileHistCompute3I2(frameGradients, frameRG, tileHist, tile, kernel, histBinsCH1, histBinsCH2, histBinsCH3); //vypocet histogramu jedne dlazdice
			
			
			float *tileHistPtr = (float*)tileHist.data;

			//normalizace na 1B
			double norm = 255.0 / (tileWidth*tileHight);
			for(int i = 0; i < tileHist.cols; i++)
			{
				tileHistPtr[i] = (float)cvFloor(tileHistPtr[i]*norm + 0.5);
			}



			//konkatenace histogramu dlazdic do jednoho vektoru
			for(int i = 0; i < tileHist.cols; i++)
			{
				descrPtr[index] = (uchar)tileHistPtr[i];
				index++;
			}


		}
	}
}


GradRGHistFrameDescriptorExtractor::GradRGHistFrameDescriptor::GradRGHistFrameDescriptor(cv::Mat data) : VectorFrameDescriptor(FrameDescriptor::GRADRGHIST, data) {
}


const int QuickHistogramFrameDescriptorExtractor::DEFAULT_X_DIVISON = 4;
const int QuickHistogramFrameDescriptorExtractor::DEFAULT_Y_DIVISON = 3;
const float QuickHistogramFrameDescriptorExtractor::DEFAULT_TILE_OVERLAP = 0.3f;
const int QuickHistogramFrameDescriptorExtractor::DEFAULT_HIST_BINS = 16;
const int QuickHistogramFrameDescriptorExtractor::DEFAULT_DOWNSAMPLING = 2;


QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptorExtractor(int xDivision, int yDivision, float tileOverlap, int bins, int downsamplingFactor) : FrameDescriptorExtractor() {
	setXDivision(xDivision);
	setYDivision(yDivision);
	setTileOverlap(tileOverlap);
	setHistogramBinCount(bins);
	setDownsampling(downsamplingFactor);
}


FrameDescriptorPtr QuickHistogramFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	cv::Mat frameGray;

	if(frame.channels() == 3) {
		cvtColor(frame, frameGray, CV_BGR2GRAY);
	}
	else if(frame.channels() == 1) {
		frameGray = frame;
	}
	else {
		CV_Error(CV_StsBadArg, "Invalid number of image channels (1 or 3 allowed).");
	}

	cv::resize(frameGray, frameGray, cv::Size(), 1.0/downsamplingFactor, 1.0/downsamplingFactor);

	cv::Mat concatHists;
	computeDescriptor(frameGray, concatHists);

	return new QuickHistogramFrameDescriptor(concatHists);
}


int QuickHistogramFrameDescriptorExtractor::getDescriptorSize() {
	int tileCount = xDivision*yDivision;
	long descrSize = tileCount*histBins;

	return descrSize;
}


void QuickHistogramFrameDescriptorExtractor::computeDescriptor(const cv::Mat & img, cv::Mat & desc) {
	int tileCount = xDivision*yDivision;
	int descrSize = getDescriptorSize();

	int tileWidth = cvFloor((img.cols) / (1 + (xDivision - 1) * (1 - tileOverlap)));
	int tileHight = cvFloor(img.rows / (1 + (yDivision - 1) * (1 - tileOverlap)));

	desc = cv::Mat::zeros(1, descrSize, CV_32FC1);
	float *concatPtr = (float*)desc.data;
	int index = 0;

	float coef = 1.0f/tileWidth/tileHight/tileCount;

	for(int yT = 0; yT < yDivision; yT++) {
		for(int xT = 0; xT < xDivision; xT++) {
			cv::Mat tileHist = cv::Mat::zeros(1, histBins, CV_32FC1);
			CvRect tile = cvRect((int)(((1-tileOverlap)*xT)*tileWidth), (int)(((1-tileOverlap)*yT)*tileHight), tileWidth, tileHight); //aktualni dlazdice

			computeHistogram(img, tile, tileHist, histBins); //vypocet histogramu jedne dlazdice
						
			float *tileHistPtr = (float*)tileHist.data;

			for(int i = 0; i < tileHist.cols; i++) {
				concatPtr[index] = tileHistPtr[i]*coef;
				index++;
			}
		}
	}
}


void QuickHistogramFrameDescriptorExtractor::computeHistogram(const cv::Mat & img, const cv::Rect & tile, cv::Mat & hist, int bins) {
	double quant = (double)bins/256;
	
	uchar *imgPtr = (uchar *)img.data;
	float *tileHistPtr = (float *)hist.data;

	int index;
	for(int y = tile.y; y < tile.height + tile.y; y++) {
		for(int x = tile.x; x < tile.width + tile.x; x++) {
			index = y*img.cols + x;
			int a = cvFloor(imgPtr[index]*quant);
			tileHistPtr[a]++;
		}
	}
}


void QuickHistogramFrameDescriptorExtractor::defaults() {
	xDivision = DEFAULT_X_DIVISON;
	yDivision = DEFAULT_Y_DIVISON;
	tileOverlap = DEFAULT_TILE_OVERLAP;
	histBins = DEFAULT_HIST_BINS;
}


int QuickHistogramFrameDescriptorExtractor::getType() {
	return FrameDescriptor::QUICKHIST;
}


FrameDescriptorPtr QuickHistogramFrameDescriptorExtractor::createEmptyDescriptor() {
	cv::Mat descriptor = cv::Mat::zeros(1, getDescriptorSize(), CV_32FC1);
	return new QuickHistogramFrameDescriptor(descriptor);
}


QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptor::QuickHistogramFrameDescriptor(const cv::Mat & data) : FrameDescriptor(FrameDescriptor::QUICKHIST) {
	this->data = data;
}


double QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptor::compare(const FrameDescriptor & second) const {
	CV_Assert(second.getType() == FrameDescriptor::QUICKHIST);
	QuickHistogramFrameDescriptor *secondPtr = (QuickHistogramFrameDescriptor *)&second;

	return 0.5*compareHist(data, secondPtr->data, CV_COMP_CHISQR);
}


void QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptor::read(cv::FileNode & nd) {
	// TODO
}


void QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptor::write(cv::FileStorage & fs) const {
	// TODO
}


void QuickHistogramFrameDescriptorExtractor::setTileOverlap(float overlap) {
	if(overlap < 0 || overlap > 1) {
		CV_Error(CV_StsBadArg, "Invalid tile overlap factor value.");
	}
	this->tileOverlap = overlap;
}


void QuickHistogramFrameDescriptorExtractor::setXDivision(int xDivision) {
	if(xDivision < 0) {
		CV_Error(CV_StsBadArg, "Invalid xDivision value.");
	}
	this->xDivision = xDivision;
}


void QuickHistogramFrameDescriptorExtractor::setYDivision(int yDivision) {
	if(yDivision < 0) {
		CV_Error(CV_StsBadArg, "Invalid yDivision value.");
	}
	this->yDivision = yDivision;
}


void QuickHistogramFrameDescriptorExtractor::setHistogramBinCount(int bins) {
	if(bins < 0) {
		CV_Error(CV_StsBadArg, "Invalid histogram bins count.");
	}
	this->histBins = bins;
}


void QuickHistogramFrameDescriptorExtractor::setDownsampling(int downsamplingFactor) {
	//TODO kontrola
	this->downsamplingFactor = downsamplingFactor;
}


NullFrameDescriptorExtractor::NullFrameDescriptorExtractor() {
}


FrameDescriptorPtr NullFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	return createEmptyDescriptor();
}


int NullFrameDescriptorExtractor::getType() {
	return FrameDescriptor::_NULL;
}


FrameDescriptorPtr NullFrameDescriptorExtractor::createEmptyDescriptor() {
	return new NullFrameDescriptor();
}


NullFrameDescriptorExtractor::NullFrameDescriptor::NullFrameDescriptor() : FrameDescriptor(FrameDescriptor::_NULL) {
}


double NullFrameDescriptorExtractor::NullFrameDescriptor::compare(const FrameDescriptor & second) const {
	return DBL_MAX;
}


void NullFrameDescriptorExtractor::NullFrameDescriptor::read(cv::FileNode & nd) {
}


void NullFrameDescriptorExtractor::NullFrameDescriptor::write(cv::FileStorage & fs) const {
}


SurfFrameDescriptorExtractor::SurfFrameDescriptorExtractor() {
}


FrameDescriptorPtr SurfFrameDescriptorExtractor::extract(const cv::Mat & frame) {
	std::vector<cv::KeyPoint> keypoints;
	cv::SurfFeatureDetector featureDetector(4000);
	featureDetector.detect(frame, keypoints);
	
	//std::cout << "kps=" << keypoints.size() << "\n"; exit(0);
	
	cv::SurfDescriptorExtractor descriptorExtractor;
	cv::Mat descriptors;
	descriptorExtractor.compute(frame, keypoints, descriptors);

	return new SurfFrameDescriptor(descriptors);
}


int SurfFrameDescriptorExtractor::getType() {
	return FrameDescriptor::LOCAL_SURF;
}


FrameDescriptorPtr SurfFrameDescriptorExtractor::createEmptyDescriptor() {
	return new SurfFrameDescriptor();
}


SurfFrameDescriptorExtractor::SurfFrameDescriptor::SurfFrameDescriptor() : FrameDescriptor(FrameDescriptor::LOCAL_SURF) {
	this->descriptors = cv::Mat();
}


SurfFrameDescriptorExtractor::SurfFrameDescriptor::SurfFrameDescriptor(const cv::Mat & descriptors) : FrameDescriptor(FrameDescriptor::LOCAL_SURF) {
	this->descriptors = descriptors;
}


double SurfFrameDescriptorExtractor::SurfFrameDescriptor::compare(const FrameDescriptor & second) const {
	CV_Assert(this->getType() == second.getType());

	SurfFrameDescriptor *secondPtr = (SurfFrameDescriptor *)(&second);

	cv::BFMatcher descriptorMatcher(cv::NORM_L2);
    std::vector<cv::DMatch> matches;

	descriptorMatcher.match(secondPtr->descriptors, this->descriptors, matches);

	double threshold = 0.6; // TODO presunout
	int inliersCount = 0;
	for(std::vector<cv::DMatch>::iterator it = matches.begin(); it != matches.end(); it++) {
		inliersCount += (it->distance < threshold);
	}
	
	return 1 - ((float)inliersCount)/this->descriptors.rows;
}


void SurfFrameDescriptorExtractor::SurfFrameDescriptor::read(cv::FileNode & nd) {
	CV_Assert(!nd.empty());
	
	CV_Assert(!nd["desc"].empty());
	nd["desc"] >> descriptors;
}


void SurfFrameDescriptorExtractor::SurfFrameDescriptor::write(cv::FileStorage & fs) const {
	fs << "desc" << descriptors;
}


}
