/***
 * Lfd Walker
 * Author: Lukas Durfina, idurfina@fit.vutbr.cz
 ***/

package main;

import grammar.LfdBaseListener;
import grammar.LfdParser;

import java.util.HashMap;
import java.util.Stack;

import org.antlr.v4.runtime.misc.NotNull;


public class LfdWalker extends LfdBaseListener {
	
	/**
	 * Map with all functions in processed file
	 */
	private HashMap<String, LfdFunction> _functions = new HashMap<String, LfdFunction>();
	
	/**
	 * Get all found functions
	 */
	public HashMap<String, LfdFunction> getFunctions() {
		return _functions;
	}
	
	/**
	 * Currently processed function
	 */
	private LfdFunction _currentFunction;
	
	
	/**
	 * Currently called function
	 */
	private LfdFunctionCall _currentFunctionCall;
	
	/**
	 * Stack with loops
	 * Stack is used because, there could be more nested loops
	 */
	private Stack<Loop> _loopStack = new Stack<Loop>();
	
	
	/**
	 * {@inheritDoc}
	 * <p/>
	 * We found a declaration of function
	 */
	@Override
	public void enterFunc(LfdParser.FuncContext ctx) {
		String functionName = ctx.getChild(0).getText();
		
		_currentFunction = getFunction(functionName);
		_currentFunction.defined = true;
		
		//LfdComparator.debug("Enter function: " + _currentFunction.name);
	}
	
	
	/**
	 * {@inheritDoc}
	 * <p/>
	 * The default implementation does nothing.
	 */
	@Override public void enterArgument(@NotNull LfdParser.ArgumentContext ctx) { 
		String argValue = ctx.getChild(0).getText();
		_currentFunctionCall.addArgument(argValue);
		//LfdComparator.debug("Enter argument: " + argValue);
	}
	

	/**
	 * {@inheritDoc}
	 * <p/>
	 * We found a call of function
	 */
	@Override
	public void enterFunc_call(LfdParser.Func_callContext ctx) {
		String functionName = ctx.getChild(0).getText();
		
		// add call of function to current function
		LfdFunction calledFunc = getFunction(functionName);
		_currentFunctionCall = new LfdFunctionCall(calledFunc);
		if (_loopStack.empty()) {
			_currentFunction.addStatement(_currentFunctionCall);
		}
		else {
			_loopStack.peek().addStatement(_currentFunctionCall);
		}
		
		//LfdComparator.debug("Call function: " + calledFunc.name);
	}
	
	/**
	 * {@inheritDoc}
	 * <p/>
	 * We go inside of loop
	 */
	@Override
	public void enterLoop_stmt(LfdParser.Loop_stmtContext ctx) { 
		_loopStack.push(new Loop());
	}
	
	/**
	 * {@inheritDoc}
	 * <p/>
	 * The end of loop.
	 * Add currently finished loop into parent
	 */
	@Override
	public void exitLoop_stmt(@NotNull LfdParser.Loop_stmtContext ctx) { 
		Loop lastLoop = _loopStack.pop();
		if (_loopStack.empty()) {
			_currentFunction.addStatement(lastLoop);
		}
		else {
			_loopStack.peek().addStatement(lastLoop);
		}
	}
	
	/**
	 * Get a function. If it does not exist, it is created.
	 * @param name
	 */
	private LfdFunction getFunction(String name) {
		if (!_functions.containsKey(name)) {
			LfdFunction func = new LfdFunction(name);
			_functions.put(name, func);
			return func;
		}
		else {
			return _functions.get(name);
		}
	}
	
}


