
/***
 * Lfd comparator
 * Author: Lukas Durfina, idurfina@fit.vutbr.cz
 ***/

package comparators;


import java.util.*;
import main.*;

public class LfdFunctionComparator extends StatementParentComparator {

	public LfdFunctionComparator() {
		_similarFunctions = new SimilarLinkedFunctions();
	}
	
	/**
	 * Class for recognizing similar linked functions
	 */
	private SimilarLinkedFunctions _similarFunctions;
	
	/**
	 * Stack with called functions for A.
	 * It is used for checking recursion.
	 */
	private static Stack<String> _calledFuncStackA = new Stack<String>();
	
	/**
	 * Stack with called functions for B
	 */
	private static Stack<String> _calledFuncStackB = new Stack<String>();
	
	
	@Override public double compare(IComparable a, IComparable b) {
		
		if (a instanceof LfdFunction && b instanceof LfdFunction) {
			
			LfdFunction fA = (LfdFunction) a;
			LfdFunction fB = (LfdFunction) b;
			
			LfdComparator.debug("Comparing: " + fA.name + " - " + fB.name);
			
			HashMap<String, LfdFunction> functions1 = LfdComparator.getFunctions1();
			HashMap<String, LfdFunction> functions2 = LfdComparator.getFunctions2();
			
			// both are linked -> nothing in defined functions
			if (!functions1.get(fA.name).defined && !functions2.get(fB.name).defined) {
				LfdComparator.debug("Both linked: " + fA.name + " - " + fB.name);
				
				if (fA.name.equals(fB.name)) {
					return 100.0;
				}
				else if (_similarFunctions.areSimilarFunctions(fA, fB)) {
					LfdComparator.debug("\t similar functions");
					return 75.0;
				}
				return 0;
			}
			
			if (checkForRecursion(fA, fB)) {
				return recursionSimilarity(fA, fB);
			}
			
			// store current functions
			_calledFuncStackA.push(fA.name);
			_calledFuncStackB.push(fB.name);
			
			double res = super.compare(a, b);
			
			// current functions were processed so drop them
			_calledFuncStackA.pop();
			_calledFuncStackB.pop();
			
			return res;
		}
		
		LfdComparator.error("Some item to compare is not a function");
		
		return 0;
	}
	
	/**
	 * Check if some of function is going to be recursive
	 * @param fA
	 * @param fB
	 * @return true if recursion was found out
	 */
	private boolean checkForRecursion(LfdFunction fA, LfdFunction fB) {
		if (_calledFuncStackA.contains(fA.name))
		{
			return true;
		}
		
		if (_calledFuncStackB.contains(fB.name))
		{
			return true;
		}
		
		return false;
	}
	
	/**
	 * Find out the similarity for the recursion.
	 * We try to find out if there is the same recursion for both functions
	 * @param fA
	 * @param fB
	 * @return similarity in %
	 */
	private double recursionSimilarity(LfdFunction fA, LfdFunction fB) {
		if (_calledFuncStackA.search(fA.name) == _calledFuncStackB.search(fB.name)) {
			return 100;
		}
		return 0;
	}
	
}
