/*
								+-----------------------------------+
								|                                   |
								|     ***  VBO / PBO class  ***     |
								|                                   |
								|   Copyright   -tHE SWINe- 2009   |
								|                                   |
								|          BufferObjects.h          |
								|                                   |
								+-----------------------------------+
*/

#pragma once
#ifndef __BUFFER_OBJECTS_INCLUDED
#define __BUFFER_OBJECTS_INCLUDED

/**
 *	@file gles2/BufferObjects.h
 *	@author -tHE SWINe-
 *	@date 2009
 *	@brief wrapper for GL_ARB_vertex_buffer_object / GL_ARB_pixel_buffer_object extensions
 *
 *	@date 2012-02-13
 *
 *	Modified CGLArraySetup and CGLElementArraySetup to support texcoord-less streams.
 *	This is done simply by passing 0 as texcoord dimension.
 *
 *	Added the __GL4VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER and
 *	the __GL4VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING macros (debugging / bugfix).
 *
 *	Added new versions of CGLArraySetup::Draw() and CGLElementArraySetup::Draw(),
 *	allowing explicit specification of the parameters to the OpenGL drawing function
 *	(glDrawArrays() and glDrawElements(), respectively).
 *
 *	@date 2012-03-26
 *
 *	Fixed a minor bug in calculation of vertex size in CGLArraySetup::CGLArraySetup()
 *	and in CGLElementArraySetup::CGLElementArraySetup(). The old code counted on texcoord
 *	data type being the same as the position data type.
 *
 *	@date 2012-06-19
 *
 *	Added \#pragma once.
 *
 *	@date 2013-05-04
 *
 *	Added support for __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT.
 *
 */

#ifdef __HAVE_GLES
#include "GLES20.h"
#else // __HAVE_GLES
#include "GLES20Emu.h"
#endif // __HAVE_GLES

/**
 *	@def __GL_BUFFER_OBJECT_HOLD_SIZE
 *	@brief if defined, CGLBufferObject will contain buffer size as a member variable,
 *		CGLBufferObject::n_Size() is compiled
 */
//#define __GL_BUFFER_OBJECT_HOLD_SIZE

/**
 *	@def __GL_BUFFER_OBJECT_HOLD_USAGE
 *	@brief if defined, CGLBufferObject will contain buffer usage as a member variable,
 *		CGLBufferObject::n_Usage() is compiled
 */
//#define __GL_BUFFER_OBJECT_HOLD_USAGE

/**
 *	@def __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO
 *	@brief if defined, CGLBufferObject will contain mapping status and access
 *		flags as member variables, CGLBufferObject::b_Mapped()
 *		and CGLBufferObject::n_AccessFlag() are compiled
 */
//#define __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO

/**
 *	@def __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
 *	@brief if defined, the default VAO 0 is bound at the end of
 *		CGLArraySetup::Draw() and CGLElementArraySetup::Draw()
 *	@note This was added to help debugging VAO issues.
 */
//#define __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING

/**
 *	@def __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER
 *	@brief if defined, index buffer is bound in CGLElementArraySetup::Draw()
 *		before calling glDrawElements()
 *	@note This fixes a bug occuring in some of the mobile chipsets where
 *		GL_ELEMENT_ARRAY_BINDING is not (properly) stored in VAO state.
 *		One example of such chipset may be even NVIDIA Quadro NVS 4200M.
 */
#define __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER

/**
 *	@def __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
 *	@brief if defined, CGLArraySetup and CGLElementArraySetup avoids using VAOs
 *	@note This is intended to increase compatibility on devices with limited
 *		features, but can get one in trouble (e.g. enabled vertex attributes seen
 *		accross different array setups, unlike with VAOs).
 */
//#define __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

/**
 *	@brief class, wrapping GL_ARB_vertex_buffer_object / GL_ARB_pixel_buffer_object extensions
 *
 *	Object of this class holds a single vertex buffer object, it is created in OpenGL when
 *		one of Bind(), BindAs() or BufferData() is called first time. OpenGL object is destroyed
 *		once destructor is called.
 */
class CGLBufferObject {
protected:
	GLuint m_n_buffer_id; /**< @brief buffer object OpenGL id */

#ifdef __GL_BUFFER_OBJECT_HOLD_SIZE
	GLuint m_n_size; /**< @brief buffer size, in bytes */
#endif // __GL_BUFFER_OBJECT_HOLD_SIZE

#ifdef __GL_BUFFER_OBJECT_HOLD_USAGE
	int m_n_usage; /**< @brief buffer usage */
#endif // __GL_BUFFER_OBJECT_HOLD_USAGE

#ifdef __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO
	int m_n_access; /**< @brief mapped buffer access mode */
	bool m_b_mapped; /**< @brief buffer mapping status */
#endif // __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO

public:
	/**
	 *	@brief default constructor; creates name for the buffer object
	 */
	CGLBufferObject();

	/**
	 *	@brief destructor; deletes the buffer object
	 */
	~CGLBufferObject();

	/**
	 *	@brief gets buffer status
	 *	@return Returns true if the buffer was created successulfy, otherwise returns false.
	 */
	inline bool b_Status() const
	{
		return m_n_buffer_id != 0;
	}

	/**
	 *	@brief binds the buffer
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *
	 *	@note With GL_PIXEL_UNPACK_BUFFER, it is possible to copy data using glReadPixels()
	 *		or glGetTexImage() with <data> parameter set to the value returned
	 *		by call to p_OffsetPointer(0).
	 */
	void Bind(GLenum n_target) const;

	/**
	 *	@brief allocates the buffer
	 *
	 *	Allocates buffer data of size n_size bytes, can be initialized to p_data in case
	 *		non-NULL pointer is supplied. In case buffer was allocated already, it's data
	 *		are freed as soon as there are no OpenGL commands, sourcing data from it
	 *		(it shouldn't block program execution, depends on graphics card vendor and
	 *		driver version) and new buffer is allocated.
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_size is buffer size in bytes
	 *	@param[in] p_data is pointer to n_size bytes of buffer data, or null pointer
	 *	@param[in] n_usage is optimalization hint only, it is one of:
	 *		GL_STREAM_DRAW, GL_STREAM_READ, GL_STREAM_COPY,
	 *		GL_STATIC_DRAW, GL_STATIC_READ, GL_STATIC_COPY,
	 *		GL_DYNAMIC_DRAW, GL_DYNAMIC_READ or GL_DYNAMIC_COPY
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 *	@note This may generate GL_OUT_OF_MEMORY error.
	 */
	bool BufferData(GLenum n_target, GLuint n_size, const void *p_data = 0, GLenum n_usage = GL_STATIC_DRAW);

	/**
	 *	@brief gets server-side pointer to the buffer
	 *
	 *	Returns pointer to buffer data for use with glVertexAttribPointer(), glDrawElements(), etc.
	 *		This pointer is only associated with this buffer, in case this buffer is currently bound.
	 *		Do not use this pointer to access data, it doesn't point to client address space.
	 *
	 *	@param[in] n_offset is offset to referenced place in buffer, in bytes
	 *
	 *	@return Returns pointer to buffer data.
	 *
	 *	@note Do not compare returned pointer to null, it is perfectly valid server-side pointer value.
	 */
	static inline void *p_OffsetPointer(GLuint n_offset)
	{
		return (void*)((char*)NULL + n_offset);
	}

	/**
	 *	@brief (re)specifies part (or all) of buffer data
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_offset is offset in buffer to start of data being specified
	 *	@param[in] n_size is data size in bytes
	 *	@param[in] p_data is pointer to n_size bytes of buffer data
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 *	@note n_offset must be greater or equal to 0, n_offset + n_size must be less or equal
	 *		to buffer size, otherwise OpenGL error is generated (call glGetError()).
	 */
	void BufferSubData(GLenum n_target, GLuint n_offset, GLuint n_size, const void *p_data);

#if 0
	/**
	 *	@brief reads part (or all) of buffer data
	 *
	 *	Copies n_size bytes of buffer data, starting at byte n_offset to p_data.
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_offset is offset in buffer to start of data being specified
	 *	@param[in] n_size is data size in bytes (n_offset + n_size must be less or equal
	 *		to buffer size, otherwise OpenGL error is generated)
	 *	@param[in] p_data is pointer, n_size bytes of buffer data will be copied to
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 */
	void GetBufferSubData(GLenum n_target, GLuint n_offset, GLuint n_size, void *p_data) const;

	/**
	 *	@brief maps buffer contents to client space
	 *
	 *	Maps buffer into client address space (one buffer only can be mapped at a time).
	 *		Mapped buffer should be un-mapped by calling UnMap(). In case buffer was mapped,
	 *		pointer returned by this function can be retrieved again, by calling
	 *		p_Get_BufferPointer().
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_access is buffer access, it can be one of GL_READ_ONLY, GL_WRITE_ONLY
	 *		or GL_READ_WRITE
	 *
	 *	@return Returns client address space pointer in case of success or NULL on failure.
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 */
	void *p_Map(GLenum n_target, GLenum n_access = GL_READ_WRITE);

	/**
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_offset is offset of range to map (in bytes)
	 *	@param[in] n_length is length of range to map (in bytes)
	 *	@param[in] n_access is buffer access, it can be 0 or combination of GL_MAP_READ_BIT,
	 *		GL_MAP_WRITE_BIT, GL_MAP_INVALIDATE_RANGE_BIT, GL_MAP_INVALIDATE_BUFFER_BIT,
	 *		GL_MAP_FLUSH_EXPLICIT_BIT or GL_MAP_UNSYNCHRONIZED_BIT.
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 *	@note This requires GL_ARB_map_buffer_range (part of OpenGL 3.0 core).
	 */
	void *p_MapRange(GLenum n_target, GLintptr n_offset, GLsizeiptr n_length, GLbitfield n_access);

	/**
	 *	@brief explicitly flushes mapped buffer sub-range
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *	@param[in] n_offset is offset of range to map (in bytes)
	 *	@param[in] n_length is length of range to map (in bytes)
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 *	@note This requires GL_ARB_map_buffer_range (part of OpenGL 3.0 core).
	 */
	void FlushMappedRange(GLenum n_target, GLintptr n_offset, GLsizeiptr n_length);

	/**
	 *	@brief gets client-side pointer to mapped buffer
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *
	 *	@return In case buffer object has been successfuly mapped, return client-space pointer,
	 *		otherwise return NULL.
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 *	@ntoe This involves calling glGetBufferPointerv() and may cause GPU-CPU synchronization.
	 */
	void *p_Get_BufferPointer(GLenum n_target) const;

	/**
	 *	@brief unmaps buffer
	 *
	 *	Ends mapping of buffer data to client address space, previously estabilished
	 *		calling p_Map().
	 *
	 *	@param[in] n_target is buffer object target, valid values include GL_ARRAY_BUFFER,
	 *		GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER, GL_DRAW_INDIRECT_BUFFER,
	 *		GL_ELEMENT_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, GL_PIXEL_UNPACK_BUFFER,
	 *		GL_TEXTURE_BUFFER, GL_TRANSFORM_FEEDBACK_BUFFER and GL_UNIFORM_BUFFER.
	 *
	 *	@return Returns true on success or false on failure (buffer contents were
	 *		lost - that may be caused by screen-mode switch while the buffer was
	 *		bound, or in case the buffer was not mapped).
	 *
	 *	@note The buffer must be bound to the specified target before calling this.
	 */
	bool Unmap(GLenum n_target);
#endif // 0

#ifdef __GL_BUFFER_OBJECT_HOLD_SIZE
	/**
	 *	@brief gets size of buffer
	 *
	 *	@return Returns size of buffer, in bytes.
	 *
	 *	@note This is only available if __GL_BUFFER_OBJECT_HOLD_SIZE is defined.
	 */
	inline GLuint n_Size() const
	{
		return m_n_size;
	}
#endif // __GL_BUFFER_OBJECT_HOLD_SIZE

#ifdef __GL_BUFFER_OBJECT_HOLD_USAGE
	/**
	 *	@brief gets buffer ussage
	 *
	 *	@return Returns buffer ussage (one of:
	 *		GL_STREAM_DRAW, GL_STREAM_READ, GL_STREAM_COPY,
	 *		GL_STATIC_DRAW, GL_STATIC_READ, GL_STATIC_COPY,
	 *		GL_DYNAMIC_DRAW, GL_DYNAMIC_READ or GL_DYNAMIC_COPY).
	 *
	 *	@note This is only available if __GL_BUFFER_OBJECT_HOLD_USAGE is defined.
	 */
	inline GLenum n_Usage() const
	{
		return m_n_usage;
	}
#endif // __GL_BUFFER_OBJECT_HOLD_USAGE

#ifdef __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO
	/**
	 *	@brief determines wheter buffer is mapped
	 *
	 *	@return Returns true in case buffer contents were mapped into client-space memory.
	 *
	 *	@note It's possible to obtain pointer to mapped buffer data by call to p_Get_BufferPointer().
	 *	@note This is only available if __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO is defined.
	 */
	inline bool b_Mapped() const
	{
		return m_b_mapped;
	}

	/**
	 *	@brief determines mapped buffer access flags
	 *
	 *	@return Returns access flag to buffer in case it has been mapped (one of GL_READ_ONLY,
	 *		GL_WRITE_ONLY, GL_READ_WRITE), or in case buffer is not mapped, returns 0.
	 *
	 *	@note This is only available if __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO is defined.
	 */
	inline GLenum n_AccessFlag() const
	{
		return (m_b_mapped)? m_n_access : 0;
	}
#endif // __GL_BUFFER_OBJECT_HOLD_MAPPING_INFO

protected:
	inline bool b_IsBound(GLenum n_target) const;
	inline CGLBufferObject(const CGLBufferObject &UNUSED(r_buff_obj)) {} /**< @brief copy-constructor (cannot be used) */
	inline CGLBufferObject &operator =(const CGLBufferObject &UNUSED(r_buff_obj)) { return *this; } /**< @brief copy-operator (cannot be used) */
};

/**
 *	@brief OpenGL buffer object wrapper with the default target specifier
 */
template <const GLenum n_default_target>
class CGLBufferObject_Target : public CGLBufferObject {
public:
	enum {
		default_Target = n_default_target /**< @brief default buffer binding target */
	};

public:
	/**
	 *	@copydoc CGLBufferObject::Bind()
	 */
	inline void Bind() const
	{
		CGLBufferObject::Bind(default_Target);
	}

	/**
	 *	@copydoc CGLBufferObject::BufferData()
	 */
	bool BufferData(GLuint n_size, const void *p_data = 0, GLenum n_usage = GL_STATIC_DRAW)
	{
		return CGLBufferObject::BufferData(default_Target, n_size, p_data, n_usage);
	}

	/**
	 *	@copydoc CGLBufferObject::BufferSubData()
	 */
	void BufferSubData(GLuint n_offset, GLuint n_size, const void *p_data)
	{
		CGLBufferObject::BufferSubData(default_Target, n_offset, n_size, p_data);
	}

#if 0
	/**
	 *	@copydoc CGLBufferObject::GetBufferSubData()
	 */
	void GetBufferSubData(GLuint n_offset, GLuint n_size, void *p_data) const
	{
		CGLBufferObject::GetBufferSubData(default_Target, n_offset, n_size, p_data);
	}

	/**
	 *	@copydoc CGLBufferObject::p_Map()
	 */
	void *p_Map(GLenum n_access = GL_READ_WRITE)
	{
		return CGLBufferObject::p_Map(default_Target, n_access);
	}

	/**
	 *	@copydoc CGLBufferObject::p_MapRange()
	 */
	void *p_MapRange(GLenum n_target, GLintptr n_offset, GLsizeiptr n_length, GLbitfield n_access)
	{
		return CGLBufferObject::p_MapRange(default_Target, n_offset, n_length, n_access);
	}

	/**
	 *	@copydoc CGLBufferObject::FlushMappedRange()
	 */
	void FlushMappedRange(GLenum n_target, GLintptr n_offset, GLsizeiptr n_length)
	{
		CGLBufferObject::FlushMappedRange(default_Target, n_offset, n_length);
	}

	/**
	 *	@copydoc CGLBufferObject::p_Get_BufferPointer()
	 */
	void *p_Get_BufferPointer() const
	{
		return CGLBufferObject::p_Get_BufferPointer(default_Target);
	}

	/**
	 *	@copydoc CGLBufferObject::Unmap()
	 */
	bool Unmap()
	{
		return CGLBufferObject::Unmap(default_Target);
	}
#endif // 0
};

/**
 *	@brief array buffer object
 */
typedef CGLBufferObject_Target<GL_ARRAY_BUFFER> CGLArrayBufferObject;

/**
 *	@brief element array buffer object (index buffer object)
 */
typedef CGLBufferObject_Target<GL_ELEMENT_ARRAY_BUFFER> CGLElementArrayBufferObject;

#if 0

/**
 *	@brief draw indirect buffer object
 */
typedef CGLBufferObject_Target<GL_DRAW_INDIRECT_BUFFER> CGLDrawIndirectBufferObject;

/**
 *	@brief texture buffer object
 */
typedef CGLBufferObject_Target<GL_TEXTURE_BUFFER> CGLTextureBufferObject;

/**
 *	@brief transform feedback buffer object
 */
typedef CGLBufferObject_Target<GL_TRANSFORM_FEEDBACK_BUFFER> CGLTransformFeedbackBufferObject;

/**
 *	@brief uniform buffer object
 */
typedef CGLBufferObject_Target<GL_UNIFORM_BUFFER> CGLUniformBufferObject;

/**
 *	@brief pixel (pack) buffer object
 *	@note When a buffer object is bound to the GL_PIXEL_PACK_BUFFER target, commands
 *		such as glReadPixels pack (write) their data into a buffer object.
 *		When a buffer object is bound to the GL_PIXEL_UNPACK_BUFFER target,
 *		commands such as glDrawPixels and glTexImage2D unpack (read) their
 *		data from a buffer object.
 */
typedef CGLBufferObject_Target<GL_PIXEL_PACK_BUFFER> CGLPixelBufferObject;

#endif // 0

/**
 *	@brief a simple class, encapsulating vertex array object
 */
class CGLVertexArrayObject {
protected:
	GLuint m_n_name;

public:
	/**
	 *	@brief default constructor; creates the vertex array object
	 */
	CGLVertexArrayObject();

	/**
	 *	@brief destructor; deletes the vertex array object, if created successfully
	 */
	~CGLVertexArrayObject();

	/**
	 *	@brief gets vertex array status
	 *	@return Returns true if the vertex array object was created successfully,
	 *		otherwise returns false.
	 */
	bool b_Status() const;

	/**
	 *	@brief binds the vertex array object
	 */
	inline void Bind() const
	{
		glBindVertexArray(m_n_name);
	}

	/**
	 *	@brief binds the default vertex array object,
	 *		effectively releasing currently bound array object
	 */
	static inline void Release()
	{
		glBindVertexArray(0);
	}

protected:
	inline CGLVertexArrayObject(const CGLVertexArrayObject &UNUSED(r_array)) {} /**< @brief copy-constructor (cannot be used) */
	inline CGLVertexArrayObject &operator =(const CGLVertexArrayObject &UNUSED(r_array)) { return *this; } /**< @brief copy-operator (cannot be used) */
};

/**
 *	@brief a simple class, encapsulating allocation of vertex
 *		array containing some simple statically allocated geometry
 *	@note This requires the GL_ARB_vertex_array_object extension (part of OpenGL 3.0 core).
 */
class CGLArraySetup {
protected:
	CGLArrayBufferObject m_vertex_buffer; /**< @brief vertex buffer */

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	CGLVertexArrayObject m_vertex_array_object; /**< @brief vertex array object */
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	int m_n_texcoord_component_num; /**< @brief  */
	int m_n_texcoord_data_type; /**< @brief  */
	int m_n_vertex_stride_bytes; /**< @brief  */
	int m_n_texcoord_offset_bytes; /**< @brief  */
	int m_n_position_component_num; /**< @brief  */
	int m_n_position_data_type; /**< @brief  */
	int m_n_position_offset_bytes; /**< @brief  */
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

	GLenum m_n_draw_mode; /**< @brief draw mode (name of primitives to render) */
	size_t m_n_vertex_num; /**< @brief number of indices to render */

public:
	/**
	 *	@brief constructor; fills the arrays with data
	 *
	 *	Use this constructor to specify indexed geometry with positions
	 *	in vertex attribute 0 and texcoords in vertex attribute 1.
	 *	To specify different layouts or different types of geometry entirely,
	 *	use CBasicStaticRenderable().
	 *
	 *	@param[in] p_vertex_list is buffer containing the vertex data
	 *	@param[in] n_vertices_size_bytes is size of all the vertex data, in bytes
	 *	@param[in] n_vertex_stride_bytes is offset between two vertices, in bytes
	 *	@param[in] n_position_data_type is OpenGL data type used to represent positions
	 *	@param[in] n_position_component_num is number of position components
	 *	@param[in] n_position_offset_bytes is offset to the first position, in bytes
	 *	@param[in] n_texcoord_data_type is OpenGL data type used to represent texcoords
	 *	@param[in] n_texcoord_component_num is number of texcoord components
	 *	@param[in] n_texcoord_offset_bytes is offset to the first texcoord, in bytes
	 *	@param[in] p_index_list is buffer, containing index data
	 *	@param[in] n_index_num is number of indices to render
	 *	@param[in] n_index_data_type is data type of vertex indices
	 *	@param[in] n_draw_mode is draw mode (name of primitives to render)
	 */
	CGLArraySetup(const void *p_vertex_list, int n_vertices_size_bytes,
		int n_vertex_stride_bytes, GLenum n_position_data_type, int n_position_component_num,
		int n_position_offset_bytes, GLenum n_texcoord_data_type, int n_texcoord_component_num,
		int n_texcoord_offset_bytes, GLenum n_draw_mode);

	/**
	 *	@brief gets status of this renderable
	 *	@return Returns true if the array buffer and VAO allocated successfully,
	 *		otherwise returns false.
	 */
	bool b_Status() const;

	/**
	 *	@brief binds the VAO and calls glDrawElements() to display the geometry
	 */
	inline void Draw() const
	{
#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_array_object.Bind();
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_buffer.Bind();
		if(m_n_texcoord_component_num) {
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_texcoord_component_num,
				m_n_texcoord_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_texcoord_offset_bytes));
			glEnableVertexAttribArray(1);
			glVertexAttribPointer(1, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		} else {
			glDisableVertexAttribArray(1);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		}
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

		glDrawArrays(m_n_draw_mode, 0, m_n_vertex_num);

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
#ifdef __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
		m_vertex_array_object.Release();
#endif // __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	}

	/**
	 *	@brief binds the VAO and calls glDrawElements() to display the geometry,
	 *		adds override option for some of the parameters
	 *
	 *	@param[in] n_draw_mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
	 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
	 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
	 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
	 *	@param[in] n_first_vertex Specifies the starting index in the enabled arrays.
	 *	@param[in] n_vertex_num Specifies the number of indices to be rendered.
	 */
	inline void Draw(GLenum n_draw_mode, size_t n_first_vertex, size_t n_vertex_num) const
	{
#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_array_object.Bind();
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_buffer.Bind();
		if(m_n_texcoord_component_num) {
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_texcoord_component_num,
				m_n_texcoord_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_texcoord_offset_bytes));
			glEnableVertexAttribArray(1);
			glVertexAttribPointer(1, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		} else {
			glDisableVertexAttribArray(1);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		}
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

		glDrawArrays(n_draw_mode, n_first_vertex, n_vertex_num);

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
#ifdef __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
		m_vertex_array_object.Release();
#endif // __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	}

protected:
	/**
	 *	@brief default constructor; has no effect
	 *	@note This constructor must be overloaded so the buffers are filled with data.
	 */
	inline CGLArraySetup()
	{}
};

/**
 *	@brief a simple class, encapsulating allocation of vertex
 *		array containing some simple statically allocated geometry
 *	@note This requires the GL_ARB_vertex_array_object extension (part of OpenGL 3.0 core).
 */
class CGLElementArraySetup {
protected:
	CGLArrayBufferObject m_vertex_buffer; /**< @brief vertex buffer */
	CGLElementArrayBufferObject m_index_buffer; /**< @brief index buffer */

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	CGLVertexArrayObject m_vertex_array_object; /**< @brief vertex array object */
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	int m_n_texcoord_component_num; /**< @brief  */
	int m_n_texcoord_data_type; /**< @brief  */
	int m_n_vertex_stride_bytes; /**< @brief  */
	int m_n_texcoord_offset_bytes; /**< @brief  */
	int m_n_position_component_num; /**< @brief  */
	int m_n_position_data_type; /**< @brief  */
	int m_n_position_offset_bytes; /**< @brief  */
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

	GLenum m_n_draw_mode; /**< @brief draw mode (name of primitives to render) */
	GLenum m_n_index_data_type; /**< @brief data type of vertex indices */
	size_t m_n_index_num; /**< @brief number of indices to render */

public:
	/**
	 *	@brief constructor; fills the arrays with data
	 *
	 *	Use this constructor to specify indexed geometry with positions
	 *	in vertex attribute 0 and texcoords in vertex attribute 1.
	 *	To specify different layouts or different types of geometry entirely,
	 *	use CBasicStaticRenderable().
	 *
	 *	@param[in] p_vertex_list is buffer containing the vertex data
	 *	@param[in] n_vertices_size_bytes is size of all the vertex data, in bytes
	 *	@param[in] n_vertex_stride_bytes is offset between two vertices, in bytes
	 *	@param[in] n_position_data_type is OpenGL data type used to represent positions
	 *	@param[in] n_position_component_num is number of position components
	 *	@param[in] n_position_offset_bytes is offset to the first position, in bytes
	 *	@param[in] n_texcoord_data_type is OpenGL data type used to represent texcoords
	 *	@param[in] n_texcoord_component_num is number of texcoord components
	 *	@param[in] n_texcoord_offset_bytes is offset to the first texcoord, in bytes
	 *	@param[in] p_index_list is buffer, containing index data
	 *	@param[in] n_index_num is number of indices to render
	 *	@param[in] n_index_data_type is data type of vertex indices
	 *	@param[in] n_draw_mode is draw mode (name of primitives to render)
	 */
	CGLElementArraySetup(const void *p_vertex_list, int n_vertices_size_bytes,
		int n_vertex_stride_bytes, GLenum n_position_data_type, int n_position_component_num,
		int n_position_offset_bytes, GLenum n_texcoord_data_type, int n_texcoord_component_num,
		int n_texcoord_offset_bytes, const void *p_index_list, size_t n_index_num,
		GLenum n_index_data_type, GLenum n_draw_mode);

	/**
	 *	@brief gets status of this renderable
	 *	@return Returns true if all the buffers and VAO allocated successfully,
	 *		otherwise returns false.
	 */
	bool b_Status() const;

	/**
	 *	@brief binds the VAO and calls glDrawElements() to display the geometry
	 */
	inline void Draw() const
	{
#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_array_object.Bind();
#ifdef __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER
		m_index_buffer.Bind(); // handle defunct VAO on mobile chipsets
#endif // __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_index_buffer.Bind();
		m_vertex_buffer.Bind();
		if(m_n_texcoord_component_num) {
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_texcoord_component_num,
				m_n_texcoord_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_texcoord_offset_bytes));
			glEnableVertexAttribArray(1);
			glVertexAttribPointer(1, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		} else {
			glDisableVertexAttribArray(1);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		}
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

		glDrawElements(m_n_draw_mode, m_n_index_num, m_n_index_data_type, 0);

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
#ifdef __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
		m_vertex_array_object.Release();
#endif // __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	}

	/**
	 *	@brief binds the VAO and calls glDrawElements() to display the geometry,
	 *		adds override option for some of the parameters
	 *
	 *	@param[in] n_draw_mode Specifies what kind of primitives to render. Symbolic constants GL_POINTS,
	 *		GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_LINE_STRIP_ADJACENCY, GL_LINES_ADJACENCY,
	 *		GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_TRIANGLES, GL_TRIANGLE_STRIP_ADJACENCY,
	 *		GL_TRIANGLES_ADJACENCY and GL_PATCHES are accepted.
	 *	@param[in] n_index_num Specifies the number of elements to be rendered.
	 *	@param[in] n_first_index_byte_offset Specifies a pointer to the location where the indices are stored.
	 */
	inline void Draw(GLenum n_draw_mode, size_t n_index_num, size_t n_first_index_byte_offset = 0) const
	{
#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_vertex_array_object.Bind();
#ifdef __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER
		m_index_buffer.Bind();
#endif // __GL_VERTEX_ARRAY_OBJECT_REBIND_INDEX_BUFFER
#else // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
		m_index_buffer.Bind();
		m_vertex_buffer.Bind();
		if(m_n_texcoord_component_num) {
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_texcoord_component_num,
				m_n_texcoord_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_texcoord_offset_bytes));
			glEnableVertexAttribArray(1);
			glVertexAttribPointer(1, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		} else {
			glDisableVertexAttribArray(1);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, m_n_position_component_num,
				m_n_position_data_type, false, m_n_vertex_stride_bytes,
				m_vertex_buffer.p_OffsetPointer(m_n_position_offset_bytes));
		}
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT

		glDrawElements(n_draw_mode, n_index_num, m_n_index_data_type,
			(const void*)n_first_index_byte_offset);

#ifndef __GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
#ifdef __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
		m_vertex_array_object.Release();
#endif // __GL_VERTEX_ARRAY_OBJECT_RELEASE_AFTER_DRAWING
#endif // !__GL_ARRAY_SETUP_DONT_USE_VERTEX_ARRAY_OBJECT
	}
	// todo - insert GLEH_ARB_vertex_array_object checks

protected:
	/**
	 *	@brief default constructor; has no effect
	 *	@note This constructor must be overloaded so the buffers are filled with data.
	 */
	inline CGLElementArraySetup()
	{}
};

#endif // __BUFFER_OBJECTS_INCLUDED
