/*
								+--------------------------------+
								|                                |
								|  ***  OpenGL bitmap font  ***  |
								|                                |
								|  Copyright  -tHE SWINe- 2010  |
								|                                |
								|         BitmapFont2.h          |
								|                                |
								+--------------------------------+
*/

#pragma once
#ifndef __BITMAP_FONT2_INCLUDED
#define __BITMAP_FONT2_INCLUDED

/**
 *	@file BitmapFont2.h
 *	@author -tHE SWINe-
 *	@brief a simple OpenGL bitmap font implementation
 *	@date 2010
 *
 *	@date 2012-02-20
 *
 *	Fixed CGLBitmapFont2::CTextureArrayBuilder by using CGLTexture::n_Target() instead
 *	of assuming one (2D array textures were emulated by 3D textures, changing target
 *	to true 3D textures made this code crash).
 */

/**
 *	@def __BITMAP_FONT_GLES20
 *	@brief if defined, OpenGL ES 2.0 is used as the
 *		OpenGL implementation (otherwise OpenGL 4.0 is used)
 */
#define __BITMAP_FONT_GLES20

/**
 *	@def __GLFONT_PRECRUNCH_GLYPH_GEOMETRY
 *	@brief if defined, font geometry is preprocessed after loading, instead of on-the-fly
 */
#define __GLFONT_PRECRUNCH_GLYPH_GEOMETRY

#include "FontTexture2.h"

#ifdef __BITMAP_FONT_GLES20
#ifdef __HAVE_GLES
#include "gles2/GLES20.h"
#else // __HAVE_GLES
#include "gles2/GLES20Emu.h"
#endif // __HAVE_GLES
#include "gles2/Texture.h"
#include "gles2/TextureUtil.h"
typedef CGLESTextureLoader CGLTextureLoader;
#else // __BITMAP_FONT_GLES20
#include "gl4/OpenGL40.h"
#include "gl4/Texture.h"
#include "gl4/TextureUtil.h"
#endif // __BITMAP_FONT_GLES20

/**
 *	@brief second generation OpenGL bitmap font
 */
class CGLBitmapFont2 {
public:
	/**
	 *	@brief character info lookup
	 */
	class CCharacterMap {
	public:
		enum {
			direct_lookup_Size = 128 /**< @brief size of the direct (O(1) time) lookup table */
		};

	protected:
		bitmapfont2::TGlyphInfo m_t_lookup[direct_lookup_Size];
		std::map<int, bitmapfont2::TGlyphInfo> m_char_map;
		// map of character code to character info

	public:
		bool Create(const CBitmapFont::CPageList &r_page_list, int n_geom_padding);
#ifdef __GLFONT_PRECRUNCH_GLYPH_GEOMETRY
		void PreCrunch(bitmapfont2::TGlyphInfo &r_t_info, int n_geom_padding);
#endif // __GLFONT_PRECRUNCH_GLYPH_GEOMETRY
		const bitmapfont2::TGlyphInfo *p_Get_CharacterInfo(int n_codepoint) const;
	};

protected:
	template <class CPassTo>
	class CPassThroughBitmapWriter : public CBitmapFont::CBitmapWriter {
	protected:
		CPassTo &m_r_pass_to;

	public:
		CPassThroughBitmapWriter(const char *p_s_format, CPassTo &r_pass_to)
			:CBitmapWriter(p_s_format), m_r_pass_to(r_pass_to)
		{}

		virtual bool operator ()(size_t n_page, size_t n_page_num, const TBmp &r_t_bitmap)
		{
			if(!m_r_pass_to(n_page, n_page_num, r_t_bitmap))
				return false;
			// call pass-through

			return CBitmapWriter::operator ()(n_page, n_page_num, r_t_bitmap);
			// call bitmap writer
		}
	};

	class CTextureArrayBuilder : public CBitmapFont::CBitmapCallback {
	protected:
		GLenum m_n_internal_format;
		bool m_b_generate_mipmaps, m_b_use_npot;
		const CBitmapFont &m_r_bitmap_font;
		CGLTexture_2D_Array *m_p_texture;

	public:
		CTextureArrayBuilder(bool b_generate_mipmaps, GLenum n_internal_format,
			bool b_use_npot, const CBitmapFont &r_bitmap_font);

		/**
		 *	@copydoc CBitmapCallback::operator()()
		 */
		virtual bool operator ()(size_t n_page, size_t n_page_num, const TBmp &r_t_bitmap);

		inline CGLTexture_2D_Array *p_Get_Texture()
		{
			return m_p_texture;
		}
	};

	class CFakeTextureArrayBuilder : public CBitmapFont::CBitmapCallback {
	protected:
		GLenum m_n_internal_format;
		bool m_b_generate_mipmaps, m_b_use_npot;
		const CBitmapFont &m_r_bitmap_font;
		CGLTexture_2D *m_p_texture;

	public:
		CFakeTextureArrayBuilder(bool b_generate_mipmaps, GLenum n_internal_format,
			bool b_use_npot, const CBitmapFont &r_bitmap_font);

		/**
		 *	@copydoc CBitmapCallback::operator()()
		 */
		virtual bool operator ()(size_t n_page, size_t n_page_num, const TBmp &r_t_bitmap);

		inline CGLTexture_2D_Array *p_Get_Texture()
		{
			return (CGLTexture_2D_Array*)m_p_texture;
		}
	};

	bitmapfont2::TFontInfo m_t_font_info; /**< @brief font information structure */
	CCharacterMap m_character_map; /**< @brief glyph info lookup map */
	int m_n_replacement_char; /**< @brief the current replacement character (utf-32) */
	CGLTexture_2D_Array *m_p_texture; /**< @brief the font texture */
	GLenum m_n_mode; /**< @brief OpenGL draw mode (GL_TRIANGLES or GL_TRIANGLE_STRIP) */
	bool m_b_allow_degenerate; /**< @brief degenerate geometry flag (only for GL_TRIANGLE_STRIP) */
	uint32_t m_n_primitive_restart_index; /**< @brief primitive restart index (only for GL_TRIANGLE_STRIP) */

public:
	/**
	 *	@brief default constructor
	 */
	CGLBitmapFont2();

	/**
	 *	@brief destructor
	 */
	~CGLBitmapFont2();

	/**
	 *	@brief gets the font texture
	 *	@return Returns const pointer to the font texture.
	 */
	inline const CGLTexture_2D_Array *p_Texture() const
	{
		return m_p_texture;
	}

	/**
	 *	@brief gets font information
	 *	@return Returns font information structure.
	 *	@note This only returns valid values after successful call to Create() or Load().
	 */
	inline const bitmapfont2::TFontInfo &t_Font_Info() const
	{
		return m_t_font_info;
	}

	// todo - support image processing in creation
	bool Create(const char *p_s_font_face, int n_font_size, CBitmapFont &r_bitmap_font,
		bool b_generate_mipmaps = false, GLenum n_internal_format =
		CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false), bool b_use_npot = false,
		const char *p_s_store_filename = 0);

	inline bool Create(const char *p_s_font_face, int n_font_size, bool b_generate_mipmaps = false,
		GLenum n_internal_format = CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false),
		bool b_use_npot = false, const char *p_s_store_filename = 0)
	{
		CBitmapFont bitmap_font;

		bitmap_font.Enable_MetaBlock_Allocation(p_s_store_filename != 0);
		// only alloc metadata if they're not being stored

		bitmap_font.Set_Max_PageSize(CGLTexture::n_Max_Size());
		// configure the bitmap font (config beyond defaults)

		return Create(p_s_font_face, n_font_size, bitmap_font, b_generate_mipmaps,
			n_internal_format, b_use_npot, p_s_store_filename);
		// use the other function to build the font
	}

	bool Load(const char *p_s_filename, bool b_generate_mipmaps = false,
		GLenum n_internal_format = CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false),
		bool b_use_npot = false);

	bool Load(CBitmapFont::CBitmapProducer &r_bitmap_producer, bool b_generate_mipmaps = false,
		GLenum n_internal_format = CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false),
		bool b_use_npot = false)
	{
		if(m_p_texture) {
			if(m_p_texture) {
				if(b_IsTexture2D(m_p_texture->n_Target()))
					delete (CGLTexture_2D*)m_p_texture;
				else {
					_ASSERTE(b_IsTexture3D(m_p_texture->n_Target()));
					delete m_p_texture;
				}
			}
			m_p_texture = 0;
		}
		// dispose of the old texture, if any

		CBitmapFont bitmap_font;
		CTextureArrayBuilder tex_builder(b_generate_mipmaps,
			n_internal_format, b_use_npot, bitmap_font);
		if(!bitmap_font.Load(r_bitmap_producer, tex_builder, false))
			return false;
		m_p_texture = tex_builder.p_Get_Texture();
		// load the bitmap font and build the texture

		m_t_font_info = bitmap_font.t_Font_Info();
		// get font info

		if(!m_character_map.Create(bitmap_font.r_Page_List(), m_t_font_info.n_glyph_geom_padding))
			return false;
		// create character map

		m_n_replacement_char = 0x20;
		// reset the replacement character to space

		return true;
	}

	bool Load_2D(CBitmapFont::CBitmapProducer &r_bitmap_producer, bool b_generate_mipmaps = false,
		GLenum n_internal_format = CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false),
		bool b_use_npot = false)
	{
		if(m_p_texture) {
			if(m_p_texture) {
				if(b_IsTexture2D(m_p_texture->n_Target()))
					delete (CGLTexture_2D*)m_p_texture;
				else {
					_ASSERTE(b_IsTexture3D(m_p_texture->n_Target()));
					delete m_p_texture;
				}
			}
			m_p_texture = 0;
		}
		// dispose of the old texture, if any

		CBitmapFont bitmap_font;
		CFakeTextureArrayBuilder tex_builder(b_generate_mipmaps,
			n_internal_format, b_use_npot, bitmap_font);
		if(!bitmap_font.Load(r_bitmap_producer, tex_builder, false))
			return false;
		m_p_texture = tex_builder.p_Get_Texture();
		// load the bitmap font and build the texture

		m_t_font_info = bitmap_font.t_Font_Info();
		// get font info

		if(!m_character_map.Create(bitmap_font.r_Page_List(), m_t_font_info.n_glyph_geom_padding))
			return false;
		// create character map

		m_n_replacement_char = 0x20;
		// reset the replacement character to space

		return true;
	}

	bool Load_2D(const char *p_s_filename, bool b_generate_mipmaps = false,
		GLenum n_internal_format = CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false),
		bool b_use_npot = false)
	{
		if(m_p_texture) {
			if(m_p_texture) {
				if(b_IsTexture2D(m_p_texture->n_Target()))
					delete (CGLTexture_2D*)m_p_texture;
				else {
					_ASSERTE(b_IsTexture3D(m_p_texture->n_Target()));
					delete m_p_texture;
				}
			}
			m_p_texture = 0;
		}
		// dispose of the old texture, if any

		CBitmapFont bitmap_font;
		CFakeTextureArrayBuilder tex_builder(b_generate_mipmaps,
			n_internal_format, b_use_npot, bitmap_font);
		if(!bitmap_font.Load(p_s_filename, tex_builder, false))
			return false;
		m_p_texture = tex_builder.p_Get_Texture();
		// load the bitmap font and build the texture

		m_t_font_info = bitmap_font.t_Font_Info();
		// get font info

		if(!m_character_map.Create(bitmap_font.r_Page_List(), m_t_font_info.n_glyph_geom_padding))
			return false;
		// create character map

		m_n_replacement_char = 0x20;
		// reset the replacement character to space

		return true;
	}

	/**
	 *	@brief sets generated primitive type
	 *
	 *	@param[in] n_mode is either GL_TRIANGLES or GL_TRIANGLE_STRIP
	 *	@param[in] b_allow_degenerate is degenerate flag (only applies on the triangle strip)
	 *	@param[in] n_primitive_restart is primitive restart vertex is (only applies on the
	 *		triangle strip if b_allow_degenerate is false)
	 *
	 *	@return Returns true on success, false on failure.
	 */
	bool Set_Mode(GLenum n_mode, bool b_allow_degenerate, int n_primitive_restart);

	bool Set_ReplacementChar(int n_replacement_char);

	inline GLenum n_Primitive_Mode() const
	{
		return m_n_mode;
	}

	inline bool b_Need_ElementArray_Buffer() const
	{
		return true;
	}

	inline bool b_Primitive_Restarts() const
	{
		return m_n_mode == GL_TRIANGLE_STRIP && !m_b_allow_degenerate;
	}

	inline uint32_t n_Primitive_RestartIndex() const
	{
		return m_n_primitive_restart_index;
	}

	inline GLenum n_Vertex_DataType() const
	{
		return GL_FLOAT;
	}

	inline size_t n_Vertex_Size() const
	{
		return 5 * sizeof(float);
	}

	inline GLenum n_Index_DataType() const
	{
		return GL_UNSIGNED_INT;
	}

	// @note the upper bound may be approximated easily using strlen()
	size_t n_Printable_Char_Num(const char *p_s_utf8) const;

	inline size_t n_Vertex_Num(size_t n_printable_char_num) const
	{
		return 4 * n_printable_char_num;
	}

	inline size_t n_Index_Num(size_t n_printable_char_num) const
	{
		return (m_n_mode == GL_TRIANGLES)? n_printable_char_num * 6 :
			n_printable_char_num * 4 + (max(size_t(1), n_printable_char_num) - 1) *
			((m_b_allow_degenerate)? 2 : 1);
	}

	/**
	 *	@brief generates geometry for displayable characters
	 *
	 *	@param[out] p_position is pointer to the array to store the vertex positions
	 *	@param[in] n_position_stride is number of bytes between vertex positions
	 *	@param[out] p_texcoord is pointer to the array to store the vertex texcoords
	 *	@param[out] n_texcoord_stride is number of bytes between vertex texcoords
	 *	@param[in,out] r_n_vertex_space is capacity of the vertex buffer on input (in vertices)
	 *		and the actual number of vertices written upon return
	 *	@param[in] n_index_offset is value of the first vertex index
	 *	@param[out] p_index	is pointer to the array to store indices
	 *	@param[in,out] r_n_index_space is capacity of the index buffer on input (in indices)
	 *		and the actual number of index written upon return
	 *	@param[in] p_s_utf8_begin is the text to be rendered, encoded as utf-8
	 *	@param[in] p_s_utf8_end is pointer to one past the last byte of the text
	 *	@param[in] v_position is position of the first character
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@note r_n_vertex_space is in vertices (ie. each unit is 2 positions + 3 texcoords,
	 *		that is 20 bytes if using 32-bit float for vertices and texcoords).
	 *	@note r_n_index_space is in indices (ie. each unit is 4 bytes if using
	 *		uint32_t for indices).
	 */
	bool Generate_Geometry(float *p_position, int n_position_stride,
		float *p_texcoord, int n_texcoord_stride, size_t &r_n_vertex_space,
		uint32_t n_index_offset, uint32_t *p_index, size_t &r_n_index_space,
		const char *p_s_utf8_begin, const char *p_s_utf8_end,
		Vector2f v_position, float f_zoom) const;

	/**
	 *	@brief generates geometry for displayable characters
	 *
	 *	@param[out] p_position is pointer to the array to store the vertex positions
	 *	@param[in] n_position_stride is number of bytes between vertex positions
	 *	@param[out] p_texcoord is pointer to the array to store the vertex texcoords
	 *	@param[out] n_texcoord_stride is number of bytes between vertex texcoords
	 *	@param[in,out] r_n_vertex_space is capacity of the vertex buffer on input (in vertices)
	 *		and the actual number of vertices written upon return
	 *	@param[in] n_index_offset is value of the first vertex index
	 *	@param[out] p_index	is pointer to the array to store indices
	 *	@param[in,out] r_n_index_space is capacity of the index buffer on input (in indices)
	 *		and the actual number of index written upon return
	 *	@param[in] p_s_utf8 is the text to be rendered, encoded as null-terminated utf-8 string
	 *	@param[in] v_position is position of the first character
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@note r_n_vertex_space is in vertices (ie. each unit is 2 positions + 3 texcoords,
	 *		that is 20 bytes if using 32-bit float for vertices and texcoords).
	 *	@note r_n_index_space is in indices (ie. each unit is 4 bytes if using
	 *		uint32_t for indices).
	 */
	inline bool Generate_Geometry(float *p_position, int n_position_stride,
		float *p_texcoord, int n_texcoord_stride, size_t &r_n_vertex_space,
		uint32_t n_index_offset, uint32_t *p_index, size_t &r_n_index_space,
		const char *p_s_utf8, Vector2f v_position, float f_zoom) const
	{
		return Generate_Geometry(p_position, n_position_stride, p_texcoord,
			n_texcoord_stride, r_n_vertex_space, n_index_offset, p_index,
			r_n_index_space, p_s_utf8, (p_s_utf8)? p_s_utf8 + strlen(p_s_utf8) : 0,
			v_position, f_zoom);
		// handle null-terminated
	}

	/**
	 *	@brief generates geometry for displayable characters (a simpler variant where the
	 *		vertices are assumed to be in interleaved 3D texcoord, 2D position format)
	 *
	 *	@param[out] p_t3v2_vertex is pointer to the array to store the vertex data
	 *	@param[in,out] r_n_vertex_space is capacity of the vertex buffer on input (in vertices)
	 *		and the actual number of vertices written upon return
	 *	@param[in] n_index_offset is value of the first vertex index
	 *	@param[out] p_index	is pointer to the array to store indices
	 *	@param[in,out] r_n_index_space is capacity of the index buffer on input (in indices)
	 *		and the actual number of index written upon return
	 *	@param[in] p_s_utf8_begin is the text to be rendered, encoded as utf-8
	 *	@param[in] p_s_utf8_end is pointer to one past the last byte of the text
	 *	@param[in] v_position is position of the first character
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@note r_n_vertex_space is in vertices (ie. each unit is 2 positions + 3 texcoords,
	 *		that is 20 bytes if using 32-bit float for vertices and texcoords).
	 *	@note r_n_index_space is in indices (ie. each unit is 4 bytes if using
	 *		uint32_t for indices).
	 */
	inline bool Generate_Geometry(float *p_t3v2_vertex, size_t &r_n_vertex_space,
		uint32_t n_index_offset, uint32_t *p_index, size_t &r_n_index_space,
		const char *p_s_utf8_begin, const char *p_s_utf8_end,
		Vector2f v_position, float f_zoom) const
	{
		return Generate_Geometry(p_t3v2_vertex + 3, 5 * sizeof(float), p_t3v2_vertex,
			5 * sizeof(float), r_n_vertex_space, n_index_offset, p_index,
			r_n_index_space, p_s_utf8_begin, p_s_utf8_end, v_position, f_zoom);
	}

	/**
	 *	@brief generates geometry for displayable characters (a simpler variant where the
	 *		vertices are assumed to be in interleaved 3D texcoord, 2D position format)
	 *
	 *	@param[out] p_t3v2_vertex is pointer to the array to store the vertex data
	 *	@param[in,out] r_n_vertex_space is capacity of the vertex buffer on input (in vertices)
	 *		and the actual number of vertices written upon return
	 *	@param[in] n_index_offset is value of the first vertex index
	 *	@param[out] p_index	is pointer to the array to store indices
	 *	@param[in,out] r_n_index_space is capacity of the index buffer on input (in indices)
	 *		and the actual number of index written upon return
	 *	@param[in] p_s_utf8 is the text to be rendered, encoded as null-terminated utf-8 string
	 *	@param[in] v_position is position of the first character
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@note r_n_vertex_space is in vertices (ie. each unit is 2 positions + 3 texcoords,
	 *		that is 20 bytes if using 32-bit float for vertices and texcoords).
	 *	@note r_n_index_space is in indices (ie. each unit is 4 bytes if using
	 *		uint32_t for indices).
	 */
	inline bool Generate_Geometry(float *p_t3v2_vertex, size_t &r_n_vertex_space,
		uint32_t n_index_offset, uint32_t *p_index, size_t &r_n_index_space,
		const char *p_s_utf8, Vector2f v_position, float f_zoom) const
	{
		return Generate_Geometry(p_t3v2_vertex + 3, 5 * sizeof(float), p_t3v2_vertex,
			5 * sizeof(float), r_n_vertex_space, n_index_offset, p_index, r_n_index_space,
			p_s_utf8, v_position, f_zoom);
	}

	/**
	 *	@brief calculates size of a block of text
	 *
	 *	@param[in] p_s_utf8 is the text to be rendered, encoded as null-terminated utf-8 string
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@return Returns size of the text as a Vector3f, where (x, y) contain the
	 *		final caret shift, z contains width of the text in the widest point.
	 */
	inline Vector3f v_Text_Size(const char *p_s_utf8, float f_zoom) const
	{
		return v_Text_Size(p_s_utf8, (p_s_utf8)? p_s_utf8 +
			strlen(p_s_utf8) : 0, f_zoom);
	}

	/**
	 *	@brief calculates size of a block of text
	 *
	 *	@param[in] p_s_utf8_begin is the text to be rendered, encoded as utf-8
	 *	@param[in] p_s_utf8_end is pointer to one past the last byte of the text
	 *	@param[in] f_zoom is relative size of the rendered text
	 *		(if set to 1, the units of generated geometry are pixels)
	 *
	 *	@return Returns size of the text as a Vector3f, where (x, y) contain the
	 *		final caret shift, z contains width of the text in the widest point.
	 */
	Vector3f v_Text_Size(const char *p_s_utf8_begin,
		const char *p_s_utf8_end, float f_zoom) const;

protected:
	static inline bool b_Is_Glyph(int n_code);
};

#endif // __BITMAP_FONT2_INCLUDED
