#include "../../UberLame_src/NewFix.h"
#include "../../UberLame_src/CallStack.h"
#include <crtdbg.h>
#include <stdio.h>
#include <vector>
#include "../../UberLame_src/Integer.h"
#include "../../UberLame_src/Dir.h"
#include "../../UberLame_src/Thread.h"
#include "../../UberLame_src/Timer.h"
#include "PNGLoad.h"
#include "DPX.h"

template <const int n_channel_num, class _TyInputData>
void ColorspaceConvert(const _TyInputData *p_channels, size_t n_pixel_num, int n_shift, uint32_t *p_dest)
{
	enum {
		shift_Right = sizeof(_TyInputData) > 1
	};

	for(size_t i = 0; i < n_pixel_num; ++ i, p_channels += n_channel_num, ++ p_dest) {
		if(n_channel_num == 1) {
			int n_gray = (shift_Right)? *p_channels >> n_shift : *p_channels << n_shift;
			*p_dest = 0xff000000U | n_gray | (n_gray << 8) | (n_gray << 16);
		} else if(n_channel_num == 3) {
			int n_r = (shift_Right)? p_channels[0] >> n_shift : p_channels[0] << n_shift;
			int n_g = (shift_Right)? p_channels[1] >> n_shift : p_channels[1] << n_shift;
			int n_b = (shift_Right)? p_channels[2] >> n_shift : p_channels[2] << n_shift;
			*p_dest = 0xff000000U | n_b | (n_g << 8) | (n_r << 16);
		} else if(n_channel_num == 4) {
			int n_r = (shift_Right)? p_channels[0] >> n_shift : p_channels[0] << n_shift;
			int n_g = (shift_Right)? p_channels[1] >> n_shift : p_channels[1] << n_shift;
			int n_b = (shift_Right)? p_channels[2] >> n_shift : p_channels[2] << n_shift;
			int n_a = (shift_Right)? p_channels[3] >> n_shift : p_channels[3] << n_shift;
			*p_dest = (n_a << 24) | n_b | (n_g << 8) | (n_r << 16);
		}
	}
}

static void PrintShortFilename(const char *p_s_filename, int n_max_length = 80,
	const char *p_s_text_before = "", const char *p_s_text_after = "")
{
	n_max_length -= strlen(p_s_text_before);
	n_max_length -= max(size_t(1), strlen(p_s_text_after)) - 1; // don't count the newline
	_ASSERTE(n_max_length >= 4);
	if(strlen(p_s_filename) > unsigned(n_max_length)) {
		const char *p_s_shorter = p_s_filename;
		while(strlen(p_s_shorter) > unsigned(n_max_length - 4)) {
			const char *p_ptr_a = strchr(p_s_shorter, '\\');
			const char *p_ptr_b = strchr(p_s_shorter, '/');
			if(p_ptr_a && p_ptr_b)
				p_s_shorter = min(p_ptr_a, p_ptr_b) + 1;
			else if(p_ptr_a)
				p_s_shorter = p_ptr_a + 1;
			else if(p_ptr_b)
				p_s_shorter = p_ptr_b + 1;
			else
				break;
		}
		if(strlen(p_s_shorter) <= unsigned(n_max_length - 4))
			printf("%s...\\%s%s", p_s_text_before, p_s_shorter, p_s_text_after);
		else
			printf("%s...%s%s", p_s_text_before, p_s_shorter + strlen(p_s_shorter) - n_max_length - 3, p_s_text_after);
	} else
		printf("%s%s%s", p_s_text_before, p_s_filename, p_s_text_after);
	// print filename, omit full path if too long
}

bool ConvertSingle(const TFileInfo &r_t_file)
{
	if(r_t_file.b_directory || _stricmp(r_t_file.p_s_Extension(), "dpx"))
		return true;
	// only .dpx files

	const char *p_s_infile = r_t_file.p_s_Path();

	//printf("converting \'%s\' ...\n", p_s_infile);
	PrintShortFilename(p_s_infile, 80, "converting \'", " ...\n");
	// verbose

	InStream i;
	if(!i.Open(p_s_infile)) {
		fprintf(stderr, "error: failed to open \'%s\'\n", p_s_infile);
		return false;
	}
	dpx::Reader reader;
	reader.SetInStream(&i);
	if(!reader.ReadHeader()) {
		fprintf(stderr, "error: failed to read header of \'%s\'\n", p_s_infile);
		return false;
	}
	int n_height = reader.header.Height();
	int n_width = reader.header.Width();
	_ASSERTE(reader.header.NumberOfElements() == 1); // one element per image?
	int n_channel_num = reader.header.ImageElementComponentCount(0);
	int n_bytes_per_channel = reader.header.ComponentByteCount(0);
	int n_bytes_per_pixel = n_channel_num * n_bytes_per_channel;
	//printf("it is an %d x %d image, %d channels, %d B/pixel\n", n_width, n_height, n_channel_num, n_bytes_per_pixel);
	size_t n_size = n_width * n_height * n_bytes_per_pixel;
	std::vector<uint8_t> image_buffer(n_size);
	if(!reader.ReadImage(0, &image_buffer[0])) {
		fprintf(stderr, "error: image decoding failed with \'%s\'\n", p_s_infile);
		return false;
	}
	i.Close();
	// read the image

	//printf("saving as png ...\n");

	TBmp *p_bitmap = TBmp::p_Alloc(n_width, n_height, false, false, 8/*reader.header.BitDepth(0)*/);
	if(!p_bitmap)
		throw std::bad_alloc(); // rethrow

	//int n_shift = abs(8 - reader.header.BitDepth(0)); 
	int n_shift = abs(8 - n_bytes_per_channel * 8); // it is MSB aligned
	if(n_bytes_per_channel == 2) {
		if(n_channel_num == 1)
			ColorspaceConvert<1>((uint16_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 3)
			ColorspaceConvert<3>((uint16_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 4)
			ColorspaceConvert<4>((uint16_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else
			return false;
	} else if(n_bytes_per_channel == 1) {
		if(n_channel_num == 1)
			ColorspaceConvert<1>((uint8_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 3)
			ColorspaceConvert<3>((uint8_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 4)
			ColorspaceConvert<4>((uint8_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else
			return false;
	} else if(n_bytes_per_channel == 4) {
		if(n_channel_num == 1)
			ColorspaceConvert<1>((uint32_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 3)
			ColorspaceConvert<3>((uint32_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else if(n_channel_num == 4)
			ColorspaceConvert<4>((uint32_t*)&image_buffer[0], n_width * n_height, n_shift, p_bitmap->p_buffer);
		else
			return false;
	} else
		return false;
	// colorspace convert

	std::string s_filename = p_s_infile;
	s_filename.erase(s_filename.rfind('.'));
	s_filename += ".png";
	CPngCodec::Save_PNG(s_filename.c_str(), *p_bitmap, true);
	// save as png

	p_bitmap->Delete();

	return true;
}

// observe, a parallel code that can be written before your gitlfriend comes back from the dance class
class CThreadedConverter {
protected:
	class CWorker : public CRunable_Thread_ShallowCopy { // working thread
	protected:
		CProducerConsumerQueue<std::string> *m_p_queue; // pointer to the queue

	public:
		CWorker(CProducerConsumerQueue<std::string> &r_queue) // default constructor
			:m_p_queue(&r_queue)
		{}

	protected:
		virtual void Run() // thread execution function
		{
			for(;;) {
				std::string s_file;
				if(!m_p_queue->Get(s_file)) { // get a next file
					if(m_p_queue->b_Finished()) // is ther no more?
						return; // finished
					throw std::runtime_error("work item queue get failed");
				}
				ConvertSingle(TFileInfo(s_file.c_str())); // convert the file using the serial function (is reentrant)
			}
		}
	};

	CProducerConsumerQueue<std::string> m_queue; // queue for passing work-items (filenames)
	std::vector<CWorker> m_workers; // vector of workers

public:
	CThreadedConverter()
		:m_queue(64) // space for 64 files to be queued
	{
		m_workers.resize(CThread::n_CPU_Num(), CWorker(m_queue)); // as many workers as there are CPUs
	}

	bool StartWorkers() // start all the workers (could be done in constructor)
	{
		for(size_t i = 0, n = m_workers.size(); i < n; ++ i) {
			if(!m_workers[i].Start())
				return false;
		}
		return true;
	}

	bool WaitForFinish() // wait for all the threads to finish
	{
		if(!m_queue.Signal_Finished())
			throw std::runtime_error("work item queue signal finished failed");
		for(size_t i = 0, n = m_workers.size(); i < n; ++ i) {
			if(!m_workers[i].WaitForFinish())
				return false;
		}
		return true;
	}

	inline bool operator ()(const TFileInfo &r_t_file) // file callback
	{
		return m_queue.Put(std::string(r_t_file.p_s_Path())); // just put it in the queue
	}
};

int main(int n_arg_num, const char **p_arg_list)
{
	if(n_arg_num != 2) {
		fprintf(stderr, "error: use DPX_to_PNG <file or folder>\n");
		return -1;
	}
	const char *p_s_infile = p_arg_list[1];
	CTimer t;
	try {
		TFileInfo t_file(p_s_infile);
		if(t_file.b_directory) {
#if 1
			CThreadedConverter runner;
			if(!runner.StartWorkers()) {
				fprintf(stderr, "error: threading failed\n");
				return -1;
			}
			if(!CDirTraversal::Traverse2(p_s_infile, runner, false))
				return -1;
			if(!runner.WaitForFinish()) {
				fprintf(stderr, "error: threading failed\n");
				return -1;
			}
#else // 1
			if(!CDirTraversal::Traverse2(p_s_infile, ConvertSingle, false)) // serial version of the code
				return -1;
#endif // 1
		} else {
			if(!ConvertSingle(t_file))
				return -1;
		}
	} catch(std::exception &r_exc) {
		fprintf(stderr, "error: uncaught exception: \'%s\'\n", r_exc.what());
		return -1;
	}
	double f_time = t.f_Time();
	printf("done. it took " PRItime "\n", PRItimeparams(f_time));
	return 0;
}
