﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Threading;
using System.Collections.ObjectModel;
using System.Windows.Controls;
using System.IO;
using System.ComponentModel;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Has information and settings of simulator and methods for start/stop/pause simulation
    /// and for performing simulation steps.
    /// </summary>
    [Serializable]
    public class World : INotifyPropertyChanged
    {
        /// <summary>
        /// Lane width
        /// </summary>
        public const double LaneWidth = 3.0;

        /// <summary>
        /// Transition constant between kph and mps
        /// </summary>
        public const double mps2kph = 3.6;

        /// <summary>
        /// After every tick is performed next simulation step
        /// </summary>
        [NonSerialized]
        protected DispatcherTimer simulationStepTimer;

        /// <summary>
        /// Time of last perform of simulation step
        /// </summary>        
        public DateTime LastSimulationStepTime { get; protected set; }

        /// <summary>
        /// Time of pressing pause button
        /// </summary>
        [NonSerialized]
        protected TimeSpan? elapsedTime;

        /// <summary>
        /// Simulation state
        /// </summary>
        [NonSerialized]
        protected ESimulationState simulationState = ESimulationState.Stopped;
        /// <summary>
        /// Simulation state
        /// </summary>        
        public ESimulationState SimulationState 
        {
            get { return simulationState; }
            set { Set<ESimulationState>(ref simulationState, value, "SimulationState"); }
        }

        /// <summary>
        /// Speed of simulation. When bigger than 1 simulation is accelerated. 
        /// When less than 1 simulation is slower
        /// </summary>
        [NonSerialized]
        protected double simulationSpeed = 1.0;
        public double SimulationSpeed
        {
            get { return simulationSpeed; }
            set 
            {
                if (simulationSpeed != value)
                    UpdateTimerIntervals(simulationSpeed, value);
                Set<double>(ref simulationSpeed, value, "SimulationSpeed"); 
            }
        }

        /// <summary>
        /// Simulation time
        /// </summary>
        [NonSerialized]
        protected TimeSpan simulationTime = TimeSpan.FromSeconds(0);
        public TimeSpan SimulationTime
        {
            get { return simulationTime; }
            set { Set<TimeSpan>(ref simulationTime, value, "SimulationTime"); }
        }
        
        /// <summary>
        /// Time in seconds between every simulation step. 
        /// Default value 0.36 causes and cellColumnPair length 0.5 metres
        /// causes to represent velocity of crossing one cellColumnPair as
        /// 5 kph
        /// </summary>
        public const double SimulationStepTime = 0.36;

        /// <summary>
        /// Cell length in metres
        /// </summary>
        public const double cellLength = 0.5;

        /// <summary>
        /// Minimal cell length
        /// </summary>
        public const double MinCellLength = 0.05;

        /// <summary>
        /// Matrix cellColumnPair width
        /// </summary>
        protected const double MatrixCellSize = 100;        

        /// <summary>
        /// Counter which assigns id to new car and then increments itself
        /// </summary>
        [NonSerialized]
        protected int carsIdCounter;

        /// <summary>
        /// List of all generators in the World
        /// </summary>
        protected List<Generator> generators = new List<Generator>();

        /// <summary>
        /// Dictionary of cars with their cellLocation on the front cellColumnPair.
        /// It is source collection during computing
        /// </summary>
        [NonSerialized]
        protected Dictionary<Car, Cell> carsOldPositions = new Dictionary<Car,Cell>();

        /// <summary>
        /// Dictionary of cars with their cellLocation on the front cellColumnPair.
        /// It is destination collection during computing
        /// </summary>
        [NonSerialized]
        protected Dictionary<Car, Cell> carsNewPositions = new Dictionary<Car,Cell>();

        /// <summary>
        /// Cars which left World in previous simulation step
        /// </summary>
        [NonSerialized]
        protected List<Car> carsToRemove = new List<Car>();

        protected List<LaneSegment> laneSegments = new List<LaneSegment>();
        
        [NonSerialized]
        protected DispatcherTimer statisticsTimer = new DispatcherTimer(DispatcherPriority.Normal);

        //[NonSerialized]
        //List<Cell> statisticsCells = new List<Cell>();        

        /// <summary>
        /// Initializes World instance by initializing its timer and lists
        /// </summary>
        public World()
        {
            //initialize simulation step timer
            simulationStepTimer = new DispatcherTimer(DispatcherPriority.Normal);
            simulationStepTimer.Interval = TimeSpan.FromSeconds(SimulationStepTime);
            simulationStepTimer.Tick += new EventHandler(simulationStepTimer_Tick);
            //initialize statistics timer
            statisticsTimer.Interval = TimeSpan.FromSeconds(10);
            statisticsTimer.Tick += new EventHandler(statisticsTimer_Tick);
        }
        
        public void ResetCars()
        {
            SimulationTime = TimeSpan.FromSeconds(0);
            foreach (var i in carsNewPositions)
            {
                if (i.Value != null)
                {
                    i.Value.RemoveCar(i.Key);
                    i.Key.ClearCrossroadCellsWhereItIsPlanningToGo();
                }
            }
            carsNewPositions.Clear();
            carsOldPositions.Clear();
            carsToRemove.Clear();
            foreach (var i in generators)
            {
                while (i.HasTicket)
                    i.GetTicket();
                if (SimulationState == ESimulationState.Stopped)
                    i.ConnectedCell = null;
            }
            GC.Collect();
        }

        /// <summary>
        /// Get all generators in World
        /// </summary>
        /// <returns>Generator</returns>
        public IEnumerable<Generator> GetGenerators()
        {
            foreach (Generator generator in generators)
                yield return generator;
        }

        /// <summary>
        /// Get all cars in World
        /// </summary>
        /// <returns>Pair of pairs car-cellColumnPair</returns>
        public IEnumerable<KeyValuePair<Car,KeyValuePair<Cell, Cell>>> GetCars()
        {
            foreach (KeyValuePair<Car, Cell> i in carsNewPositions)
                yield return new KeyValuePair<Car, KeyValuePair<Cell, Cell>>
                    (i.Key, new KeyValuePair<Cell, Cell>(i.Value, carsOldPositions[i.Key]));
        }
        

        /// <summary>
        /// Determines if point is inside of rectangle
        /// </summary>
        /// <param name="rectanglePoint1">Left-down vertex of rectangle</param>
        /// <param name="rectanglePoint2">Right-up vertex of rectangle</param>
        /// <param name="point">The point</param>
        /// <returns>Result if the point is inside rectangle</returns>
        public bool IsPointInRectangle(Point rectanglePoint1, Point rectanglePoint2, Point point)
        {
            if ((point.X > rectanglePoint1.X && point.X > rectanglePoint2.X) ||
                (point.X < rectanglePoint1.X && point.X < rectanglePoint2.X) ||
                (point.Y > rectanglePoint1.Y && point.Y > rectanglePoint2.Y) ||
                (point.Y < rectanglePoint1.Y && point.Y < rectanglePoint2.Y))
                return false;
            return true;
        }

        /// <summary>
        /// Performs simulation step
        /// </summary>
        protected void SimulateStep()
        {
            SimulationTime += TimeSpan.FromSeconds(World.SimulationStepTime);            
            
            //replace old cars references by new cars references
            Dictionary<Car, Cell> pom = carsOldPositions;
            carsOldPositions = carsNewPositions;
            carsNewPositions = pom;

            //remove cars which left World in previous simulation step from dictionaries
            foreach (Car car in carsToRemove)
            {
                carsOldPositions.Remove(car);
                carsNewPositions.Remove(car);
            }
            carsToRemove.Clear();

            //check generator for tickets
            foreach (Generator generator in generators)
                if (generator.HasTicket)
                    //generator has ticket so try add new car
                    if (AddNewCar(generator.ConnectedCell))
                        //car was possible to add so get one ticket
                        generator.GetTicket();

            //perform transition to all cars without releasing any cell
            foreach (KeyValuePair<Car, Cell> frontCellCarPair in carsOldPositions)
            {
                Car car = frontCellCarPair.Key;
                Cell currentLocation = frontCellCarPair.Value;                                
                //compute new car's cellLocation
                Cell newLocation = car.Move(currentLocation);
                //assign new cellColumnPair cellLocation to dictionary
                carsNewPositions[car] = newLocation;
                if (newLocation == null)
                    //car left World so add it to carsToRemove list
                    carsToRemove.Add(car);
            }

            //release cells which car were crossing through during this simulate step and not already occupying them
            foreach (KeyValuePair<Car, Cell> frontCellCarPair in carsNewPositions)
            {
                Car car = frontCellCarPair.Key;
                Cell currentLocation = frontCellCarPair.Value;
                //release
                if (currentLocation != null)
                    car.Release(currentLocation);                
            }
        }

        /// <summary>
        /// Add new car on the cellColumnPair
        /// </summary>
        /// <param name="cellColumnPair">Cell where to add the car</param>
        /// <returns>Operation success</returns>
        protected bool AddNewCar(Cell cell)
        {
            //check if cellColumnPair is free
            while (!cell.IsFree)
            {
                if (cell.CellOnTheLeft == null || cell.CellOnTheLeft.CellOnTheLeft == cell)
                    break;
                cell = cell.CellOnTheLeft;
            }
            if (!cell.IsFree)
                return false;
            //find free id number add new car to cellColumnPair and dictionaries
            Car newCar;
            while (carsOldPositions.ContainsKey(newCar = new Car(++carsIdCounter, 14, 18))) ;
            cell.Seize(newCar);
            carsOldPositions.Add(newCar, cell);
            carsNewPositions.Add(newCar, cell);
            return true;
        }

        /// <summary>
        /// Event handler for simulation step timer
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void simulationStepTimer_Tick(object sender, EventArgs e)
        {
            //set interval to correct interval (after calling StartTimers could be shorter)
            simulationStepTimer.Interval = TimeSpan.FromSeconds(SimulationStepTime);
            //update lasttick
            LastSimulationStepTime = DateTime.Now;
            //perform simulation step
            SimulateStep();
        }

        /// <summary>
        /// Starts simulation timer and generator timers
        /// </summary>
        public void StartTimers(bool previousConstructionAvailable)
        {
            //build cells
            if (previousConstructionAvailable)
                BuildCells();
            SimulationState = ESimulationState.Running;
            //check if elapsed time is not set
            if (elapsedTime != null)
            {
                //timer has been already running, decrease timer interval
                if (simulationStepTimer.Interval - elapsedTime.Value > TimeSpan.FromSeconds(0))
                    simulationStepTimer.Interval -= elapsedTime.Value;
                else
                    simulationStepTimer.Interval = TimeSpan.FromSeconds(0);
            }
            //get time of starting timer
            LastSimulationStepTime = DateTime.Now;
            //start timer
            simulationStepTimer.Start();
            statisticsTimer.Start();
            //start generator timers   
            foreach (Generator generator in generators)
                generator.StartTimer();
            //show cell statistics window
            //foreach (Cell cell in statisticsCells)
            //    cell.OpenStatisticsWindow();
        }

        private void ConnectFollowingCells(Cell far, Cell near, double? priority)
        {            
            if (near.FollowingCells.Count == 0)
                near.FollowingCell = far;
            else
                near.AddFollowingCell(far);

            if (far.PreviousCells.Count == 0)
                far.PreviousCell = near;
            else
                far.AddPreviousCell(near);
            near.Priority = priority;
        }
        
        private void BuildCells()
        {
            int cellIdCounter = 0;
            //cell references for building left-right relationships
            Cell firstCellOfRightNeighborLane_ForNextSegment = null;
            Cell firstCellOfRightNeighborLane_Current = null;
            //previous lane segment reference
            LaneSegment previousLaneSegment = null;
            //current way
            EWay currentWay = EWay.Normal;
            //go through all segments
            foreach (LaneSegment ls in laneSegments)
            {
                //copy generator instance to list of generators
                if (ls.Generator != null)
                    generators.Add(ls.Generator);
                
                //set flags to proper setting of left-right references
                bool setFirstCellOfRightNeighborLane = false;
                bool makeCellNeighborReferences = false;
                if (ls.IsEdgeLaneSegmentOnTheRight)
                {
                    setFirstCellOfRightNeighborLane = true;
                }
                else if (ls.IsEdgeLaneSegmentOnTheLeft)
                {
                    makeCellNeighborReferences = true;
                }
                else if (!ls.IsSoleLane)
                {
                    setFirstCellOfRightNeighborLane = true;
                    makeCellNeighborReferences = true;
                }
                firstCellOfRightNeighborLane_Current = firstCellOfRightNeighborLane_ForNextSegment;
                firstCellOfRightNeighborLane_ForNextSegment = null;

                //help variables
                Cell firstCell = null;
                Cell cellBefore = null;
                Cell currentCell = null;

                bool firstSettingOfFollowingCell = true;
                bool firstCellIsBuilt = false;
                bool lastCellIsBuilt = false;
                bool maxTurnSpeedWasComputed = false; //very short 

                //left neighbor lane
                if (ls.IsEdgeLaneSegmentOnTheRight && ls.RightNeighborLane != null)
                    currentWay = EWay.Reversed;

                //build cells                
                foreach (Tuple<Point, Vector, double> cellLocationAndDirectionVector in
                    ls.GetLanePointsAndDirectionVectors(World.cellLength, 0.0, 1.0))
                {
                    //store cell built in previous iteration
                    cellBefore = currentCell;
                    //gained priority
                    double? priority = null;

                    if (cellLocationAndDirectionVector.Item3 == 0.0)
                    {
                        //first cell
                        if (ls.StartCell == null)
                        {
                            //build new cell
                            currentCell = new Cell(++cellIdCounter, cellLocationAndDirectionVector.Item1,
                                cellLocationAndDirectionVector.Item2, 18, cellLocationAndDirectionVector.Item3, ls.Way);
                            //set startcell reference
                            ls.SetStartCell(currentCell);                            
                            if (ls.PreviousSegment != null)
                            {
                                //set reference for previous cell
                                if (ls.PreviousSegment.TOfCrossingSegment == 1.0)
                                    ls.PreviousSegment.CrossingSegment.SetEndCell(currentCell);
                                else
                                //segment ends in the middle of other segment
                                //should not reach this code
                                {
                                    throw new ApplicationException("Segment, start in the middle - should not occur");
                                    //ls.PreviousSegment.CrossingSegment.SetCrossingCell(ls, currentCell, EStartEnd.START,
                                    //    ls.PreviousSegment.TOfCrossingSegment);
                                }
                            }                            
                        }
                        else
                        {
                            firstCellIsBuilt = true;
                            //use already built cell
                            currentCell = ls.StartCell;
                            //set startcell reference
                            ls.SetStartCell(currentCell);                            
                            //process priorities
                            priority = ls.GetPriority(EStartEnd.START);
                        }

                        //set first cell
                        firstCell = currentCell;

                        //remember first cell to make neighbor cell interactions during constructing
                        //cells of next neighbor lane
                        if (setFirstCellOfRightNeighborLane && currentWay == EWay.Normal)
                        {
                            setFirstCellOfRightNeighborLane = false;
                            firstCellOfRightNeighborLane_ForNextSegment = currentCell;
                        }

                        //set start generator
                        if (ls.Generator != null)
                        {
                            ls.Generator.ConnectedCell = currentCell;
                        }
                    }
                    else if (cellLocationAndDirectionVector.Item3 != 1.0)
                    {
                        currentCell = ls.GetBuiltCell(cellLocationAndDirectionVector.Item1);
                        if (currentCell == null)
                            currentCell = new Cell(++cellIdCounter, cellLocationAndDirectionVector.Item1,
                                cellLocationAndDirectionVector.Item2, 18, cellLocationAndDirectionVector.Item3, ls.Way);
                        else
                            throw new ApplicationException("Should not find built cell");                        
                        priority = ls.SetCrossingCell(currentCell, cellLocationAndDirectionVector.Item1);
                    }
                    else
                    {
                        if (ls.EndCell == null)
                        {                            
                            //build new cell
                            currentCell = new Cell(++cellIdCounter, cellLocationAndDirectionVector.Item1,
                                cellLocationAndDirectionVector.Item2, 18, cellLocationAndDirectionVector.Item3, ls.Way);
                            ls.SetEndCell(currentCell);
                            if (ls.FollowingSegment != null)
                            {
                                if (ls.FollowingSegment.TOfCrossingSegment == 0.0)
                                    ls.FollowingSegment.CrossingSegment.SetStartCell(currentCell);
                                else
                                {
                                    //should not reach this code
                                    throw new ApplicationException("Segment, start in the middle - should not occur");
                                    //ls.FollowingSegment.CrossingSegment.SetCrossingCell(ls, currentCell, EStartEnd.END,
                                    //    ls.FollowingSegment.TOfCrossingSegment);
                                }
                            }
                        }
                        else
                        {
                            lastCellIsBuilt = true;
                            //use already built cell
                            currentCell = ls.EndCell;
                            //set references
                            ls.SetEndCell(currentCell);
                            //process priorities
                            priority = ls.GetPriority(EStartEnd.END);
                        }
                
                        //remember first cell to make neighbor cell interactions during constructing
                        //cells of next neighbor lane
                        if (setFirstCellOfRightNeighborLane && currentWay == EWay.Reversed)
                        {
                            setFirstCellOfRightNeighborLane = false;
                            firstCellOfRightNeighborLane_ForNextSegment = currentCell;
                        }
                    }

                    if (cellLocationAndDirectionVector.Item3 == 1.0 && !maxTurnSpeedWasComputed && currentCell != cellBefore)
                    {
                        //compute max turn speed when the segment is composed of 3 cells but not when it is composed of 2 cells                        
                        currentCell.ComputeMaxTurnSpeed(cellBefore.Location, cellBefore.DirectionVector);
                    }
                    if (cellLocationAndDirectionVector.Item3 != 0.0)
                    {    
                        ConnectFollowingCells(currentCell, cellBefore, priority);
                        if (cellBefore != firstCell && cellLocationAndDirectionVector.Item3 != 1.0)                            
                        {
                            //compute max turn speed. Do not compute max turn speed for first and last cell, because it is cell
                            //of other segment and there could be higher angle because user cannot do perfect 
                            //join of two segments
                            currentCell.ComputeMaxTurnSpeed(cellBefore.Location, cellBefore.DirectionVector);
                            maxTurnSpeedWasComputed = true;
                        }
                        if (firstSettingOfFollowingCell)
                        {                            
                            cellBefore.SetFollowingCell(currentCell, ls.rankListOfFollowingCells);
                            firstSettingOfFollowingCell = false;
                        }
                        else
                            cellBefore.SetFollowingCell(currentCell, null);
                    }
                    else if (priority != null)
                    {
                        currentCell.PreviousCell.Priority = priority;
                    }
                }

                //make neighbor cell interactions
                if (makeCellNeighborReferences)
                {
                    EWay wayOfLaneOnTheRight = EWay.Normal;
                    //cell associations are innverse for reversed lanes
                    EDirection direction1;
                    EDirection direction2;
                    if (currentWay == EWay.Normal)
                    {
                        direction1 = EDirection.FromRightToLeft;
                        direction2 = EDirection.FromLeftToRight;
                    }
                    else if (currentWay == EWay.Reversed && ls.LeftNeighborLane != null && 
                        ls.LeftNeighborLane.LeftNeighborLane == ls)
                    {
                        wayOfLaneOnTheRight = EWay.Reversed;
                        direction1 = EDirection.FromRightToLeft;
                        direction2 = EDirection.FromRightToLeft;
                    }
                    else if (currentWay == EWay.Reversed)
                    {
                        wayOfLaneOnTheRight = EWay.Reversed;
                        direction1 = EDirection.FromLeftToRight;
                        direction2 = EDirection.FromRightToLeft;
                    }
                    else
                    {
                        throw new ApplicationException("Bad association of lanes directions");
                    }
                    //make neighbor interactions from right lane cells to left lane cells
                    MakeCellNeighborReferences(firstCellOfRightNeighborLane_Current, firstCell, direction1, 
                        wayOfLaneOnTheRight, currentWay);
                    //make neighbor interactions from left lane cells to right lane cells
                    MakeCellNeighborReferences(firstCell, firstCellOfRightNeighborLane_Current, direction2,
                        currentWay, wayOfLaneOnTheRight);

                    if (!setFirstCellOfRightNeighborLane)
                        currentWay = EWay.Normal;
                }                

                if (ls.LeftNeighborLane != null && ls.LeftNeighborLane.LeftNeighborLane == ls)
                    currentWay = EWay.Reversed;

                previousLaneSegment = ls;
            }
        }
        
        private void MakeCellNeighborReferences(Cell settingLaneFirstCell, Cell comparingLaneFirstCell, EDirection eDirection,
            EWay wayOfSettingLane, EWay wayOfComparingLane)
        {
            Cell settingLaneCurrentCell = settingLaneFirstCell;
            Cell comparingLaneCurrentCell = comparingLaneFirstCell;
            bool firstConnectionWasRealized = false;
            while (!(settingLaneCurrentCell == null || (firstConnectionWasRealized && settingLaneCurrentCell.IsCrossroadCell)))
            {
                double previousDistance = double.MaxValue;
                double distance = (settingLaneCurrentCell.Location - comparingLaneCurrentCell.Location).Length;
                bool moveBackwards = true;
                while (distance < previousDistance)
                {
                    if (wayOfComparingLane == EWay.Normal ? 
                        comparingLaneCurrentCell.FollowingCellLastAdded == null || 
                        (firstConnectionWasRealized && comparingLaneCurrentCell.FollowingCellLastAdded.IsCrossroadCell) : 
                        comparingLaneCurrentCell.PreviousCellLastAdded == null ||
                        (firstConnectionWasRealized && comparingLaneCurrentCell.PreviousCellLastAdded.IsCrossroadCell))
                    {
                        moveBackwards = false;
                        break;
                    }
                    comparingLaneCurrentCell = wayOfComparingLane == EWay.Normal ?
                        comparingLaneCurrentCell.FollowingCellLastAdded :
                        comparingLaneCurrentCell.PreviousCellLastAdded;
                    previousDistance = distance;
                    distance = (settingLaneCurrentCell.Location - comparingLaneCurrentCell.Location).Length;
                }
                if (moveBackwards)
                {
                    if (wayOfComparingLane == EWay.Normal ?
                        comparingLaneCurrentCell.PreviousCellLastAdded != null && comparingLaneCurrentCell != comparingLaneFirstCell :
                        comparingLaneCurrentCell.FollowingCellLastAdded != null && comparingLaneCurrentCell != comparingLaneFirstCell)
                        //this condition should always occur. 
                        //Just not if comparing lane had only cell.
                        comparingLaneCurrentCell = wayOfComparingLane == EWay.Normal ?
                            comparingLaneCurrentCell.PreviousCellLastAdded :
                            comparingLaneCurrentCell.FollowingCellLastAdded;
                }
                
                switch (eDirection)
                {
                    case EDirection.FromRightToLeft:
                        if (settingLaneCurrentCell.CellsOnTheLeft.Count > 0)
                            settingLaneCurrentCell.AddCellOnTheLeft(comparingLaneCurrentCell);
                        else
                            settingLaneCurrentCell.CellOnTheLeft = comparingLaneCurrentCell; 
                        break;

                    case EDirection.FromLeftToRight:
                        if (settingLaneCurrentCell.CellsOnTheRight.Count > 0)
                            settingLaneCurrentCell.AddCellOnTheRight(comparingLaneCurrentCell);
                        else
                            settingLaneCurrentCell.CellOnTheRight = comparingLaneCurrentCell;
                        break;
                }

                settingLaneCurrentCell = wayOfSettingLane == EWay.Normal ?
                    settingLaneCurrentCell.FollowingCellLastAdded :
                    settingLaneCurrentCell.PreviousCellLastAdded;

                firstConnectionWasRealized = true;
            }
        }

        /// <summary>
        /// Stops simulation and generator timers
        /// </summary>
        public void StopTimers()
        {
            SimulationState = ESimulationState.Stopped;
            //set elapsed time from last tick
            elapsedTime = DateTime.Now - LastSimulationStepTime;
            //stop timer
            simulationStepTimer.Stop();
            statisticsTimer.Stop();
            //stop generator timers
            foreach (Generator generator in generators)
                generator.StopTimer();
        }

        internal void BuildRoad(List<Tuple<Point, Point?, Point?, Point>> roadPreviewPoints,
            int numberOfLanes1, int allowedSpeed, double maxDistance)
        {
            List<LaneSegment> laneSegments = new List<LaneSegment>(roadPreviewPoints.Count);
            int laneCounter = 0;
            foreach (var i in roadPreviewPoints)
            {
                //swap points if way direction is reversed
                EWay way = laneCounter++ < numberOfLanes1 ? EWay.Normal : EWay.Reversed;
                Tuple<Point, Point?, Point?, Point> points = way == EWay.Normal ?
                    i : new Tuple<Point, Point?, Point?, Point>(i.Item4, i.Item3, i.Item2, i.Item1);

                //create segments 
                LaneSegment newSegment;
                if (points.Item2 == null && points.Item3 == null)
                {
                    newSegment = new StraightLaneSegment() { StartPoint = points.Item1, EndPoint = points.Item4, Way = way };
                }
                else
                {
                    newSegment = new BezierLaneSegment()
                    {
                        StartPoint = points.Item1,
                        StartControlPoint = points.Item2.Value,
                        EndPoint = points.Item4,
                        EndControlPoint = points.Item3.Value,
                        Way = way,
                    };
                }
                if (roadPreviewPoints.Count > 1)
                {
                    if (laneCounter == 1)
                        newSegment.IsEdgeLaneSegmentOnTheRight = true;
                    else if (laneCounter == roadPreviewPoints.Count - 1)
                        newSegment.IsEdgeLaneSegmentOnTheLeft = true;
                }

                //create segment interactions
                try
                {
                    new InteractionMaker(newSegment, maxDistance, this.laneSegments.AsReadOnly());
                }
                catch (ApplicationException appEx)
                {
                    MessageBox.Show(appEx.Message, "Message", MessageBoxButton.OK, MessageBoxImage.Exclamation);
                    break;
                }

                //create neighbor interactions 
                if (laneSegments.Count > 0)
                {
                    if (laneSegments.Count < numberOfLanes1)
                    {
                        newSegment.RightNeighborLane = laneSegments[laneSegments.Count - 1];
                        laneSegments[laneSegments.Count - 1].LeftNeighborLane = newSegment;
                    }
                    else if (laneSegments.Count == numberOfLanes1)
                    {
                        newSegment.LeftNeighborLane = laneSegments[laneSegments.Count - 1];
                        laneSegments[laneSegments.Count - 1].LeftNeighborLane = newSegment;
                    }
                    else
                    {
                        newSegment.LeftNeighborLane = laneSegments[laneSegments.Count - 1];
                        laneSegments[laneSegments.Count - 1].RightNeighborLane = newSegment;
                    }
                }

                laneSegments.Add(newSegment);
            }
            this.laneSegments.AddRange(laneSegments);
        }

        
        public static void GetLaneSegmentPoints(List<Tuple<Point, Point?, Point?, Point>> laneSegmentPoints, Point start, Point? startControl, 
            Point? endControl, Point end, int numberOfLanes, double laneWidth)
        {
            laneSegmentPoints.Clear();
            if (!startControl.HasValue)
            {
                Vector directionVector = end - start;
                directionVector.Normalize();
                Vector directionPerpendicularCWVector = RotateVector90degreesCW(directionVector);
                Vector directionPerpendicularCCWVector = directionPerpendicularCWVector; directionPerpendicularCCWVector.Negate();
                Point laneStart = start + directionPerpendicularCWVector * laneWidth * ((numberOfLanes - 1) / 2.0);
                for (int i = 0; i < numberOfLanes; i++)
                {
                    Vector roadMiddleToLane = laneStart - start;
                    laneSegmentPoints.Add(new Tuple<Point, Point?, Point?, Point>(
                        laneStart, null, null, end + roadMiddleToLane));
                    laneStart += directionPerpendicularCCWVector * laneWidth;
                }
            }
            else
            {
                Vector startDirectionVector = BezierLaneSegment.ComputePoint(new Tuple<Point, Point, Point, Point>
                        (start, startControl.Value, endControl.Value, end), 0.000001) - start;
                Vector endDirectionVector = BezierLaneSegment.ComputePoint(new Tuple<Point, Point, Point, Point>
                        (start, startControl.Value, endControl.Value, end), 0.999999) - end;
                startDirectionVector.Normalize();
                endDirectionVector.Normalize();
                Vector startDirectionPerpendicularCWVector = RotateVector90degreesCW(startDirectionVector);
                Vector endDirectionPerpendicularCCWVector = RotateVector90degreesCCW(endDirectionVector);
                Vector startDirectionPerpendicularCCWVector = startDirectionPerpendicularCWVector; startDirectionPerpendicularCCWVector.Negate();
                Vector endDirectionPerpendicularCWVector = endDirectionPerpendicularCCWVector; endDirectionPerpendicularCWVector.Negate();
                Point startLaneStart = start + startDirectionPerpendicularCWVector * laneWidth * ((numberOfLanes - 1) / 2.0);
                Point endLaneStart = end + endDirectionPerpendicularCCWVector * laneWidth * ((numberOfLanes - 1) / 2.0);
                Vector startRoadMiddleToLane = startLaneStart - start;
                Vector endRoadMiddleToLane = endLaneStart - end;                    
                for (int i = 0; i < numberOfLanes; i++)
                {
                    laneSegmentPoints.Add(new Tuple<Point, Point?, Point?, Point>(
                        startLaneStart,
                        startControl + (startLaneStart - start),// * (2.0 / 3.0) + (endLaneStart - end) * (1.0 / 3.0),
                        endControl + (endLaneStart - end),// * (2.0 / 3.0) + (startLaneStart - start) * (1.0 / 3.0),
                        endLaneStart));
                    startLaneStart += startDirectionPerpendicularCCWVector * laneWidth;
                    endLaneStart += endDirectionPerpendicularCWVector * laneWidth;
                }
            }
        }

        private static Vector RotateVector90degreesCCW(Vector directionVector)
        {
            return new Vector(directionVector.X * Math.Cos(Math.PI / 2) - directionVector.Y * Math.Sin(Math.PI / 2),
                directionVector.X * Math.Sin(Math.PI / 2) + directionVector.Y * Math.Cos(Math.PI / 2));
        }

        private static Vector RotateVector90degreesCW(Vector directionVector)
        {
            return new Vector(directionVector.X * Math.Cos(-Math.PI / 2) - directionVector.Y * Math.Sin(-Math.PI / 2),
                directionVector.X * Math.Sin(-Math.PI / 2) + directionVector.Y * Math.Cos(-Math.PI / 2));
        }


        private void EditPointToConnectSegments(ref Point? point, ref Point? controlPoint, double maxDistance)
        {
            double minDistance = maxDistance;
            LaneSegment nearestLaneSegment = null;
            EStartEnd startEnd = EStartEnd.NONE;
            foreach (LaneSegment laneSegment in laneSegments)
            {
                if ((laneSegment.StartPoint - point.Value).Length <= minDistance)
                {
                    minDistance = (laneSegment.StartPoint - point.Value).Length;
                    nearestLaneSegment = laneSegment;
                    startEnd = EStartEnd.START;
                }
                if ((laneSegment.EndPoint - point.Value).Length <= minDistance)
                {
                    minDistance = (laneSegment.EndPoint - point.Value).Length;
                    nearestLaneSegment = laneSegment;
                    startEnd = EStartEnd.END;
                }
            }
            if (nearestLaneSegment != null)
            {
                if (nearestLaneSegment is StraightLaneSegment)
                {
                    if (startEnd == EStartEnd.START)
                    {
                        point = nearestLaneSegment.StartPoint;
                        
                    }
                }
                else if (nearestLaneSegment is BezierLaneSegment)
                {
                }
            }
        }            

        ///// <summary>
        ///// Builds road
        ///// </summary>
        ///// <param name="point">The first end</param>
        ///// <param name="secondPoint">The second end</param>
        //public void BuildRoad(Point firstPoint, Point secondPoint, int numberOfLanes1, int numberOfLanes2, int allowedSpeed, double maxDistance)
        //{
        //    if ((firstPoint - secondPoint).Length < 5)
        //        return;
        //    //check if edge road is not near to first and second point
        //    Cell nearestCellFirstPoint = null;
        //    Cell nearestCellSecondPoint = null;
        //    double minDistanceFirst = maxDistance;
        //    double minDistanceSecond = maxDistance;
        //    foreach (var i in roadEdgesLocations)
        //    {
        //        Point roadEdgeLocation = i.Key.Location + (i.Value.Location - i.Key.Location) / 2;
        //        if ((roadEdgeLocation - firstPoint).Length < minDistanceFirst)
        //        {
        //            if (i.Key != nearestCellSecondPoint)
        //            {
        //                //found near edge road
        //                minDistanceFirst = (roadEdgeLocation - firstPoint).Length;
        //                nearestCellFirstPoint = i.Key;
        //            }
        //        }
        //        if ((roadEdgeLocation - secondPoint).Length < minDistanceSecond)
        //        {
        //            if (i.Key != nearestCellFirstPoint)
        //            {
        //                //found near edge road
        //                minDistanceSecond = (roadEdgeLocation - secondPoint).Length;
        //                nearestCellSecondPoint = i.Key;
        //            }
        //        }
        //    }

        //    //check if crossroad is not near to first or second point
        //    Crossroad nearestCrossroadFirstPoint = null;
        //    Crossroad nearestCrossroadSecondPoint = null;
        //    foreach (var i in crossroadLocations)
        //    {
        //        if ((i.Location - firstPoint).Length < minDistanceFirst && i.Location != secondPoint)
        //        {
        //            //found near crossroad
        //            minDistanceFirst = (i.Location - firstPoint).Length;
        //            nearestCrossroadFirstPoint = i;
        //            nearestCellFirstPoint = null;
        //            firstPoint = i.Location;
        //        }
        //        if ((i.Location - secondPoint).Length < minDistanceSecond && i.Location != firstPoint)
        //        {
        //            //found near crossroad
        //            minDistanceSecond = (i.Location - secondPoint).Length;
        //            nearestCrossroadSecondPoint = i;
        //            nearestCellSecondPoint = null;
        //            secondPoint = i.Location;
        //        }
        //    }
            
        //    //get cell column of near road
        //    List<Cell> cellColumnExistingRoad1 = null;
        //    List<Cell> cellColumnExistingRoad2 = null;
        //    if (nearestCellFirstPoint != null)
        //    {
        //        try
        //        {
        //            ServeRoadNearPoint(out cellColumnExistingRoad1, ref firstPoint, 
        //                numberOfLanes1, numberOfLanes2, nearestCellFirstPoint, true);
        //        }
        //        catch (ArgumentException argEx)
        //        {
        //            MessageBox.Show(argEx.Message, "Error", MessageBoxButton.OK, MessageBoxImage.Error);
        //            return;
        //        }
                
        //    }
        //    if (nearestCellSecondPoint != null)
        //    {
        //        try
        //        {
        //            ServeRoadNearPoint(out cellColumnExistingRoad2, ref secondPoint,
        //                numberOfLanes2, numberOfLanes1, nearestCellSecondPoint, false);
        //        }
        //        catch (ArgumentException argEx)
        //        {
        //            MessageBox.Show(argEx.Message, "Error", MessageBoxButton.OK, MessageBoxImage.Error);
        //            return;
        //        }
        //    }

        //    //BUILD ROAD
        //    //compute deltax,deltay to move build road column by column
        //    Vector vector = ComputeVector(secondPoint - firstPoint, cellLength);
        //    //vectorLength from road-axis. Set it to vectorLength one of the edges to road-axis
        //    double distance = (numberOfLanes1 + numberOfLanes2 - 1) * World.LaneWidth / 2;
        //    //initialize roadmidpoint which will be used to build road
        //    Point roadMidpoint = firstPoint;
        //    //initialize two columns of cells
        //    List<Cell> column1 = new List<Cell>();
        //    List<Cell> column2 = new List<Cell>();
        //    List<Cell> firstColumn = null;
        //    List<Cell> columnToConnect = new List<Cell>();
        //    //build road
        //    while (IsPointInRectangle(firstPoint, secondPoint, roadMidpoint))
        //    {
        //        //move new column to old
        //        List<Cell> pom = column1;
        //        column1 = column2;
        //        column2 = pom;
        //        column2.Clear();
        //        //connect with previous road
        //        if (nearestCellFirstPoint != null && column1.Count != 0 && columnToConnect.Count == 0)
        //        {
        //            columnToConnect = new List<Cell>(column1);
                    
        //        }
        //        //get first column of road to connect it to crossroad
        //        if (column1.Count != 0 && firstColumn == null)
        //        {
        //            firstColumn = new List<Cell>(column1);
        //        }
        //        //lists of cells, on the right and left side of the road axis
        //        List<Cell> rightCells = new List<Cell>();
        //        List<Cell> leftCells = new List<Cell>();
        //        //initializes helper variable holding number of lanes
        //        int numberOfLanes1ToBuild = numberOfLanes1;
        //        int numberOfLanes2ToBuild = numberOfLanes2;
        //        double currentDistance = distance;
        //        //build column
        //        do
        //        {
        //            //compute deltax and deltay for creating cells in columns
        //            Vector vectorWidth = ComputeVector(secondPoint - firstPoint, currentDistance);
        //            //compute cellLocation of new right cellColumnPair
        //            Point cellPoint = roadMidpoint;
        //            cellPoint.X += vectorWidth.Y;
        //            cellPoint.Y -= vectorWidth.X;
        //            //determine direction
        //            Vector directionVector = numberOfLanes1ToBuild > 0 ? secondPoint - firstPoint : firstPoint - secondPoint;
        //            //build cellColumnPair, add to new cells column 
        //            Cell newCell = new Cell(cellPoint, directionVector, allowedSpeed);
        //            column2.Add(newCell);
        //            if (numberOfLanes1ToBuild > 0)                    
        //                rightCells.Add(newCell);
        //            else
        //                leftCells.Add(newCell);
        //            //connect with cellColumnPair in the previous column
        //            if (column1.Count > 0)
        //            {
        //                if (numberOfLanes1ToBuild > 0)
        //                {
        //                    column1[column2.Count - 1].FollowingCell = column2[column2.Count - 1];
        //                    column2[column2.Count - 1].PreviousCell = column1[column2.Count - 1];
        //                }
        //                else
        //                {
        //                    column1[column2.Count - 1].PreviousCell = column2[column2.Count - 1];
        //                    column2[column2.Count - 1].FollowingCell = column1[column2.Count - 1];
        //                }
        //            }
        //            //decrement number of lanes of directions
        //            if (numberOfLanes1ToBuild > 0)
        //                numberOfLanes1ToBuild--;
        //            else
        //                numberOfLanes2ToBuild--;
                     
        //            //decrement current vectorLength
        //            currentDistance -= LaneWidth;
        //        } while (currentDistance >= -distance);
        //        //connect cells in the new created column
        //        //connect cells on the right side
        //        for (int i = 0; i < rightCells.Count - 1; i++)
        //        {
        //            rightCells[i].CellOnTheLeft = rightCells[i + 1];
        //            rightCells[i + 1].CellOnTheRight = rightCells[i];
        //        }
        //        //connect cells on the border - roadaxis
        //        if (leftCells.Count > 0)
        //        {
        //            leftCells[0].CellOnTheLeft = rightCells[rightCells.Count - 1];
        //            rightCells[rightCells.Count - 1].CellOnTheLeft = leftCells[0];
        //        }
        //        //connect cells on the left side
        //        for (int i = 0; i < leftCells.Count - 1; i++)
        //        {
        //            leftCells[i].CellOnTheRight = leftCells[i + 1];
        //            leftCells[i + 1].CellOnTheLeft = leftCells[i];
        //        }
        //        //add to global cells list-dictionary
        //        Point middlePointRegion = new Point();
        //        if (roadMidpoint.X >= 0)
        //            //if coordinate is bigger than zero truncate it
        //            middlePointRegion.X = Math.Truncate(roadMidpoint.X
        //                / World.MatrixCellSize) * World.MatrixCellSize;
        //        else
        //            //if coordinate is less than zero subtract MatrixCellSize before truncating
        //            //to create correct net on the left-negative part of World
        //            middlePointRegion.X = Math.Truncate((roadMidpoint.X - World.MatrixCellSize)
        //                / World.MatrixCellSize) * World.MatrixCellSize;
        //        if (roadMidpoint.Y >= 0)
        //            //if coordinate is bigger than zero truncate it
        //            middlePointRegion.Y = Math.Truncate(roadMidpoint.Y
        //                / World.MatrixCellSize) * World.MatrixCellSize;
        //        else
        //            //if coordinate is less than zero subtract MatrixCellSize before truncating
        //            //to create correct net on the down-negative part of World
        //            middlePointRegion.Y = Math.Truncate((roadMidpoint.Y - World.MatrixCellSize)
        //                / World.MatrixCellSize) * World.MatrixCellSize;
        //        //if first cellColumnPair in the region initialize list of cells of that region
        //        if (!cells.ContainsKey(middlePointRegion))
        //            cells.Add(middlePointRegion, new List<KeyValuePair<Cell, Cell>>());
        //        cells[middlePointRegion].Add(new KeyValuePair<Cell, Cell>(column2[0], column2[column2.Count - 1]));
        //        //move point
        //        roadMidpoint += vector;
        //    }
        //    //connect with following road
        //    if (nearestCellSecondPoint != null)
        //    {
        //        ConnectWithOtherRoad(column2, cellColumnExistingRoad2, false);
        //    }
        //    if (nearestCellFirstPoint != null)
        //    {
        //        ConnectWithOtherRoad(columnToConnect, cellColumnExistingRoad1, true);
        //    }
            
        //    //connect to crossroad
        //    if (nearestCrossroadFirstPoint != null)
        //    {
        //        ConnectToCrossroad(nearestCrossroadFirstPoint, firstColumn);
        //    }
        //    if (nearestCrossroadSecondPoint != null)
        //    {
        //        ConnectToCrossroad(nearestCrossroadSecondPoint, column2);
        //    }
        //    //add road edges
        //    if (nearestCellFirstPoint == null && nearestCrossroadFirstPoint == null)
        //    {
        //        roadEdgesLocations.Add(new KeyValuePair<Cell, Cell>(firstColumn[0], GetOtherSideEdgeCell(firstColumn[0])));
        //    }
        //    if (nearestCellSecondPoint == null && nearestCrossroadSecondPoint == null)
        //    {
        //        roadEdgesLocations.Add(new KeyValuePair<Cell, Cell>(column2[0], GetOtherSideEdgeCell(column2[0])));
        //    }
        //}

        private void ConnectToCrossroad(Crossroad nearestCrossroad, List<Cell> roadEdgeCells)
        {
            nearestCrossroad.AddRoad(
                    new KeyValuePair<Cell, Cell>(roadEdgeCells[0], GetOtherSideEdgeCell(roadEdgeCells[0])));

        }

        internal static Cell CellForwards(Cell currentCell, DirectionPoint.Way.Instructions currentInstructions, 
            out bool crossroadCrossed)
        {
            crossroadCrossed = false;
            if (currentCell.FollowingCell != null && currentCell.FollowingCell.PreviousCells.Count > 1)
            {
                crossroadCrossed = true;
                return currentCell.GetConnectedCell(currentInstructions);
            }
            while (currentCell.FollowingCell == null && currentCell.CellOnTheRight != null)
            {
                currentCell = currentCell.CellOnTheRight;
            }
            return currentCell.FollowingCell;
        }

        public static Cell CellForwards(Cell currentCell, Vector? currentInstructions,
            out bool crossroadCrossed)
        {
            crossroadCrossed = false;
            if (currentCell.FollowingCell != null && currentCell.FollowingCell.PreviousCells.Count > 1)
            {
                crossroadCrossed = true;
                return currentCell.GetConnectedCell(currentInstructions);
            }
            while (currentCell.FollowingCell == null && currentCell.CellOnTheRight != null)
            {
                currentCell = currentCell.CellOnTheRight;
            }
            return currentCell.FollowingCell;
        }

        public static Cell CellBackwards(Cell currentCell)
        {
            while (currentCell.PreviousCell == null && currentCell.CellOnTheRight != null)
                currentCell = currentCell.CellOnTheRight;
            return currentCell.PreviousCell;
        }

        public static ReadOnlyCollection<Cell> CellsBackwards(Cell currentCell)
        {
            while (currentCell.PreviousCell == null && currentCell.CellOnTheRight != null)
                currentCell = currentCell.CellOnTheRight;
            return currentCell.PreviousCells;
        }

        public static Cell GetEdgeCell(Cell cell)
        {
            while (cell.CellOnTheRight != null)
            {
                cell = cell.CellOnTheRight;
            }
            return cell;
        }

        private int GetNumberOfLanes(Cell cell)
        {
            LaneMover laneMover = new LaneMover(cell);
            int numberOfLanes = 0;
            foreach (Cell i in laneMover.GetNext())
                numberOfLanes++;
            return numberOfLanes;
        }
        

        /// <summary>
        /// Serve near point to get inner edge cell and last column of existing road
        /// </summary>
        /// <param name="innerEdgeCell"></param>
        /// <param name="cellColumnExistingRoad"></param>
        /// <param name="point"></param>
        /// <param name="numberOfLanes1"></param>
        /// <param name="numberOfLanes2"></param>
        /// <param name="nearestCell"></param>
        /// <param name="beginningTrueEndingFalse"></param>
        private void ServeRoadNearPoint(out List<Cell> cellColumnExistingRoad, 
            ref Point point, int numberOfLanes1, int numberOfLanes2, Cell nearestCell, bool beginningTrueEndingFalse)
        {
            List<bool> laneStructureTrueForwardFalseBackward;
            Cell innerEdgeCell = GetInnerEdgeCell(nearestCell, point);
            laneStructureTrueForwardFalseBackward = GetLaneStructure(innerEdgeCell);
            CheckTwoWayFromOneWay(numberOfLanes1, numberOfLanes2, laneStructureTrueForwardFalseBackward);
            point = GetRoadMidpoint(innerEdgeCell);
            if (beginningTrueEndingFalse && !(innerEdgeCell.FollowingCell == null && innerEdgeCell.CellOnTheRight == null))
                innerEdgeCell = GetOtherSideEdgeCell(innerEdgeCell);
            else if (!beginningTrueEndingFalse && !(innerEdgeCell.PreviousCell == null && innerEdgeCell.CellOnTheRight == null))
                innerEdgeCell = GetOtherSideEdgeCell(innerEdgeCell);
            LaneMover laneMover = new LaneMover(innerEdgeCell);
            cellColumnExistingRoad = new List<Cell>();
            foreach (Cell i in laneMover.GetNext())
                cellColumnExistingRoad.Add(i);
        }

        /// <summary>
        /// Returns road midpoint
        /// </summary>
        /// <param name="edgeCell">Cell on the edge</param>
        /// <returns>Road midpoint</returns>
        private Point GetRoadMidpoint(Cell innerCell)
        {
            LaneMover laneMover = new LaneMover(innerCell);
            Point edgeLocation1 = innerCell.Location;
            Point edgeLocation2 = new Point();
            foreach (Cell i in laneMover.GetNext())
                edgeLocation2 = i.Location;
            return edgeLocation1 + (edgeLocation2 - edgeLocation1) / 2;
        }

        private void CheckTwoWayFromOneWay(int numberOfLanes1, int numberOfLanes2, List<bool> laneStructureTrueForwardFalseBackward)
        {
            //check if it is needed to change width of road (number of lanes)
            int currentNumberOfLanes1 = laneStructureTrueForwardFalseBackward.Count(j => j == true);
            int currentNumberOfLanes2 = laneStructureTrueForwardFalseBackward.Count(j => j == false);
            if (currentNumberOfLanes1 + currentNumberOfLanes2 != numberOfLanes1 + numberOfLanes2)
            {
                if ((numberOfLanes1 == 0 && currentNumberOfLanes1 != 0) || (numberOfLanes1 != 0 && currentNumberOfLanes1 == 0) ||
                    (numberOfLanes2 == 0 && currentNumberOfLanes2 != 0) || (numberOfLanes2 != 0 && currentNumberOfLanes2 == 0))
                    throw new ArgumentException("Cannot make two-way road from one-way road");
            }
        }

        /// <summary>
        /// Get list of direction vectors of all lanes
        /// </summary>
        /// <param name="innerEdgeCell">Edge cell</param>
        /// <returns>List of direction vectors</returns>
        private List<bool> GetLaneStructure(Cell innerEdgeCell)
        {
            List<bool> directionTrueForwardFalseBackward = new List<bool>();
            LaneMover laneMover = new LaneMover(innerEdgeCell);
            foreach (Cell i in laneMover.GetNext())
            {
                if (i.FollowingCell == null)
                    directionTrueForwardFalseBackward.Add(true);
                else
                    directionTrueForwardFalseBackward.Add(false);
            }
            return directionTrueForwardFalseBackward;
        }

        /// <summary>
        /// Get rotated normalized vector
        /// </summary>
        /// <param name="vector">Base vector</param>
        /// <param name="degrees">Size of rotation</param>
        /// <returns></returns>
        protected Vector GetRotatedVector(Vector vector, double degrees)
        {
            //convert degress to radians
            double radians = degrees * Math.PI / 180.0;
            //compute new rotated vector
            Vector rotatedVector = new Vector();
            rotatedVector.X = vector.X * Math.Cos(radians) - vector.Y * Math.Sin(radians);
            rotatedVector.Y = vector.X * Math.Sin(radians) + vector.Y * Math.Cos(radians);
            rotatedVector.Normalize();
            return rotatedVector;
        }

        /// <summary>
        /// Get cell which is on the edge of the existing road and which is on the inner side of turn
        /// </summary>
        /// <param name="nearestCell">Cell near the road edge</param>
        /// <returns>Edge cell on the inner side of turn</returns>
        private Cell GetInnerEdgeCell(Cell nearestCell, Point otherPoint)
        {
            //get edge cell
            Cell cellInFront = nearestCell;
            Cell cellInBack = nearestCell;
            Cell nextFrontCell;
            Cell nextBackCell;
            Cell edgeCell = null;
            while (true)
            {
                nextFrontCell = cellInFront.FollowingCell;
                nextBackCell = cellInBack.PreviousCell;
                if (nextFrontCell == null)
                {
                    edgeCell = cellInFront;
                    break;
                }
                if (nextBackCell == null)
                {
                    edgeCell = cellInBack;
                    break;
                }
                cellInFront = nextFrontCell;
                cellInBack = nextBackCell;
            } 
            //determine which of edge cells will be inner cell
            double distanceToOtherPoint = (edgeCell.Location - otherPoint).Length;
            Cell otherSideEdgeCell = GetOtherSideEdgeCell(edgeCell);
            if (distanceToOtherPoint < (otherSideEdgeCell.Location - otherPoint).Length)
                return edgeCell;
            return otherSideEdgeCell;
        }

        /// <summary>
        /// Search in nearest cells for keyvaluepair
        /// </summary>
        /// <param name="nearestCells">List of nearest cells</param>
        /// <param name="keyValuePair">Keyvaluepair to find</param>
        /// <returns>Result of search</returns>
        private bool ContainsColumn(List<KeyValuePair<KeyValuePair<Cell, Cell>, KeyValuePair<Cell, Cell>>> nearestCells, KeyValuePair<Cell, Cell> keyValuePair)
        {
            bool found = false;
            foreach (var i in nearestCells) if (i.Key.Equals(keyValuePair) || i.Value.Equals(keyValuePair))
            {
                found = true;
                break;
            }
            return found;
        }

        /// <summary>
        /// Gets other side edge cell
        /// </summary>
        /// <param name="cell">Edge cell</param>
        /// <returns>Other side edge cell</returns>
        public static Cell GetOtherSideEdgeCell(Cell cell)
        {
            LaneMover laneMover = new LaneMover(cell);
            foreach (Cell i in laneMover.GetNext())
                cell = i;
            return cell;
        }

        /// <summary>
        /// Computes vector of required length
        /// </summary>
        /// <param name="vector">Vector to use to compute new vector</param>
        /// <param name="vectorLength">Required length of new vector</param>
        /// <returns>Vector of required length</returns>
        public Vector ComputeVector(Vector vector, double vectorLength)
        {
            return vector * vectorLength / vector.Length;
        }


        /// <summary>
        /// Adds generator to World's list of generators
        /// </summary>
        /// <param name="generator">Generator to add</param>
        public void AddGenerator(Generator generator)
        {
            generators.Add(generator);
        }
        
        private void statisticsTimer_Tick(object sender, EventArgs e)
        {
            //foreach (Cell c in statisticsCells)
            //    c.SaveAndShowStatistics();
        }

        public void SetStatisticsCell(Cell nearCell)
        {
            //statisticsCells.Add(nearCell);
            nearCell.SetAsStatistics();
        }

        public List<Cell> GetStatisticsCell()
        {
            throw new NotImplementedException();
            //return statisticsCells;
        }

        public void CloseStatisticsFiles()
        {            
            //foreach (Cell c in statisticsCells)
            //    c.CloseStatisticsFile();
        }

        internal void ClearAll()
        {
            StopTimers();
            generators.Clear();
            laneSegments.Clear();
            carsNewPositions.Clear();
            carsOldPositions.Clear();
            carsToRemove.Clear();
            elapsedTime = null;
            CloseStatisticsFiles();
            //statisticsCells.Clear();

            //ZedGraph.
        }

        /// <summary>
        /// Restore timer and car instances after loading new topology
        /// </summary>
        public void RestoreWorld()
        {
            simulationStepTimer = new DispatcherTimer(DispatcherPriority.Normal);
            simulationStepTimer.Interval = TimeSpan.FromSeconds(SimulationStepTime);
            simulationStepTimer.Tick += new EventHandler(simulationStepTimer_Tick);
            statisticsTimer = new DispatcherTimer(DispatcherPriority.Normal);
            statisticsTimer.Interval = TimeSpan.FromSeconds(10);
            statisticsTimer.Tick += new EventHandler(statisticsTimer_Tick);
            carsToRemove = new List<Car>();
            carsNewPositions = new Dictionary<Car, Cell>();
            carsOldPositions = new Dictionary<Car, Cell>();
            foreach (var i in laneSegments)
            {
                if (i.Generator != null)
                    i.Generator.RestoreGenerator();
                i.Restore(); 
            }
        }

        [field: NonSerialized]
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Set property and notify property changed
        /// </summary>
        /// <typeparam name="T">Generic variable</typeparam>
        /// <param name="field">Backing field reference</param>
        /// <param name="newValue">New value to set</param>
        /// <param name="propertyName">Property name</param>
        public void Set<T>(ref T field, T newValue, string propertyName)
        {
            if (!field.Equals(newValue))
            {
                field = newValue;
                if (PropertyChanged != null)
                    PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        private void UpdateTimerIntervals(double oldSimulationSpeed, double newSimulationSpeed)
        {
            throw new NotImplementedException();
        }

        //public IEnumerable<Cell> GetStatisticsCells { get { return statisticsCells.Select(i => i); } }
       
        internal ReadOnlyCollection<LaneSegment> LaneSegments { get { return laneSegments.AsReadOnly(); } }

        internal void RemoveLastLaneSegment()
        {
            if (laneSegments.Count > 0)
            {
                LaneSegment laneSegmentToRemove = laneSegments[laneSegments.Count - 1];
                laneSegmentToRemove.RemoveNeighborsAndInteractions();               
                laneSegments.RemoveAt(laneSegments.Count - 1);
            }
        }

        internal void ResetLaneSegments()
        {
            foreach (var ls in laneSegments)
                ls.ResetSegment();
        }

        internal void ClearGenerators()
        {
            generators.Clear();
        }

        internal void RemoveLaneSegment(Point worldPoint, double maxDistance)
        {
            LaneSegment segmentToRemove;
            new InteractionEdgeToLaneMaker(worldPoint, maxDistance, LaneSegments, out segmentToRemove);                    
            if (segmentToRemove != null)
            {
                segmentToRemove.RemoveNeighborsAndInteractions();
                laneSegments.Remove(segmentToRemove);
            }
        }

        internal void RemoveGenerator(Point worldPoint, double maxDistance)
        {            
            new InteractionEdgeToEdgeMaker(worldPoint, maxDistance, LaneSegments);                   
        }

        internal void RemoveDirectionCell(Point worldPoint, double maxDistance)
        {
            new InteractionEdgeToLaneMaker(worldPoint, maxDistance, LaneSegments);
        }

        internal static Cell CellForwards(Cell currentPositionFrontCell, List<EInstruction> currentInstructions, out bool crossroadCrossed)
        {
            throw new NotImplementedException();
        }
    }
}


