﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Numerics;
using System.Text;

namespace TrafficMicroSimulator
{
    static class RootComputer
    {
        static internal Complex[] ComputeRoots(double[] a)
        {
            //http://ecourses.vtu.ac.in/nptel/courses/Webcourse-contents/IIT-KANPUR/Numerical%20Analysis/numerical-analysis/Rathish-kumar/ratish-1/f3node9.html
            if (a.Length < 2)
                throw new ApplicationException("Computing root of polynomial less than 2. degree");
            if (a.Length == 2)
                return new Complex[] { new Complex(-a[0] / a[1], 0) };
            if (a.Length == 3)
                return ComputeRoots(a[0], a[1], a[2]);

            double epsilon;            
            double[] b = new double[a.Length];
            double[] c = new double[a.Length];
            double r = a[a.Length - 2] / a[a.Length - 1];
            double s = a[a.Length - 3] / a[a.Length - 1];
            ulong antiLoopCounter = 0;
            do
            {
                b[a.Length - 1] = a[a.Length - 1];
                b[a.Length - 2] = a[a.Length - 2] + r * b[a.Length - 1];
                for (int i = a.Length - 3; i >= 0; i--)
                    b[i] = a[i] + r * b[i + 1] + s * b[i + 2];
                c[a.Length - 1] = b[a.Length - 1];
                c[a.Length - 2] = b[a.Length - 2] + r * c[a.Length - 1];
                for (int i = a.Length - 3; i > 0; i--)
                    c[i] = b[i] + r * c[i + 1] + s * c[i + 2];

                double delta_r = (-b[0] * c[3] + c[2] * b[1]) / (c[1] * c[3] - c[2] * c[2]);
                double delta_s = (-b[1] - c[2] * delta_r) / c[3];

                r += delta_r;
                s += delta_s;

                double eps_r = Math.Abs(delta_r / r) * 100;
                double eps_s = Math.Abs(delta_s / s) * 100;
                epsilon = Math.Max(eps_r, eps_s); //epsilon = Math.Max(eps_r, eps_r);
            } while (epsilon > 0.000001 && ++antiLoopCounter < 20000000U);

            double[] d = new double[] { -s, -r, 1 };
            double[] q = PolynomDivision(a, d);

            return ComputeRoots(-s, -r, 1).Concat<Complex>(ComputeRoots(q)).ToArray<Complex>();
        }

        static private double[] PolynomDivision(double[] a, double[] d)
        {
            //http://en.wikipedia.org/wiki/Polynomial_long_division#Pseudo-code
            List<double> q = new List<double>(a.Length);
            List<double> n = a.ToList<double>();
            while (n.Count > 0 && n.Count >= d.Length)
            {
                //t ← lead(r)/lead(d)
                double t = n[n.Count - 1] / d[d.Length - 1];
                //q + t
                int t_degree = n.Count - d.Length;
                for (int i = q.Count; i <= t_degree; i++)
                    q.Add(0);
                q[t_degree] += t;
                //n = n - (t * d);
                for (int i = d.Length - 1; i >= 0; i--)
                    n[i + t_degree] -= t * d[i];
                n.RemoveAt(n.Count - 1);
            }
            return q.ToArray<double>();
        }

        static private Complex[] ComputeRoots(double a0, double a1, double a2)
        {
            Complex D = Complex.Sqrt(new Complex(a1 * a1 - 4 * a2 * a0, 0));
            return new Complex[] 
            {
                (new Complex(-a1, 0) + D) / (2 * a2),
                (new Complex(-a1, 0) - D) / (2 * a2),
            };
        }
    }
}
