﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Media;
using System.Windows.Threading;
using System.Windows;
using System.ComponentModel;
using System.Windows.Controls;
using System.Collections.ObjectModel;
using System.Windows.Shapes;
using System.Windows.Media.Imaging;
using System.Diagnostics;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Renders World
    /// </summary>
    public class Renderer : INotifyPropertyChanged
    {
        /// <summary>
        /// Instance of World which is rendering
        /// </summary>
        protected World World { get; set; }

        /// <summary>
        /// Instance of canvas where is rendering World on
        /// </summary>
        protected Canvas elementToRenderOn;

        /// <summary>
        /// After every tick is invalidated simulation board
        /// </summary>
        protected DispatcherTimer renderingTimer;
        
        /// <summary>
        /// Number of renderings during one second
        /// </summary>
        protected int frameRate = 100;

        /// <summary>
        /// Time length of one frame
        /// </summary>
        protected double frameDuration { get { return 1000 / frameRate; } }

        /// <summary>
        /// Midpoint of view on the World. 
        /// </summary>
        protected Point viewMidpoint = new Point(0, 0);

        /// <summary>
        /// View scale in metres per logical point
        /// </summary>
        protected double scale = 1.0;

        /// <summary>
        /// View scale in logical points per meter
        /// </summary>
        public double Scale
        {
            get { return scale; }
            set { Set<double>(ref scale, value, "Scale"); }
        }

        List<Tuple<Point, Point?, Point?, Point>> linePoints = new List<Tuple<Point, Point?, Point?, Point>>();

        /// <summary>
        /// Initialize renderer
        /// </summary>
        /// <param name="World">World to render</param>
        /// <param name="elementToRenderOn">Canvas to render on</param>
        public Renderer(World world, Canvas elementToRenderOn, List<Tuple<Point, Point?, Point?, Point>> roadPreviewPoints)
        {
            //set World and canvas instance
            this.World = world;
            this.elementToRenderOn = elementToRenderOn;
            this.linePoints = roadPreviewPoints;
            //initialize timer
            renderingTimer = new DispatcherTimer(DispatcherPriority.Render);
            renderingTimer.Interval = TimeSpan.FromMilliseconds(1000 / frameRate);
            renderingTimer.Tick += new EventHandler(renderingTimer_Tick);
        }

        /// <summary>
        /// Redraw background with roads, generators
        /// </summary>
        protected void RedrawTopology()
        {                        
            //initialize drawing visual and drawing context            
            DrawingVisual dv = new DrawingVisual();
            using (DrawingContext dc = dv.RenderOpen())
            {                
                Pen roadPen = new Pen(Brushes.Black, 1);
                roadPen.Freeze();
                //draw roads, generators and crossroads
                foreach (LaneSegment laneSegment in World.LaneSegments)
                {
                    if (laneSegment is BezierLaneSegment)
                    {                       
                        dc.DrawGeometry(null, roadPen, GetBezierCurveGeometry(laneSegment as BezierLaneSegment));                     
                    }
                    else if (laneSegment is StraightLaneSegment)
                    {
                        dc.DrawLine(roadPen, 
                            TransformPointFromWorldToCanvasCoordinates(laneSegment.StartPoint), 
                            TransformPointFromWorldToCanvasCoordinates(laneSegment.EndPoint));
                    }
                    if (laneSegment.Generator != null)
                    {
                        dc.DrawEllipse(Brushes.Blue, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.StartPoint), 3, 3);
                    }
                    if (laneSegment.PreviousSegment != null)
                    {
                        dc.DrawEllipse(Brushes.Orange, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.StartPoint), 3, 3);
                        //Console.WriteLine("LaneSegment [" + laneSegment.StartPoint + "|" + laneSegment.EndPoint + "] has start continuation " +
                        //        "as LaneSegment [" + laneSegment.PreviousSegment.StartPoint + "|" + laneSegment.PreviousSegment.EndPoint + "]");                       
                    }                    
                    if (laneSegment.FollowingSegment != null)
                    {
                        dc.DrawEllipse(Brushes.Orange, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.EndPoint), 3, 3);
                        //Console.WriteLine("LaneSegment [" + laneSegment.StartPoint + "|" + laneSegment.EndPoint + "] has end continuation " +
                        //        "as LaneSegment [" + laneSegment.FollowingSegment.StartPoint + "|" + laneSegment.FollowingSegment.EndPoint + "]");
                    }                    
                    foreach (var i in laneSegment.CrossingSegments)
                    {
                        if (i.TOfCrossingSegment == 0.0 || i.TOfCrossingSegment == 1.0)
                            dc.DrawEllipse(Brushes.Orange, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(i.TOfMyCrossing)), 3, 3);
                        else
                            dc.DrawEllipse(Brushes.Red, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(i.TOfMyCrossing)), 3, 3);
                        if (i.Priority == EPriority.High)
                            DrawMainRoadMarks(dc, laneSegment, i.TOfMyCrossing);                            
                        else if (i.Priority == EPriority.Low)
                            DrawMainRoadMarks(dc, i.CrossingSegment, i.TOfCrossingSegment, laneSegment, i.TOfMyCrossing);                                                
                        //Console.WriteLine("LaneSegment [" + laneSegment.StartPoint + "|" + laneSegment.EndPoint + "] is crossed by " +
                        //        "LaneSegment [" + i.CrossingSegment.StartPoint + "|" + i.CrossingSegment.EndPoint + "] in [" + i.TOfMyCrossing + "]");
                    }
                    foreach (var i in laneSegment.DirectionPoints)
                        dc.DrawEllipse(Brushes.Green, null, TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(i.Item1)), 3, 3);
                    //Console.WriteLine("__________________________");
                }
                //Console.WriteLine("#######################");
            
                bool draw = false;
                List<Cell> drawn = new List<Cell>();
                foreach (Generator generator in World.GetGenerators())
                {
                    Point generatorLocation = generator.ConnectedSegment.Generator == generator ?
                        generator.ConnectedSegment.StartPoint : generator.ConnectedSegment.EndPoint;
                    dc.DrawRectangle(null, new Pen(Brushes.Green, 4),
                        new Rect(TransformPointFromWorldToCanvasCoordinates(generatorLocation),
                            TransformPointFromWorldToCanvasCoordinates(generatorLocation)));
                    if (draw)
                    {
                        //draw = false;
                        Cell cell = generator.ConnectedCell;                        
                        Queue<Cell> q = new Queue<Cell>();                        
                        while (cell != null || q.Count != 0)
                        {
                            while ((cell == null || drawn.Contains(cell)) && q.Count != 0)
                                cell = q.Dequeue();

                            if (cell == null)
                                break;

                            dc.DrawEllipse(Brushes.Orange, null, TransformPointFromWorldToCanvasCoordinates(cell.Location), 2, 2);

                            foreach (Cell c in cell.CellsOnTheLeft)
                            {
                                Brush brush = Brushes.LightBlue;
                                foreach (Cell cc in c.CellsOnTheLeft)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.DarkBlue;
                                        break;
                                    }
                                }
                                foreach (Cell cc in c.CellsOnTheRight)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.Yellow;
                                        break;
                                    }
                                }
                                dc.DrawLine(new Pen(brush, 2), TransformPointFromWorldToCanvasCoordinates(cell.Location),
                                    TransformPointFromWorldToCanvasCoordinates(c.Location));
                            }

                            foreach (Cell c in cell.CellsOnTheRight)
                            {
                                Brush brush = Brushes.Orange;
                                foreach (Cell cc in c.CellsOnTheLeft)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.Red;
                                        break;
                                    }
                                }
                                foreach (Cell cc in c.CellsOnTheRight)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.DarkRed;
                                        break;
                                    }
                                }
                                dc.DrawLine(new Pen(brush, 2), TransformPointFromWorldToCanvasCoordinates(cell.Location),
                                    TransformPointFromWorldToCanvasCoordinates(c.Location));
                            }

                            foreach (Cell c in cell.FollowingCells)
                            {
                                Brush brush = Brushes.LightGreen;
                                foreach (Cell cc in c.PreviousCells)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.DarkGreen;
                                        break;
                                    }
                                }
                                dc.DrawLine(new Pen(brush, 2), TransformPointFromWorldToCanvasCoordinates(cell.Location),
                                    TransformPointFromWorldToCanvasCoordinates(c.Location));
                            }

                            foreach (Cell c in cell.PreviousCells)
                            {
                                Brush brush = Brushes.Pink;
                                foreach (Cell cc in c.FollowingCells)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.Purple;
                                        break;
                                    }
                                }
                                dc.DrawLine(new Pen(brush, 2), TransformPointFromWorldToCanvasCoordinates(cell.Location),
                                    TransformPointFromWorldToCanvasCoordinates(c.Location));
                            }                            

                            foreach (Cell c in cell.CrossingCells)
                            {
                                Brush brush = Brushes.Brown;
                                foreach (Cell cc in c.CrossingCells)
                                {
                                    if (cc == cell)
                                    {
                                        brush = Brushes.Silver;
                                        break;
                                    }
                                }
                                dc.DrawLine(new Pen(brush, 2), TransformPointFromWorldToCanvasCoordinates(cell.Location),
                                    TransformPointFromWorldToCanvasCoordinates(c.Location));
                            }
                            if (cell.IsCrossroadCell)
                            {
                                for (int i = 1; i < cell.FollowingCells.Count; i++)
                                {
                                    q.Enqueue(cell.FollowingCells[i]);
                                    drawn.Add(cell.FollowingCells[i]);
                                }                                
                            }
                            
                            cell = cell.FollowingCell;
                                                        
                        }
                        //if (generator.ConnectedCell != null)
                        //{
                        //    Cell cell = generator.ConnectedCell;
                        //    while (cell != null)
                        //    {
                        //        dc.DrawRectangle(null, new Pen(Brushes.Red, 3),
                        //            new Rect(TransformPointFromWorldToCanvasCoordinates(cell.Location),
                        //                TransformPointFromWorldToCanvasCoordinates(cell.Location)));
                        //        cell = cell.FollowingCell;
                        //    }
                        //}
                    }
                }
                
                //foreach (Cell c in World.GetStatisticsCell())
                //{
                //   dc.DrawRectangle(null, new Pen(Brushes.Yellow, 3),
                //        new Rect(TransformPointFromWorldToCanvasCoordinates(c.Location),
                //            TransformPointFromWorldToCanvasCoordinates(c.Location)));
                //}
            }            
            //set this as background to canvas

            VisualBrush visualBrush = new VisualBrush(dv);
            visualBrush.Viewbox = new Rect(0, 0, elementToRenderOn.ActualWidth, elementToRenderOn.ActualHeight);
            visualBrush.ViewboxUnits = BrushMappingMode.Absolute;
            elementToRenderOn.Background = visualBrush;                        
        }

        private void DrawMainRoadMarks(DrawingContext dc, LaneSegment laneSegment1, double t1, LaneSegment laneSegment2, double t2)
        {
            double plusOffset = 10.0 / Scale;
            double minusOffset = -plusOffset;
            if (t1 == 0.0 || t1 == 1.0)
            {
                Pen mainRoadPen = new Pen(Brushes.Black, 3);
                mainRoadPen.Freeze();
                if (t1 == 0.0)
                {
                    dc.DrawLine(mainRoadPen,
                        TransformPointFromWorldToCanvasCoordinates(laneSegment1.ComputePoint(t1)),
                        TransformPointFromWorldToCanvasCoordinates(laneSegment1.ComputePoint(t1, plusOffset)));
                    dc.DrawLine(mainRoadPen,
                        TransformPointFromWorldToCanvasCoordinates(laneSegment2.ComputePoint(t2)),
                        TransformPointFromWorldToCanvasCoordinates(laneSegment2.ComputePoint(t2, plusOffset)));
                        //laneSegment2.Way == EWay.Normal ? plusOffset : minusOffset)));
                }
                else
                {
                    dc.DrawLine(mainRoadPen,
                        TransformPointFromWorldToCanvasCoordinates(laneSegment1.ComputePoint(t1)),
                        TransformPointFromWorldToCanvasCoordinates(laneSegment1.ComputePoint(t1, minusOffset)));
                    dc.DrawLine(mainRoadPen,
                        TransformPointFromWorldToCanvasCoordinates(laneSegment2.ComputePoint(t2)),
                        TransformPointFromWorldToCanvasCoordinates(laneSegment2.ComputePoint(t2, plusOffset)));
                        //laneSegment2.Way == EWay.Normal ? plusOffset : minusOffset)));
                }
            }
            else
                DrawMainRoadMarks(dc, laneSegment1, t1);
        }

        private void DrawMainRoadMarks(DrawingContext dc, LaneSegment laneSegment, double tOfCrossing)
        {
            double plusOffset = 10.0 / Scale;
            double minusOffset = -plusOffset;            
            Pen mainRoadPen = new Pen(Brushes.Black, 3);
            mainRoadPen.Freeze();                
            dc.DrawLine(mainRoadPen,
                TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(tOfCrossing)),
                TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(tOfCrossing, plusOffset)));
            dc.DrawLine(mainRoadPen,
                TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(tOfCrossing)),
                TransformPointFromWorldToCanvasCoordinates(laneSegment.ComputePoint(tOfCrossing, minusOffset)));
        }

        /// <summary>
        /// Draws built road
        /// </summary>
        /// <param name="dc">Drawing context</param>
        /// <param name="brushPreview">Brush used for drawing road</param>
        public void DrawRoad(DrawingContext dc, Brush brush, Cell firstCell, int length)
        {
            //draw in the first direction
            Cell currentCell = null;
            do
            {
                if (currentCell == null)
                    currentCell = firstCell;
                else
                    currentCell = currentCell.CellOnTheLeft;
                DrawLaneLine(dc, brush, currentCell, length, false, true);
            } while (currentCell.CellOnTheLeft != null && currentCell.CellOnTheLeft.CellOnTheLeft != currentCell);
            //draw one more lane line
            DrawLaneLine(dc, brush, currentCell, length, true, true);
            //draw in the second direction
            currentCell = currentCell.CellOnTheLeft;
            while (currentCell != null)
            {
                DrawLaneLine(dc, brush, currentCell, length, false, false);
                currentCell = currentCell.CellOnTheRight;
            }
        }

        /// <summary>
        /// Draws lane line
        /// </summary>
        /// <param name="dc">Drawing context do draw</param>
        /// <param name="brush">Brush used to draw</param>
        /// <param name="cell">First cell of road to draw</param>
        /// <param name="length">Number of cells of road do draw</param>
        /// <param name="rightTrueLeftFalse">Bool indicating if it has to be drawn left or right line of lane</param>
        /// <param name="backwardsTrueForwardsFalse">Bool indicating if lane line has to be drawn backwards or forwards</param>
        public void DrawLaneLine(DrawingContext dc, Brush brush, Cell cell, int length, bool rightTrueLeftFalse, bool backwardsTrueForwardsFalse)
        {
            //get point of beginning of drawing road
            Point frontMidPoint;
            if (backwardsTrueForwardsFalse)
                frontMidPoint = cell.Location + cell.DirectionVector * World.cellLength / 2;
            else
                frontMidPoint = cell.Location - cell.DirectionVector * World.cellLength / 2;
            //get point where to start drawing line representing one border of lane
            Vector toFrontLinePoint = cell.DirectionVector * World.LaneWidth / 2;
            //turn vector by 90 degrees
            double pom = toFrontLinePoint.X;
            toFrontLinePoint.X = toFrontLinePoint.Y;
            toFrontLinePoint.Y = -pom;
            if (!rightTrueLeftFalse)
                toFrontLinePoint.Negate();
            //get front line point which will be used for drawing line
            Point frontLinePoint = frontMidPoint - toFrontLinePoint;
            //get back line point which will be used for drawing line
            Point backLinePoint;
            if (backwardsTrueForwardsFalse)
                backLinePoint = frontLinePoint - cell.DirectionVector * World.cellLength * length;
            else
                backLinePoint = frontLinePoint + cell.DirectionVector * World.cellLength * length;
            //draw lane
            dc.DrawLine(new Pen(brush, 0.5), TransformPointFromWorldToCanvasCoordinates(frontLinePoint),
                TransformPointFromWorldToCanvasCoordinates(backLinePoint));
        }
        
        /// <summary>
        /// Render World on canvas
        /// </summary>
        /// <param name="dc">Drawing context used to drawing</param>
        /// <param name="width">Width of canvas</param>
        /// <param name="height">Height of canvas</param>
        public void Render(DrawingContext dc, double width, double height)
        {
            DrawRoadPreview(dc);
            
            //get elapsed part of frame to determine where exactly to render car
            double elapsedPartOfFrame = (DateTime.Now - World.LastSimulationStepTime).TotalMilliseconds / (World.SimulationStepTime * 1000);
            if (World.SimulationState == ESimulationState.Stopped)
                elapsedPartOfFrame = 0;
            //draw cars
            foreach (KeyValuePair<Car,KeyValuePair<Cell, Cell>> i in World.GetCars())
            {
                Vector oldNewPositions;
                Vector averageDirection;
                if (i.Value.Key != null)
                {
                    //get vector from old position to new
                    oldNewPositions = new Vector(i.Value.Key.Location.X - i.Value.Value.Location.X,
                        i.Value.Key.Location.Y - i.Value.Value.Location.Y);
                    //get average direction of directions of old and new cellColumnPair position
                    averageDirection = (i.Value.Key.DirectionVector + i.Value.Value.DirectionVector) / 2;
                    //normalize vector
                    averageDirection.Normalize();
                }
                else
                {
                    //new position not available because car leaves World. Current position is solved by current speed
                    oldNewPositions = i.Value.Value.DirectionVector * World.cellLength * i.Key.CurrentSpeed;
                    //get direction of old cellColumnPair
                    averageDirection = i.Value.Value.DirectionVector;
                }
                //multiply elapsed part of frame with vector
                oldNewPositions *= elapsedPartOfFrame;
                //add vector to oldpoint to create render point
                Point carPosition = i.Value.Value.Location + oldNewPositions + i.Value.Value.DirectionVector * World.cellLength / 2;
                //DRAW CAR
                //get midpoint of front car's side - move from center of imaginary cellColumnPair (because position is interpolated between
                //old and new cellColumnPair's positions) to its front side midpoint
                Point frontMidPoint = carPosition + averageDirection * World.cellLength / 2;
                //get midpoint of back car's  side
                Point backMidPoint = frontMidPoint - averageDirection * World.cellLength * i.Key.Length;
                //draw car
                dc.DrawLine(new Pen(new SolidColorBrush(i.Key.Color), 2 * Scale), 
                    TransformPointFromWorldToCanvasCoordinates(frontMidPoint), 
                    TransformPointFromWorldToCanvasCoordinates(backMidPoint));                
            }
        }

        private void DrawRoadControlPoints(DrawingContext dc, Brush brush, Point point1, Point point2)
        {
            dc.DrawEllipse(brush, null, point1, 3, 3);
            dc.DrawEllipse(brush, null, point2, 3, 3);
        }

        private void DrawRoadPreview(DrawingContext dc)
        {
            byte[] greens = new byte[] { 255, 175, 215, 135, 235, 155, 195, 115 };
            Brush[] brushes = new Brush[8];
            for (byte i = 0; i < 8; i++)
                brushes[i] = new SolidColorBrush(Color.FromRgb(0, greens[i], 0));
            
            int counter = 0;
            foreach(var i in linePoints)
            {
                if (i.Item2 == null) //straight line
                {
                    dc.DrawLine(new Pen(Brushes.Green, 1), i.Item1, i.Item4);
                }
                else //bezier curve
                {
                    dc.DrawGeometry(null, new Pen(brushes[counter % brushes.Length], 1), GetBezierCurveGeometry(i.Item1,
                        i.Item4, i.Item2.Value, i.Item3.Value));
                    DrawRoadControlPoints(dc, brushes[counter % brushes.Length], i.Item2.Value, i.Item3.Value);
                    counter++;
                }
            }
        }

        /// <summary>
        /// Event handler for rendering timer which invalidates canvas for new rendering
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void renderingTimer_Tick(object sender, EventArgs e)
        {
            //update interval if FrameRate was changed
            renderingTimer.Interval = TimeSpan.FromMilliseconds(1000 / frameRate);
            //invalidation of canvas
            elementToRenderOn.InvalidateVisual();
        }
        
        /// <summary>
        /// Starts rendering timer
        /// </summary>
        public void StartTimer()
        {
            //start
            renderingTimer.Start();
        }

        /// <summary>
        /// Stops rendering timer
        /// </summary>
        public void StopTimer()
        {
            renderingTimer.Stop();
        }

        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Changes cellLocation of view midpoint
        /// </summary>
        /// <param name="vector">Change by vector</param>
        public void ChangeViewMidpoint(Vector vector)
        {
            //change viewmidpoint
            viewMidpoint -= vector;
            //redraw canvas
            RedrawTopology();
            elementToRenderOn.InvalidateVisual();
        }

        /// <summary>
        /// Redraws canvas
        /// </summary>
        public void RedrawCanvas()
        {
            RedrawTopology();
            RedrawPreview();
        }

        /// <summary>
        /// Transform points to World coordinates system
        /// </summary>
        /// <param name="point">Point to transform</param>
        /// <returns>Transformed point</returns>
        public Point TransformPointFromCanvasToWorldCoordinates(Point point)
        {
            return (point - new Point(elementToRenderOn.ActualWidth / 2.0, elementToRenderOn.ActualHeight / 2.0)) / scale + viewMidpoint;
        }

        /// <summary>
        /// Transform point from world to canvas coordinates system
        /// </summary>
        /// <param name="point">Point to transform</param>
        /// <returns>Transformed point</returns>
        public Point TransformPointFromWorldToCanvasCoordinates(Point point)
        {
            return (point - viewMidpoint) * scale + new Point(elementToRenderOn.ActualWidth / 2.0, elementToRenderOn.ActualHeight / 2.0);
        }

        /// <summary>
        /// Set property and notify property changed
        /// </summary>
        /// <typeparam name="T">Generic variable</typeparam>
        /// <param name="field">Backing field reference</param>
        /// <param name="newValue">New value to set</param>
        /// <param name="propertyName">Property name</param>
        public void Set<T>(ref T field, T newValue, string propertyName)
        {
            if (!field.Equals(newValue))
            {
                field = newValue;
                if (PropertyChanged != null)
                    PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// Changes World instance (after loading from file)
        /// </summary>
        /// <param name="World">New World instance</param>
        public void ChangeWorld(World world)
        {
            this.World = world;
            Scale = 1;
            viewMidpoint = new Point();
        }

        /// <summary>
        /// Creates geometry with bezier curve from canvas-coordinated points
        /// </summary>
        /// <param name="p1">Start point</param>
        /// <param name="p2">End point</param>
        /// <param name="c1">Start control point</param>
        /// <param name="c2">End control point</param>
        /// <returns>Path geometry with one Bezier curve</returns>
        public static PathGeometry GetBezierCurveGeometry(Point p1, Point p2, Point c1, Point c2)
        {
            BezierSegment bezierSegment = new BezierSegment(c1, c2, p2, true);
            PathFigure pathFigure = new PathFigure(p1, new List<PathSegment>() { bezierSegment }, false);
            pathFigure.Freeze();
            PathGeometry geometry = new PathGeometry(new List<PathFigure>() { pathFigure });
            geometry.Freeze();
            return geometry;
        }

        /// <summary>
        /// Creates geometry with bezier curve from BezierLaneSegment instance with points in world coordinates
        /// </summary>
        /// <param name="bezierLaneSegment">Instance of bezier lane segment</param>
        /// <returns>Path geometry with one Bezier curve</returns>
        private PathGeometry GetBezierCurveGeometry(BezierLaneSegment bezierLaneSegment)
        {
            BezierSegment bezierSegment = new BezierSegment(
                TransformPointFromWorldToCanvasCoordinates(bezierLaneSegment.StartControlPoint),
                TransformPointFromWorldToCanvasCoordinates(bezierLaneSegment.EndControlPoint),
                TransformPointFromWorldToCanvasCoordinates(bezierLaneSegment.EndPoint), true);
            PathFigure pathFigure = new PathFigure(
                TransformPointFromWorldToCanvasCoordinates(bezierLaneSegment.StartPoint),
                new List<PathSegment>() { bezierSegment }, false);
            pathFigure.Freeze();

            PathGeometry geometry = new PathGeometry(new List<PathFigure>() { pathFigure });
            geometry.Freeze();
            return geometry;
        }

        internal void RedrawPreview()
        {
            elementToRenderOn.InvalidateVisual();
        }

        public static Point FromWorldSystemToCanvasSystem(Point point, Point averagePoint, 
            double maxDistanceToAveragePoint, double canvasSize)
        {
            //do not draw almost to border of canvas
            double sizeOfPlaceToDraw = canvasSize - 10.0;
            //crate middle point of canvas
            Point middlePoint = new Point(canvasSize / 2.0, canvasSize / 2.0);
            //maximum distance from point to middle of canvas
            double maxDistanceToAveragePointInCanvas = sizeOfPlaceToDraw / 2.0;
            //quotient of canvas coordinates of world coordinates
            double quocient = maxDistanceToAveragePointInCanvas / maxDistanceToAveragePoint;
            //get vector from average point to point
            Vector fromAveragePointToPoint = point - averagePoint;
            //apply quocient to vector
            fromAveragePointToPoint *= quocient;
            //return converted point
            return middlePoint + fromAveragePointToPoint;
        }

        internal static Point GetAveragePoint(IEnumerable<Point> points)
        {
            Point averagePoint = new Point(0, 0);
            foreach (var point in points)
                averagePoint += FromPointToVector(point);
            return FromVectorToPoint(FromPointToVector(averagePoint) / points.Count());
        }

        public static double GetMaxDistance(Point point, IEnumerable<Point> surroundingPoints)
        {
            double max = double.MinValue;
            foreach (var i in surroundingPoints)
                if ((i - point).Length > max)
                    max = (i - point).Length;
            return max;
        }


        private static Point FromVectorToPoint(Vector v)
        {
            return new Point(0.0, 0.0) + v;
        }

        private static Vector FromPointToVector(Point p)
        {
            return p - new Point(0.0, 0.0);
        }

    }
}
