﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Interaction logic for PrioritiesWindow.xaml
    /// </summary>
    public partial class PrioritiesWindow : Window
    {
        LaneSegment ls1, ls2;
        CrossingSegmentInfo csi1, csi2;        

        private PrioritiesWindow()
        {            
            InitializeComponent();
            DataContext = this;
        }

        public PrioritiesWindow(LaneSegment ls1, CrossingSegmentInfo csi1,
            LaneSegment ls2, CrossingSegmentInfo csi2) : this()
        {
            this.ls1 = ls1;
            this.ls2 = ls2;
            this.csi1 = csi1;
            this.csi2 = csi2;
            DrawCanvas();
        }

        private void DrawCanvas()
        {
            List<Point> points = new List<Point>() { ls1.StartPoint, ls1.EndPoint,
                ls2.StartPoint, ls2.EndPoint};
            if (ls1 is BezierLaneSegment)
            {
                points.Add((ls1 as BezierLaneSegment).StartControlPoint);
                points.Add((ls1 as BezierLaneSegment).EndControlPoint);
            }
            if (ls2 is BezierLaneSegment)
            {
                points.Add((ls2 as BezierLaneSegment).StartControlPoint);
                points.Add((ls2 as BezierLaneSegment).EndControlPoint);
            }
            Point averagePoint = Renderer.GetAveragePoint(points);
            double maxDistanceToAveragePoint = Renderer.GetMaxDistance(averagePoint, points);
            
            DrawingVisual dv = new DrawingVisual();
            using (DrawingContext dc = dv.RenderOpen())
            {
                //draw the first segment
                Pen pen1 = new Pen(Brushes.Black, csi1.Priority == EPriority.High ? 3.0 : 1.0);
                pen1.Freeze();
                if (ls1 is StraightLaneSegment)
                    DrawLine(dc, pen1, ls1, averagePoint, maxDistanceToAveragePoint);
                else if (ls1 is BezierLaneSegment)
                    DrawBezier(dc, pen1, ls1 as BezierLaneSegment, averagePoint, maxDistanceToAveragePoint);

                //draw the second segment
                Pen pen2 = new Pen(Brushes.Black, csi1.Priority == EPriority.Low ? 3.0 : 1.0);
                pen2.Freeze();
                if (ls2 is StraightLaneSegment)
                    DrawLine(dc, pen2, ls2, averagePoint, maxDistanceToAveragePoint);
                else if (ls2 is BezierLaneSegment)
                    DrawBezier(dc, pen2, ls2 as BezierLaneSegment, averagePoint, maxDistanceToAveragePoint);

                //draw ellipse
                Brush brush = csi1.TOfMyCrossing == 0.0 || csi1.TOfMyCrossing == 1.0 ||
                    csi1.TOfCrossingSegment == 0.0 || csi1.TOfCrossingSegment == 1.0 ?
                    Brushes.Orange : Brushes.Red;
                DrawEllipse(dc, brush, ls1, csi1, averagePoint, maxDistanceToAveragePoint);
            }
            VisualBrush visualBrush = new VisualBrush(dv);
            visualBrush.Viewbox = new Rect(0, 0, canvasCrossroadScheme.ActualWidth, 
                canvasCrossroadScheme.ActualHeight);
            visualBrush.ViewboxUnits = BrushMappingMode.Absolute;
            canvasCrossroadScheme.Background = visualBrush;    
        }

        private void DrawEllipse(DrawingContext dc, Brush brush, LaneSegment ls1, CrossingSegmentInfo csi1, 
            Point averagePoint, double maxDistanceToAveragePoint)
        {
            dc.DrawEllipse(brush, null,
                Renderer.FromWorldSystemToCanvasSystem(ls1.ComputePoint(csi1.TOfMyCrossing),
                averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                canvasCrossroadScheme.ActualWidth)), 3.0, 3.0);
        }

        private void DrawBezier(DrawingContext dc, Pen pen, BezierLaneSegment bezierLaneSegment, 
            Point averagePoint, double maxDistanceToAveragePoint)
        {
            dc.DrawGeometry(null, pen,
                Renderer.GetBezierCurveGeometry(
                Renderer.FromWorldSystemToCanvasSystem(bezierLaneSegment.StartPoint,
                averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth)),
                Renderer.FromWorldSystemToCanvasSystem(bezierLaneSegment.EndPoint,
                averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth)),
                Renderer.FromWorldSystemToCanvasSystem(bezierLaneSegment.StartControlPoint,
                averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth)),
                Renderer.FromWorldSystemToCanvasSystem(bezierLaneSegment.EndControlPoint,
                averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth))));
        }

        private void DrawLine(DrawingContext dc, Pen pen, LaneSegment ls1, 
            Point averagePoint, double maxDistanceToAveragePoint)
        {
            dc.DrawLine(pen,
                        Renderer.FromWorldSystemToCanvasSystem(ls1.StartPoint,
                            averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth)),
                        Renderer.FromWorldSystemToCanvasSystem(ls1.EndPoint,
                            averagePoint, maxDistanceToAveragePoint, Math.Min(canvasCrossroadScheme.ActualHeight,
                            canvasCrossroadScheme.ActualWidth)));
        }

        private void buttonChangePriority_Click(object sender, RoutedEventArgs e)
        {
            if (csi1.Priority == EPriority.NotSet || csi1.Priority == EPriority.Low)
            {
                csi1.Priority = EPriority.High;
                SetPriorityToTheSecondLaneWhichCanBeAlsoEdgeSegment(csi1, csi2, EPriority.Low);
            }
            else
            {
                csi1.Priority = EPriority.Low;
                SetPriorityToTheSecondLaneWhichCanBeAlsoEdgeSegment(csi1, csi2, EPriority.High);
            }

            DrawCanvas();
        }

        private void SetPriorityToTheSecondLaneWhichCanBeAlsoEdgeSegment(CrossingSegmentInfo csi1, 
            CrossingSegmentInfo csi2, EPriority ePriority)
        {
            if (csi2 == null)
            {
                if (csi1.TOfCrossingSegment == 0.0)
                    ls2.PreviousSegment.Priority = ePriority;
                else if (csi1.TOfCrossingSegment == 1.0)
                    ls2.FollowingSegment.Priority = ePriority;
                else
                    throw new ApplicationException(
                        "Expected that crossing segment is beginning or ending on this segment");
            }
            else
                csi2.Priority = ePriority;
        }

        private void Window_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            DrawCanvas();
        }

        public object TranslationInvoker { get { return new Object(); } set { } }
    }
}
