﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.ComponentModel;
using System.Collections.ObjectModel;
using System.Windows.Controls.Primitives;
using System.Windows.Threading;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using TrafficMicroSimulator.Properties;
using System.Windows.Data;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window, INotifyPropertyChanged
    {        
        protected ERoadContructionStatus roadConstructionStatus = ERoadContructionStatus.NoConstruction;
        public ERoadContructionStatus RoadConstructionStatus
        {
            get { return roadConstructionStatus; }
            set { Set<ERoadContructionStatus>(ref roadConstructionStatus, value, "RoadConstructionStatus"); }
        }                                                                         
                
        private List<Tuple<Point, Point?, Point?, Point>> roadPreviewPoints = new List<Tuple<Point, Point?, Point?, Point>>();
        private Point firstPoint;
        private Point firstControlPoint;
        private Point secondPoint;
        private Point secondControlPoint;
        private Tuple<int, int> whichPointToMove = null;

        /// <summary>
        /// String used to translating
        /// </summary>
        public object TranslationInvoker
        {
            get { return new Object(); }
            set
            {
                object foo = new Object();
                Set<object>(ref foo, new Object(), "TranslationInvoker");
            }
        }

        /// <summary>
        /// Instance of World which contains roads, cars...
        /// </summary>
        public World World { get; set; }

        /// <summary>
        /// Instance of renderer which renders World
        /// </summary>
        public Renderer Renderer { get; set; }

        /// <summary>
        /// Indicates whether constructing controls are enabled
        /// </summary>
        private  bool constructionAvailable = true;
        public bool ConstructionAvailable
        {
            get { return constructionAvailable; }
            set { Set<bool>(ref constructionAvailable, value, "ConstructionAvailable"); }
        }
        
        /// <summary>
        /// Indicates last position of mouse cursor which is updating after every event
        /// MouseMove
        /// </summary>
        protected Point lastMouseCursorPosition = new Point();

        /// <summary>
        /// To reach that the constructing road active point 
        /// has the same world position
        /// during zooming and not moving the cursor
        /// </summary>
        protected Point lastMovingPointPosition = new Point();

        /// <summary>
        /// Program current active mode
        /// </summary>
        protected EMode activeMode = EMode.RoadConstruction;
        /// <summary>
        /// Program current active mode
        /// </summary>
        public EMode ActiveMode 
        {
            get { return activeMode; }
            set { Set<EMode>(ref activeMode, value, "ActiveMode"); }
        }

        /// <summary>
        /// Road constructing mode
        /// </summary>
        protected ERoadConstructionMode roadConstructionMode = ERoadConstructionMode.DoNotConnect;
        /// <summary>
        /// Program current active mode
        /// </summary>
        public ERoadConstructionMode RoadConstructionMode
        {
            get { return roadConstructionMode; }
            set { Set<ERoadConstructionMode>(ref roadConstructionMode, value, "RoadConstructionMode"); }
        }

        /// <summary>
        /// Current road shape type
        /// </summary>
        protected ERoadShapeType roadShapeType = ERoadShapeType.Straight;
        /// <summary>
        /// Current road shape type
        /// </summary>
        public ERoadShapeType RoadShapeType
        {
            get { return roadShapeType; }
            set { Set<ERoadShapeType>(ref roadShapeType, value, "RoadShapeType"); }
        }

        /// <summary>
        /// Chosen generator type 
        /// </summary>
        protected EGeneratorType generatorType = EGeneratorType.Deterministic;
        /// <summary>
        /// Chosen generator type
        /// </summary>
        public EGeneratorType GeneratorType
        {
            get { return generatorType; }
            set { Set<EGeneratorType>(ref generatorType, value, "GeneratorType"); }
        }

        /// <summary>
        /// Deterministic interval to generate cars
        /// </summary>
        protected double deterministicInterval = 5.0;
        /// <summary>
        /// Deterministic interval to generate cars
        /// </summary>
        public double DeterministicInterval
        {
            get { return deterministicInterval; }
            set { Set<double>(ref deterministicInterval, value, "DeterministicInterval"); }
        }

        /// <summary>
        /// A parameter to exponential generator
        /// </summary>
        protected double a_parameter = 5.0;
        /// <summary>
        /// A parameter to exponential generator
        /// </summary>
        public double A_parameter
        {
            get { return a_parameter; }
            set { Set<double>(ref a_parameter, value, "A_parameter"); }
        }

        /// <summary>
        /// Ignore click
        /// </summary>
        bool ignoreClickBecauseActionFromMenuWasExecuted = false;

        /// <summary>
        /// mi parameter to gauss generator
        /// </summary>
        protected double mi = 5.0;
        /// <summary>
        /// mi parameter to gauss generator
        /// </summary>
        public double Mi
        {
            get { return mi; }
            set { Set<double>(ref mi, value, "Mi"); }
        }

        /// <summary>
        /// sigma parameter to gauss generator
        /// </summary>
        protected double sigma = 1.0;
        /// <summary>
        /// sigma parameter to gauss generator
        /// </summary>
        public double Sigma
        {
            get { return sigma; }
            set { Set<double>(ref sigma, value, "Sigma"); }
        }

        /// <summary>
        /// Selected roadtype
        /// </summary>
        protected ERoadWays roadType = ERoadWays.OneWay;
        /// <summary>
        /// Selected roadtype
        /// </summary>
        public ERoadWays RoadType
        {
            get { return roadType; }
            set { Set<ERoadWays>(ref roadType, value, "RoadType"); }
        }

        /// <summary>
        /// Selected number of lanes of the first direction
        /// </summary>
        protected int numberOfLanes1 = 1;
        /// <summary>
        /// Selected number of lanes of the first direction
        /// </summary>
        public int NumberOfLanes1
        {
            get { return numberOfLanes1; }
            set { Set<int>(ref numberOfLanes1, value, "NumberOfLanes1"); }
        }

        /// <summary>
        /// Selected number of lanes of the second direction
        /// </summary>
        protected int numberOfLanes2 = 1;
        /// <summary>
        /// Selected number of lanes of the first direction
        /// </summary>
        public int NumberOfLanes2
        {
            get 
            {
                if (roadType == ERoadWays.OneWay)
                    return 0;
                return numberOfLanes2; 
            }
            set { Set<int>(ref numberOfLanes2, value, "NumberOfLanes2"); }
        }
        
        /// <summary>
        /// Max allowed speed on the road
        /// </summary>
        protected int allowedSpeed = 18;
        /// <summary>
        /// Max allowed speed on the road
        /// </summary>
        public double AllowedSpeed
        {
            get { return (double)allowedSpeed * (World.cellLength 
                / World.SimulationStepTime) * World.mps2kph; }
            set { Set<int>(ref allowedSpeed, (int)Math.Round(value 
                / World.mps2kph / (World.cellLength / World.SimulationStepTime)), "AllowedSpeed"); }
        }

        protected Popup popupStatistics = new Popup();
        protected DispatcherTimer popupTimer = new DispatcherTimer(DispatcherPriority.Normal);

        public MainWindow()
        {
            //initialize form, active mode World and renderer            
            InitializeComponent();
            World = new World();
            Renderer = new Renderer(World, canvasWorld, roadPreviewPoints);
            //set datacontext
            DataContext = this;            
            //pass renderer instance to canvas
            canvasWorld.renderer = Renderer;
            popupTimer.Stop();
            popupTimer.Tick += new EventHandler(popupTimer_Tick);            
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            base.OnRender(drawingContext);
            //draw zoom canvas
            canvasZoom.Height = statusBar.ActualHeight - 10;
            DrawingVisual dv = new DrawingVisual();
            using (DrawingContext dc = dv.RenderOpen())
            {
                dc.DrawLine(new Pen(Brushes.Black, 1), new Point(0, canvasZoom.Height / 2),
                    new Point(canvasZoom.ActualWidth, canvasZoom.Height / 2));
                dc.DrawLine(new Pen(Brushes.Black, 2), new Point(0, 0), new Point(0, canvasZoom.Height));
                dc.DrawLine(new Pen(Brushes.Black, 2), new Point(canvasZoom.ActualWidth, 0),
                     new Point(canvasZoom.ActualWidth, canvasZoom.Height));
            }
            VisualBrush canvasZoomBackground = new VisualBrush(dv);
            canvasZoomBackground.Viewbox = new Rect(0, 0, canvasZoom.ActualWidth, canvasZoom.Height);
            canvasZoomBackground.ViewboxUnits = BrushMappingMode.Absolute;
            canvasZoom.Background = canvasZoomBackground;
        }

        private void popupTimer_Tick(object sender, EventArgs e)
        {
            if (World.SimulationState != ESimulationState.Running)
                return;
            if (!World.IsPointInRectangle(new Point(0,0), new Point(canvasWorld.Width, canvasWorld.Height), lastPoint))
                return;
            //get nearest cell
            double distance = 0;
            Cell nearestCell = null;
            Point transformedPoint = Renderer.TransformPointFromCanvasToWorldCoordinates(lastPoint);
            
            if (nearestCell == null || distance > 10 / Renderer.Scale)
                return;
            StackPanel stackPanel = new StackPanel();
            Label labelSpeed = new Label();
            Label labelFlow = new Label();
            Label labelDensity = new Label();
            double? averageSpeed = nearestCell.GetAverageSpeed();
            int? flow = nearestCell.GetFlow();
            if (averageSpeed == null || averageSpeed == 0.0)
            {
                labelSpeed.Content = "Speed: N/A";
                labelFlow.Content = "Flow: N/A";
                labelDensity.Content = "Density: N/A";
            }
            else
            {
                double toKmh = (World.cellLength / World.SimulationStepTime) * World.mps2kph;
                labelSpeed.Content = "Speed: " + (averageSpeed.Value * toKmh).ToString("0.000") + " km/h";
                labelFlow.Content = "Flow: " + flow.Value + " veh/min";
                labelDensity.Content = "Density: " + ((60 / toKmh) * flow.Value / averageSpeed.Value).ToString("0.000") + " veh/km";
            }
            stackPanel.Children.Add(labelSpeed);
            stackPanel.Children.Add(labelFlow);
            stackPanel.Children.Add(labelDensity);
            popupStatistics.AllowsTransparency = true;
            popupStatistics.Placement = PlacementMode.Mouse;
            //popupStatistics.StaysOpen = false;
            popupStatistics.Height = 100;
            popupStatistics.Width = 200;
            popupStatistics.IsOpen = true;
            popupStatistics.Child = stackPanel;
        }

        /// <summary>
        /// Enable rendering and simulation step timers
        /// </summary>
        public void StartTimers(bool previousConstructionAvailable)
        {
            popupTimer.Start();
            popupTimer.Interval = TimeSpan.FromSeconds(2);
            World.StartTimers(previousConstructionAvailable);
            //Renderer.RedrawCanvas();
            Renderer.StartTimer();
        }

        /// <summary>
        /// Stop rendering and simulation step timers
        /// </summary>
        public void StopTimers()
        {
            popupTimer.Stop();
            World.StopTimers();
            Renderer.StopTimer();
        }      
       
        /// <summary>
        /// Occurs when mouse wheel is moved
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainWindow_PreviewMouseWheel(object sender, MouseWheelEventArgs e)
        {                       
            //wheel up 120 -> zoom slider right, wheel down -120 -> zoom slider left
            if (e.Delta > 0)
            {
                //search where is slider pointer
                for (double i = 0.25; i <= 10.0; i += 0.25)
                {
                    if (sliderZoom.Value < i - 0.05)
                    {
                        //slider pointer found, adjust it. (0.1 -> because of calling
                        //(Convert and ConvertBack methods in converter after updating
                        //pointer -> slider may slightly move -> impossible to move it
                        //more left)
                        sliderZoom.Value = i;
                        break;
                    }
                }
            }
            else
            {
                //search where is slider pointer
                for (double i = 9.75; i >= 0.0; i -= 0.25)
                {
                    if (sliderZoom.Value > i + 0.05)
                    {
                        //slider pointer found, adjust it. (0.1 -> because of calling
                        //(Convert and ConvertBack methods in converter after updating
                        //pointer -> slider may slightly move -> impossible to move it
                        //more left)
                        sliderZoom.Value = i;
                        break;
                    }
                }
            }
            e.Handled = true;
        }

        /// <summary>
        /// Occurs when left mouse button is clicked when cursor is over canvas
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void canvasWorld_MouseLeftButtonUp(object sender, MouseButtonEventArgs e)
        {
            if (ignoreClickBecauseActionFromMenuWasExecuted)
            {
                ignoreClickBecauseActionFromMenuWasExecuted = false;
                return;
            }

            Point worldPoint = Renderer.TransformPointFromCanvasToWorldCoordinates
                (e.GetPosition(canvasWorld));
            //check active mode
            if (ActiveMode == EMode.RoadConstruction)
            {
                Point currentPoint = e.GetPosition(canvasWorld);
                //road construction mode
                if (RoadConstructionStatus == ERoadContructionStatus.NoConstruction)
                {
                    firstPoint = currentPoint;
                    //store first point
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null,
                        firstPoint + new Vector(0, 0.000001), NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                    RoadConstructionStatus = ERoadContructionStatus.FirstPointPlaced;
                    Renderer.RedrawPreview();
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.FirstPointPlaced)
                {
                    secondPoint = currentPoint;
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null, secondPoint,
                        NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                    RoadConstructionStatus = ERoadContructionStatus.Editing;
                    Renderer.RedrawPreview();
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.SecondPointPlaced)
                {
                    firstControlPoint = currentPoint;
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, firstControlPoint, firstControlPoint, secondPoint,
                        NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                    RoadConstructionStatus = ERoadContructionStatus.FirstControlPointPlaced;
                    Renderer.RedrawPreview();
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.FirstControlPointPlaced)
                {
                    secondControlPoint = currentPoint;
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, firstControlPoint, secondControlPoint, secondPoint,
                        NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                    RoadConstructionStatus = ERoadContructionStatus.Editing;
                    Renderer.RedrawPreview();
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.Editing)
                {
                    whichPointToMove = null;
                }
            }
            else if (ActiveMode == EMode.GeneratorConstruction)
            {                
                LaneSegment nearestLaneSegment = null;
                foreach (LaneSegment ls in World.LaneSegments)
                {
                    if (nearestLaneSegment == null || nearestLaneSegment.DistanceToStartPoint(worldPoint, 10 / Renderer.Scale) > 
                        ls.DistanceToStartPoint(worldPoint, 10 / Renderer.Scale))
                    {
                        nearestLaneSegment = ls;
                    }
                }
                //create generator
                if (nearestLaneSegment != null && nearestLaneSegment.DistanceToStartPoint(worldPoint, 10 / Renderer.Scale) != 
                    double.MaxValue)
                {
                    try
                    {
                        switch (generatorType)
                        {
                            case EGeneratorType.Deterministic:
                                DeterministicGenerator dg = new DeterministicGenerator(nearestLaneSegment, TimeSpan.FromSeconds(deterministicInterval));
                                nearestLaneSegment.Generator = dg;
                                break;
                            case EGeneratorType.Exponential:
                                ExponentialGenerator eg = new ExponentialGenerator(nearestLaneSegment, TimeSpan.FromSeconds(a_parameter));
                                nearestLaneSegment.Generator = eg;
                                break;
                            case EGeneratorType.Gauss:
                                GaussGenerator gg = new GaussGenerator(nearestLaneSegment, TimeSpan.FromSeconds(mi), TimeSpan.FromSeconds(sigma));
                                nearestLaneSegment.Generator = gg;
                                break;
                        }
                    }
                    catch (ApplicationException ex)
                    {
                        MessageBox.Show(ex.Message, "Message", MessageBoxButton.OK, MessageBoxImage.Exclamation);
                    }
                    Renderer.RedrawCanvas();
                }
            }
            else if (ActiveMode == EMode.DirectionCells)
            {                                
                DirectionPoint chosenDirectionPoint = new InteractionEdgeToLaneMaker(worldPoint, 
                    World.LaneSegments, 10.0 / Renderer.Scale).ChosenDirectionPoint;
                if (chosenDirectionPoint == null)
                    Renderer.RedrawCanvas();
                else                
                    new DirectionsWindow(chosenDirectionPoint).ShowDialog();                                    
            }
            else if (ActiveMode == EMode.Selection)
            {                
                
            }
            else if (ActiveMode == EMode.Priorities)
            {
                LaneSegment ls1, ls2;
                CrossingSegmentInfo csi1, csi2;
                new InteractionEdgeToLaneMaker(worldPoint, 10.0 / Renderer.Scale, World.LaneSegments,
                    out ls1, out ls2, out csi1, out csi2);
                if (ls1 != null)
                {
                    new PrioritiesWindow(ls1, csi1, ls2, csi2).ShowDialog();
                    Renderer.RedrawCanvas();
                }
            }
        }

        /// <summary>
        /// True when user moves canvas
        /// </summary>
        bool canvasWasMoved = false;

        /// <summary>
        /// Occurs when mouse right button is clicked and mouse is over canvas
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void canvasWorld_MouseRightButtonUp_1(object sender, MouseButtonEventArgs e)
        {            
            //check if user was not moving canvas
            if (!canvasWasMoved)
            {
                Point worldPoint = Renderer.TransformPointFromCanvasToWorldCoordinates(e.GetPosition(canvasWorld));
                if (ActiveMode == EMode.RoadConstruction)
                {
                    Point currentPoint = e.GetPosition(canvasWorld);
                    if (RoadConstructionStatus == ERoadContructionStatus.FirstPointPlaced)
                    {
                        secondPoint = currentPoint;
                        World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null, secondPoint,
                            NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                        RoadConstructionStatus = ERoadContructionStatus.SecondPointPlaced;
                        Renderer.RedrawPreview();
                    }
                    else if (RoadConstructionStatus == ERoadContructionStatus.SecondPointPlaced)
                    {
                        secondPoint = currentPoint;
                        World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null, secondPoint,
                            NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                        RoadConstructionStatus = ERoadContructionStatus.FirstPointPlaced;
                        Renderer.RedrawPreview();
                    }
                    else if (RoadConstructionStatus == ERoadContructionStatus.FirstControlPointPlaced)
                    {
                        firstControlPoint = currentPoint;
                        World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, firstControlPoint, 
                            firstControlPoint, secondPoint,
                            NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                        RoadConstructionStatus = ERoadContructionStatus.SecondPointPlaced;
                        Renderer.RedrawPreview();
                    }
                    else if (RoadConstructionStatus == ERoadContructionStatus.Editing)
                    {
                        for (int i = 0; i < roadPreviewPoints.Count; i++)
                        {
                            roadPreviewPoints[i] = new Tuple<Point, Point?, Point?, Point>(
                                Renderer.TransformPointFromCanvasToWorldCoordinates(roadPreviewPoints[i].Item1),
                                roadPreviewPoints[i].Item2.HasValue ?
                                    Renderer.TransformPointFromCanvasToWorldCoordinates(roadPreviewPoints[i].Item2.Value) : (Point?)null,
                                roadPreviewPoints[i].Item3.HasValue ?
                                    Renderer.TransformPointFromCanvasToWorldCoordinates(roadPreviewPoints[i].Item3.Value) : (Point?)null,
                                Renderer.TransformPointFromCanvasToWorldCoordinates(roadPreviewPoints[i].Item4));
                        }
                        World.BuildRoad(roadPreviewPoints, NumberOfLanes1, allowedSpeed, 10 / Renderer.Scale);
                        RoadConstructionStatus = ERoadContructionStatus.NoConstruction;
                        whichPointToMove = null;
                        roadPreviewPoints.Clear();
                        Renderer.RedrawCanvas();                        
                    }
                    else if (RoadConstructionStatus == ERoadContructionStatus.NoConstruction)
                    {
                        World.RemoveLaneSegment(worldPoint, 10 / Renderer.Scale);
                        Renderer.RedrawCanvas();
                    }
                }
                else if (ActiveMode == EMode.GeneratorConstruction)
                {
                    World.RemoveGenerator(worldPoint, 10 / Renderer.Scale);
                    Renderer.RedrawCanvas();
                }
                else if (ActiveMode == EMode.DirectionCells)
                {
                    World.RemoveDirectionCell(worldPoint, 10 / Renderer.Scale);
                    Renderer.RedrawCanvas();
                }
            }
        }

        /// <summary>
        /// Occures when mouse right button is pressed and mouse is over canvas
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void canvasWorld_MouseRightButtonDown_1(object sender, MouseButtonEventArgs e)
        {
            canvasWasMoved = false;
        }


        private void canvasWorld_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            if (ActiveMode == EMode.RoadConstruction && RoadConstructionStatus == ERoadContructionStatus.Editing)
            {
                Point clickedPoint = e.GetPosition(canvasWorld);
                double minDistance = 10.0;
                for (int i = 0; i < roadPreviewPoints.Count; i++)
                {
                    CompareDistance(ref minDistance, clickedPoint, i, 1);
                    CompareDistance(ref minDistance, clickedPoint, i, 2);
                    CompareDistance(ref minDistance, clickedPoint, i, 3);
                    CompareDistance(ref minDistance, clickedPoint, i, 4);
                }
            }
        }

        private void CompareDistance(ref double minDistance, Point clickedPoint, int roadPreviewRoadListIndex, int tupleIndex)
        {
            Tuple<Point, Point?, Point?, Point> points = roadPreviewPoints[roadPreviewRoadListIndex];
            Point point;
            switch (tupleIndex)
            {
                case 1: point = points.Item1; break;
                case 2: if (!points.Item2.HasValue) return; point = points.Item2.Value; break;
                case 3: if (!points.Item3.HasValue) return; point = points.Item3.Value; break;
                case 4: point = points.Item4; break;
                default: throw new ApplicationException("Tuple index not in <1,4>");
            }
            if ((point - clickedPoint).Length < minDistance)
            {
                minDistance = (point - clickedPoint).Length;
                whichPointToMove = new Tuple<int,int>(roadPreviewRoadListIndex, tupleIndex);
            }            
        }

        /// <summary>
        /// Occurs when mouse moves while cursor is over canvas
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void canvasWorld_MouseMove(object sender, MouseEventArgs e)
        {
            popupStatistics.IsOpen = false;
            lastPoint = e.GetPosition(canvasWorld);
            if (World.SimulationState == ESimulationState.Running)
            {
                popupTimer.Stop();
                popupTimer.Interval = TimeSpan.FromSeconds(2);
                popupTimer.Start();
            }

            //get vector of move
            Vector moveVector = (e.GetPosition(canvasWorld) - lastMouseCursorPosition) / Renderer.Scale;

            if (e.RightButton == MouseButtonState.Pressed)
            {
                //right button is pressed, change view midpoint
                Renderer.ChangeViewMidpoint(moveVector);
                //recompute points
                RecomputePointAfterCanvasMove(moveVector);                
                canvasWasMoved = true;
            }

            //update last known cursor position to proper defining of move vector
            lastMouseCursorPosition = e.GetPosition(canvasWorld);
            lastMovingPointPosition = lastMouseCursorPosition;

            //redraw road preview
            if (!(RoadConstructionStatus == ERoadContructionStatus.NoConstruction))
            {                
                if (e.RightButton == MouseButtonState.Pressed)
                {
                    firstPoint = RecomputePointAfterCanvasMove(firstPoint, moveVector).Value;
                    secondPoint = RecomputePointAfterCanvasMove(secondPoint, moveVector).Value;
                    firstControlPoint = RecomputePointAfterCanvasMove(firstControlPoint, moveVector).Value;
                }
                Point currentPoint = e.GetPosition(canvasWorld);
                if (RoadConstructionStatus == ERoadContructionStatus.FirstPointPlaced)
                {
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null, currentPoint,
                        NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.SecondPointPlaced)
                {
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, currentPoint,
                        currentPoint, secondPoint, NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.FirstControlPointPlaced)
                {
                    World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, firstControlPoint,
                        currentPoint, secondPoint, NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
                }
                else if (RoadConstructionStatus == ERoadContructionStatus.Editing && whichPointToMove != null)
                {
                    Tuple<Point,Point?,Point?,Point> tuple = roadPreviewPoints[whichPointToMove.Item1];
                    switch (whichPointToMove.Item2)
                    {
                        case 1: roadPreviewPoints[whichPointToMove.Item1] =
                            new Tuple<Point, Point?, Point?, Point>(RecomputePointAfterCanvasMove(tuple.Item1, moveVector).Value,
                                tuple.Item2, tuple.Item3, tuple.Item4); break;
                        case 2: roadPreviewPoints[whichPointToMove.Item1] =
                            new Tuple<Point, Point?, Point?, Point>(tuple.Item1,
                                RecomputePointAfterCanvasMove(tuple.Item2, moveVector), tuple.Item3, tuple.Item4); break;
                        case 3: roadPreviewPoints[whichPointToMove.Item1] =
                            new Tuple<Point, Point?, Point?, Point>(tuple.Item1,
                                tuple.Item2, RecomputePointAfterCanvasMove(tuple.Item3, moveVector), tuple.Item4); break;
                        case 4: roadPreviewPoints[whichPointToMove.Item1] =
                            new Tuple<Point, Point?, Point?, Point>(tuple.Item1,
                                tuple.Item2, tuple.Item3, RecomputePointAfterCanvasMove(tuple.Item4, moveVector).Value); break;
                        default: throw new ApplicationException("Tuple index not in <1,4>");
                    }
                }
                Renderer.RedrawPreview();
            }
        }

        /// <summary>
        /// Recomputes point after user moves canvas
        /// </summary>
        /// <param name="point">Point to recompute</param>
        /// <param name="moveVector">Vector of canvas move</param>
        protected void RecomputePointAfterCanvasMove(Vector moveVector)
        {
            for (int i = 0; i < roadPreviewPoints.Count; i++)
            {
                roadPreviewPoints[i] = new Tuple<Point, Point?, Point?, Point>(
                    RecomputePointAfterCanvasMove(roadPreviewPoints[i].Item1, moveVector).Value,
                    RecomputePointAfterCanvasMove(roadPreviewPoints[i].Item2, moveVector),
                    RecomputePointAfterCanvasMove(roadPreviewPoints[i].Item3, moveVector),
                    RecomputePointAfterCanvasMove(roadPreviewPoints[i].Item4, moveVector).Value);
            }
        }

        protected Point? RecomputePointAfterCanvasMove(Point? point, Vector moveVector)
        {
            if (!point.HasValue)
                return null;
            return point + moveVector * Renderer.Scale;            
        }

        public event PropertyChangedEventHandler PropertyChanged;
        private Point lastPoint;
        

        /// <summary>
        /// Set property and notify property changed
        /// </summary>
        /// <typeparam name="T">Generic variable</typeparam>
        /// <param name="field">Backing field reference</param>
        /// <param name="newValue">New value to set</param>
        /// <param name="propertyName">Property name</param>
        public void Set<T>(ref T field, T newValue, string propertyName, bool changeAlways = false)
        {
            if (changeAlways || !field.Equals(newValue))
            {
                field = newValue;
                if (PropertyChanged != null)
                    PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// Size of canvas changed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void canvasWorld_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            //repaint background and invalidate visual to redraw cars
            Renderer.RedrawCanvas();
        }

        /// <summary>
        /// Zoom slider changed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void sliderZoom_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            //redraw canvas
            if (ActiveMode == EMode.RoadConstruction && RoadConstructionStatus != ERoadContructionStatus.NoConstruction)
            {
                RecomputePointsAfterZoomChange((double)new ZoomConverter().ConvertBack(e.OldValue, typeof(double), null, null));
            }
            //repaint background and invalidate visual to redraw cars
            Renderer.RedrawCanvas();
        }


        private void RecomputePointsAfterZoomChange(double oldScale)
        {
            lastMovingPointPosition = RecomputePointAfterZoomChange(lastMovingPointPosition, oldScale);
            Point currentPoint = lastMovingPointPosition;
            
            if (RoadConstructionStatus == ERoadContructionStatus.FirstPointPlaced)
            {                
                firstPoint = RecomputePointAfterZoomChange(firstPoint, oldScale);             
                World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, null, null, currentPoint, NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
            }
            else if (RoadConstructionStatus == ERoadContructionStatus.SecondPointPlaced)
            {
                firstPoint = RecomputePointAfterZoomChange(firstPoint, oldScale);
                secondPoint = RecomputePointAfterZoomChange(secondPoint, oldScale);
                World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, currentPoint,
                    currentPoint, secondPoint, NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
            }
            else if (RoadConstructionStatus == ERoadContructionStatus.FirstControlPointPlaced)
            {
                firstPoint = RecomputePointAfterZoomChange(firstPoint, oldScale);
                secondPoint = RecomputePointAfterZoomChange(secondPoint, oldScale);
                firstControlPoint = RecomputePointAfterZoomChange(firstControlPoint, oldScale);
                World.GetLaneSegmentPoints(roadPreviewPoints, firstPoint, firstControlPoint,
                    currentPoint, secondPoint, NumberOfLanes1 + NumberOfLanes2, Renderer.Scale * World.LaneWidth);
            }
            else if (RoadConstructionStatus == ERoadContructionStatus.Editing)
            {
                for (int i = 0; i < roadPreviewPoints.Count; i++)
                {
                    Tuple<Point, Point?, Point?, Point> tuple = roadPreviewPoints[i];
                    roadPreviewPoints[i] = new Tuple<Point, Point?, Point?, Point>(
                        RecomputePointAfterZoomChange(tuple.Item1, oldScale),
                        tuple.Item2.HasValue ? RecomputePointAfterZoomChange(tuple.Item2.Value, oldScale) : (Point?)null,
                        tuple.Item3.HasValue ? RecomputePointAfterZoomChange(tuple.Item3.Value, oldScale) : (Point?)null,
                        RecomputePointAfterZoomChange(tuple.Item4, oldScale));
                }
            }
        }

        private Point RecomputePointAfterZoomChange(Point point, double oldScale)
        {
            Point canvasMidPoint = new Point(canvasWorld.ActualWidth / 2, canvasWorld.ActualHeight / 2);
            return canvasMidPoint + (point - canvasMidPoint) * (Renderer.Scale / oldScale);
        }

        private void buttonStart_Click(object sender, RoutedEventArgs e)
        {            
            StartTimers(ConstructionAvailable);
            ConstructionAvailable = false;
        }

        /// <summary>
        /// Key pressed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainWindow_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.Key == Key.Escape)
            {
                //remove green line drawing new road
                roadPreviewPoints.Clear();
                whichPointToMove = null;
                RoadConstructionStatus = ERoadContructionStatus.NoConstruction;
                Renderer.RedrawPreview();
            }            
        }

        private void Window_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Key == Key.Z && (Keyboard.Modifiers & ModifierKeys.Control) == ModifierKeys.Control)
            {
                World.RemoveLastLaneSegment();
                Renderer.RedrawCanvas();
            }
        }

        private void buttonPause_Click(object sender, RoutedEventArgs e)
        {
            StopTimers();
        }

        private void buttonRestart_Click(object sender, RoutedEventArgs e)
        {
            if (World.SimulationState == ESimulationState.Stopped)
            {
                ConstructionAvailable = true;
                World.ResetLaneSegments();
                World.ClearGenerators();
            }            
            World.ResetCars();
            Renderer.RedrawCanvas();
            World.CloseStatisticsFiles();
            World.SimulationTime = TimeSpan.FromSeconds(0);
            //if (World.SimulationState == ESimulationState.Running)
            //    //show cell statistics window
            //    foreach (Cell cell in World.GetStatisticsCells)
            //        cell.OpenStatisticsWindow();
                
        }

        private void Window_Closed(object sender, EventArgs e)
        {
            World.CloseStatisticsFiles();
        }

        private void buttonClear_Click(object sender, RoutedEventArgs e)
        {
            RestartAll();
        }

        /// <summary>
        /// Clears canvas and all cars. If simulation is started then it will be stopped
        /// </summary>
        private void RestartAll()
        {
            World.CloseStatisticsFiles();
            StopTimers();
            World.ClearAll();
            Renderer.RedrawCanvas();
            ConstructionAvailable = true;
            World.SimulationTime = TimeSpan.FromSeconds(0);
        }

        private void menuNew_Click(object sender, RoutedEventArgs e)
        {
            RestartAll();
            Renderer.RedrawCanvas();
            Focus();
        }

        private void menuSave_Click(object sender, RoutedEventArgs e)
        {
            ignoreClickBecauseActionFromMenuWasExecuted = true;
            System.Windows.Forms.SaveFileDialog saveFileDialog = new System.Windows.Forms.SaveFileDialog();
            saveFileDialog.DefaultExt = ".top";
            saveFileDialog.Filter = "Topology files (.top)|*.top";
            System.Windows.Forms.DialogResult result = saveFileDialog.ShowDialog();
            if (result == System.Windows.Forms.DialogResult.OK)
            {
                try
                {
                    using (Stream output = File.Create(saveFileDialog.FileName))
                    {
                        BinaryFormatter bf = new BinaryFormatter();
                        bf.Serialize(output, World);
                    }
                }
                catch (Exception ex)
                {
                    System.Windows.MessageBox.Show("Unable to save file: " + ex.Message,
                        "Error!", MessageBoxButton.OK, MessageBoxImage.Error);
                }
            }
            Focus();
        }

        private void menuLoad_Click(object sender, RoutedEventArgs e)
        {
            ignoreClickBecauseActionFromMenuWasExecuted = true;
            System.Windows.Forms.OpenFileDialog openFileDialog = new System.Windows.Forms.OpenFileDialog();
            openFileDialog.DefaultExt = ".top";
            openFileDialog.Filter = "Topology files (.top)|*.top";
            System.Windows.Forms.DialogResult result = openFileDialog.ShowDialog();
            if (result == System.Windows.Forms.DialogResult.OK)
            {
                try
                {
                    using (Stream input = File.OpenRead(openFileDialog.FileName))
                    {
                        BinaryFormatter bf = new BinaryFormatter();
                        StopTimers();
                        //World.CloseStatisticsFiles();
                        World = (World)bf.Deserialize(input);
                        World.RestoreWorld();
                        Renderer.ChangeWorld(World);
                        Renderer.RedrawCanvas();
                        ConstructionAvailable = true;
                        World.SimulationTime = TimeSpan.FromSeconds(0);
                        //bind again properties from world instance
                        Binding binding = new Binding("World.SimulationState");
                        binding.Mode = BindingMode.TwoWay;
                        binding.ConverterParameter = "Stopped";
                        binding.Converter = new StateToVisibilityConverter();
                        buttonStart.SetBinding(Button.IsEnabledProperty, binding);
                        binding = new Binding("World.SimulationState");
                        binding.Mode = BindingMode.TwoWay;
                        binding.Converter = new StateToVisibilityConverter();
                        binding.ConverterParameter = "Running";
                        buttonPause.SetBinding(Button.IsEnabledProperty, binding);
                        binding = new Binding("World.SimulationTime");
                        binding.StringFormat = "hh\\:mm\\:ss";
                        textTime.SetBinding(TextBlock.TextProperty, binding);
                    }
                }
                catch (Exception ex)
                {
                    System.Windows.MessageBox.Show("Unable to load file: " + ex.Message,
                        "Error!", MessageBoxButton.OK, MessageBoxImage.Error);
                    return;
                }
            }
            Focus();
        }

        private void menuClose_Click(object sender, RoutedEventArgs e)
        {
            Close();
        }

        private void menuSk_Click(object sender, RoutedEventArgs e)
        {
            if (!menuSk.IsChecked && !menuEn.IsChecked)
                menuEn.IsChecked = true;
            Translate();
        }

        private void Translate()
        {
            Settings.Default.Save();
            TranslationInvoker = new Object();
            comboGenerator.ItemsSource = (List<string>)new GeneratorTypeListProvider().Convert(null, null, null, null);
            Set<ERoadContructionStatus>(ref roadConstructionStatus, roadConstructionStatus, "RoadConstructionStatus", true);
            if (PropertyChanged != null)
                PropertyChanged(this, new PropertyChangedEventArgs("GeneratorType"));
            foreach (var i in World.GetStatisticsCell())
                i.TranslateGraphWindow();
        }

        private void menuEn_Click(object sender, RoutedEventArgs e)
        {
            if (!menuSk.IsChecked && !menuEn.IsChecked)
                menuSk.IsChecked = true;
            Translate();
        }

        private void menuAbout_Click(object sender, RoutedEventArgs e)
        {
            new AboutWindow().ShowDialog();
        }

        private void menuFoo_Click(object sender, RoutedEventArgs e)
        {
            List<Tuple<Point,Point?,Point?,Point>> points = new List<Tuple<Point,Point?,Point?,Point>>();
            points.Add(new Tuple<Point, Point?, Point?, Point>(new Point(0, -4), null, null, new Point(30, -4)));
            points.Add(new Tuple<Point,Point?,Point?,Point>(new Point(0,0),null,null,new Point(30,0)));
            World.BuildRoad(points, 1, 18, 0.000001);
            Renderer.RedrawCanvas();
        }
    }
}
