﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;

namespace TrafficMicroSimulator
{
    [Serializable]
    class StraightLaneSegment : LaneSegment
    {        
        internal override Point ComputePoint(double t)
        {
            return StartPoint + (EndPoint - StartPoint) * t;
        }

        internal override Point ComputePoint(double t, double offset)
        {
            return ComputePoint(t + 1.0 / Length * offset);
        }
        
        internal override double Length
        {
            get { return (EndPoint - StartPoint).Length; }
        }

        internal override double LengthPart(double tStart, double tEnd)
        {
            if (tEnd < tStart)
            {
                throw new ArgumentException("Cannot compute length if end t-parameter is less than start-parameter");
            }
            return (tEnd - tStart) * Length;
        }

        internal override IEnumerable<Tuple<Point, Vector, double>> GetLanePointsAndDirectionVectors(double distance, double tBegin, double tEnd)
        {
            if (tEnd < tBegin)
                throw new ArgumentException("Cannot get lane points backwards");
            if (distance <= 0.0)
                throw new ArgumentException("Distance must be positive");

            Point startTrackPoint = ComputePoint(tBegin);
            Point endTrackPoint = ComputePoint(tEnd);
            int numberOfPoints = (int)Math.Round((endTrackPoint - startTrackPoint).Length / distance) + 1;
            double adjustedDistance = numberOfPoints > 1 ? (endTrackPoint - startTrackPoint).Length / (double)(numberOfPoints - 1) 
                : (endTrackPoint - startTrackPoint).Length;
            double tIncrement = adjustedDistance / Length;

            double t = tBegin;
            for (int i = 0; i < numberOfPoints - 1; i++)
            {
                yield return new Tuple<Point, Vector, double>(ComputePoint(t), EndPoint - StartPoint, t);
                t += tIncrement;
            }
            
            yield return new Tuple<Point, Vector, double>(ComputePoint(tEnd), EndPoint - StartPoint, tEnd);
        }        
    }
}
