﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Windows;


namespace TrafficMicroSimulator
{
    /// <summary>
    /// Represents segment of one lane
    /// </summary>
    [Serializable]
    public abstract class LaneSegment
    {
        protected LaneSegment leftNeighborLane = null;
        /// <summary>
        /// Neighbor lane on the left
        /// </summary>
        public LaneSegment LeftNeighborLane
        {
            get { return leftNeighborLane; }
            set { leftNeighborLane = value; }
        }

        protected LaneSegment rightNeighborLane = null;
        /// <summary>
        /// Neighbor lane on the right
        /// </summary>
        public LaneSegment RightNeighborLane
        {
            get { return rightNeighborLane; }
            set { rightNeighborLane = value; }
        }
        
        private ETransitionType toLeftTransitionType;
        /// <summary>
        /// Lane transition type to left neighbor lane
        /// </summary>
        public ETransitionType ToLeftTransitionType
        {
            get { return toLeftTransitionType; }
            set
            {
                if (LeftNeighborLane == null)
                    throw new ApplicationException(
                        "Cannot set transition type to neighbor lane which not exists");
                else
                    toLeftTransitionType = value;
            }
        }

        private ETransitionType toRightTransitionType;
        /// <summary>
        /// Lane transition type to right neighbor lane
        /// </summary>
        public ETransitionType ToRightTransitionType
        {
            get { return toRightTransitionType; }
            set
            {
                if (RightNeighborLane == null)
                    throw new ApplicationException(
                        "Cannot set transition type to neighbor lane which not exists");
                else
                    toRightTransitionType = value;
            }
        }

        /// <summary>
        /// Max speed in km/h
        /// </summary>
        public uint MaxSpeed { get; set; }

        /// <summary>
        /// Type of road
        /// </summary>
        public ERoadType RoadType { get; set; }

        /// <summary>
        /// Start point of segment lane
        /// </summary>
        public Point StartPoint { get; set; }

        /// <summary>
        /// End point of segment lane
        /// </summary>
        public Point EndPoint { get; set; }

        public EWay Way { get; set; }

        public bool IsEdgeLaneSegmentOnTheRight { get; set; }
        public bool IsEdgeLaneSegmentOnTheLeft { get; set; }
        public bool IsSoleLane { get { return LeftNeighborLane == null && RightNeighborLane == null; } }
        public bool IsLaneBetweenOtherTwoLanes 
        {
            get
            {
                return !IsEdgeLaneSegmentOnTheLeft && !IsEdgeLaneSegmentOnTheRight &&
                    (LeftNeighborLane != null || RightNeighborLane != null);
            }
        }
        
        
        public List<int> rankListOfFollowingCells = new List<int>();

        public CrossingSegmentInfo PreviousSegment { get; set; }
        [NonSerialized]
        protected Cell startCell = null;
        public Cell StartCell { get { return startCell; } set { startCell = value; } }
        public CrossingSegmentInfo FollowingSegment { get; set; }
        [NonSerialized]
        protected Cell endCell = null;
        public Cell EndCell { get { return endCell; } set { endCell = value; } }        
        private List<CrossingSegmentInfo> crossingSegments = new List<CrossingSegmentInfo>();
        internal ReadOnlyCollection<CrossingSegmentInfo> CrossingSegments { get { return crossingSegments.AsReadOnly(); } }
        private List<Tuple<double, DirectionPoint>> directionPoints = new List<Tuple<double, DirectionPoint>>();
        internal ReadOnlyCollection<Tuple<double, DirectionPoint>> DirectionPoints 
            { get { return directionPoints.AsReadOnly(); } }

        [NonSerialized]
        private List<List<Tuple<Cell, EPriority>>> crossingSegmentsVeryNearCells = new List<List<Tuple<Cell, EPriority>>>();

        
        protected Generator generator = null;
        public Generator Generator
        {
            get { return generator; }
            set
            {
                if (value != null && PreviousSegment != null)
                    throw new ApplicationException("Cannot place generator on non-edge segment");
                generator = value;
            }
        }        

        internal double DistanceToPoint(Point worldPoint)
        {
            double startDistance = double.MaxValue;
            double endDistance = double.MaxValue;
            if (PreviousSegment == null)
                startDistance = (StartPoint - worldPoint).Length;
            if (FollowingSegment == null)
                endDistance = (EndPoint - worldPoint).Length;
            return Math.Min(startDistance, endDistance);
        }

        internal double DistanceToStartPoint(Point worldPoint, double maxDistance)
        {
            if (PreviousSegment != null)
                return double.MaxValue;
            if ((StartPoint - worldPoint).Length > maxDistance)
                return double.MaxValue;
            return (StartPoint - worldPoint).Length;
        }

        internal void RemoveNeighborsAndInteractions()
        {
            if (LeftNeighborLane != null)
                LeftNeighborLane.RightNeighborLane = null;
            LeftNeighborLane = null;
            if (RightNeighborLane != null)
                RightNeighborLane.LeftNeighborLane = null;
            RightNeighborLane = null;
            if (PreviousSegment != null)
                PreviousSegment.CrossingSegment.RemoveInteractingSegment(this);
            if (FollowingSegment != null)
                FollowingSegment.CrossingSegment.RemoveInteractingSegment(this);
            PreviousSegment = null;
            FollowingSegment = null; 
            foreach (var i in crossingSegments)
                i.CrossingSegment.RemoveInteractingSegment(this);
            crossingSegments.Clear();            
            crossingSegmentsVeryNearCells.Clear();
            foreach (var i in directionPoints)
                i.Item2.RemoveThisSegment(this);
            directionPoints.Clear();
        }

        private void RemoveInteractingSegment(LaneSegment laneSegment)
        {
            if (FollowingSegment != null && FollowingSegment.CrossingSegment == laneSegment)
                FollowingSegment = null;
            if (PreviousSegment != null && PreviousSegment.CrossingSegment == laneSegment)
                PreviousSegment = null;
            crossingSegments.RemoveAll(i => i.CrossingSegment == laneSegment);             
        }

        public void ResetSegment()
        {
            StartCell = EndCell = null;
            foreach (var i in crossingSegmentsVeryNearCells)
                i.Clear();
            foreach (var i in crossingSegments)
                i.Processed = false;
        }

        internal abstract Point ComputePoint(double t);
        internal abstract Point ComputePoint(double t, double offset);        
        internal abstract IEnumerable<Tuple<Point, Vector, double>> GetLanePointsAndDirectionVectors(double distance, double tBegin, double tEnd);
        internal abstract double Length { get; }
        internal abstract double LengthPart(double tStart, double tEnd);

        internal void AddCrossingSegment(double tParameter, LaneSegment segment, double tParameterOfConnectingLane)
        {
            if (tParameter == 0.0 || tParameter == 1.0)
                throw new ApplicationException("Segments cannot cross other segments at t=0.0 or t=1.0");

            crossingSegments.Add(new CrossingSegmentInfo()
            {
                CrossingSegment = segment,
                TOfCrossingSegment = tParameterOfConnectingLane,
                TOfMyCrossing = tParameter
            });            
            crossingSegmentsVeryNearCells.Add(new List<Tuple<Cell, EPriority>>());            
            crossingSegments.Sort(delegate(CrossingSegmentInfo i1, CrossingSegmentInfo i2)
                { return i1.TOfMyCrossing.CompareTo(i2.TOfMyCrossing); });
        }
        
        internal double? SetCrossingCell(Cell currentCell, Point pointOfCellNearest)
        {            
            //set
            List<int> crossingsIndices = new List<int>();
            for (int i = 0; i < crossingSegments.Count; i++)
            {
                //set cell references to intersections belonging to this point of crossing
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - pointOfCellNearest).Length <= World.cellLength / 2.0)
                {
                    Crossings(currentCell, i, false);
                    //store index of ending road to process its priority
                    if (CrossingSegments[i].TOfCrossingSegment == 1.0)
                        crossingsIndices.Add(i);                    
                }
            }
            SetCellAsDirectionCell(currentCell);
            var minMaxPriority = ProcessPriorities(crossingsIndices);
            if (minMaxPriority == null)
                return null;
            return ComputePriorityMiddle(minMaxPriority);
        }

        private void SetCellAsDirectionCell(Cell currentCell)
        {
            for (int i = 0; i < directionPoints.Count; i++)
            {
                if ((ComputePoint(directionPoints[i].Item1) - currentCell.Location).Length <= World.cellLength / 2.0)                    
                    currentCell.SetDirections(directionPoints[i].Item2, this);
            }
        }

        private double? ProcessPrioritiesMiddle(List<int> crossingIndices)
        {
            Tuple<double, double> priorities = ProcessPriorities(crossingIndices);
            if (priorities == null)
                return null;
            return ComputePriorityMiddle(priorities);
        }

        private double ComputePriorityMiddle(double min, double max)
        {
            return ComputePriorityMiddle(new Tuple<double, double>(min, max));
        }

        private double ComputePriorityMiddle(Tuple<double, double> minmax)
        {
            return minmax.Item1 + (minmax.Item2 - minmax.Item1) / 2;
        }

        private Tuple<double, double> ProcessPriorities(List<int> crossingIndices,
            Tuple<double, double> minMaxPriority)
        {
            return ProcessPriorities(crossingIndices, minMaxPriority.Item1, minMaxPriority.Item2);
        }

        private Tuple<double, double> ProcessPriorities(List<int> crossingIndices, 
            double minPriority = -1.0, double maxPriority = 1.0)
        {
            if (crossingIndices.Count == 0)
                return new Tuple<double,double>(minPriority, maxPriority);
            crossingIndices.Reverse();
            foreach (int i in crossingIndices)
            {
                double middlePriority = ComputePriorityMiddle(minPriority, maxPriority);
                if (crossingSegments[i].Priority == EPriority.High || crossingSegments[i].Priority == EPriority.NotSet)
                {                    
                    crossingSegments[i].MinPriority = middlePriority;
                    crossingSegments[i].MaxPriority = maxPriority;
                    crossingSegments[i].CrossingSegment.FollowingSegment.MinPriority = minPriority;
                    crossingSegments[i].CrossingSegment.FollowingSegment.MaxPriority = middlePriority;
                    minPriority = middlePriority;
                }
                else
                {
                    crossingSegments[i].MinPriority = minPriority;
                    crossingSegments[i].MaxPriority = middlePriority;
                    crossingSegments[i].CrossingSegment.FollowingSegment.MinPriority = middlePriority;
                    crossingSegments[i].CrossingSegment.FollowingSegment.MaxPriority = maxPriority;
                    maxPriority = middlePriority;
                }                
            }
            return new Tuple<double, double>(minPriority, maxPriority);
        }

        private void Crossings(Cell currentCell, int i, bool useRankListOfFollowingCells)
        {
            if (crossingSegments[i].Processed)
                return;
            crossingSegments[i].Processed = true;            

            //edge very near segments                    
            if (crossingSegments[i].TOfCrossingSegment == 1.0)
            {                
                crossingSegments[i].CrossingSegment.EndCell = currentCell;                
            }
            else if (crossingSegments[i].TOfCrossingSegment == 0.0)
            {
                crossingSegments[i].CrossingSegment.StartCell = currentCell;                
                if (useRankListOfFollowingCells)
                {
                    int rankIncrementBecauseOfPreviousSegment = 0;
                    if (PreviousSegment != null)
                        rankIncrementBecauseOfPreviousSegment = PreviousSegment.CrossingSegment.CrossingSegments.Count;
                    currentCell.AddFollowingNonLeafNode(i + rankIncrementBecauseOfPreviousSegment, rankListOfFollowingCells);
                    crossingSegments[i].CrossingSegment.rankListOfFollowingCells = rankListOfFollowingCells.Concat(
                        new List<int>() { i + rankIncrementBecauseOfPreviousSegment }).ToList();
                }
                else
                {
                    currentCell.AddFollowingNonLeafNode(i, new List<int>());
                    crossingSegments[i].CrossingSegment.rankListOfFollowingCells = new List<int>() { i };
                }                
            }
            //crossing very near segments
            else
            {
                crossingSegments[i].CrossingSegment.AddVeryNearCrossingCell(currentCell, this, 
                    crossingSegments[i].TOfCrossingSegment, crossingSegments[i].Priority);                
            }
            foreach (var j in crossingSegmentsVeryNearCells[i])
            {                
                j.Item1.AddCrossingCell(currentCell, j.Item2);             
                currentCell.AddCrossingCell(j.Item1, j.Item2 == EPriority.Low ? EPriority.High : EPriority.Low);
            }
        }

        private void AddVeryNearCrossingCell(Cell currentCell, LaneSegment crossingLaneSegment, double myTParameret, EPriority priority)
        {
            int index = crossingSegments.IndexOf(crossingSegments.Single(i => i.TOfMyCrossing == myTParameret &&
                i.CrossingSegment == crossingLaneSegment));
            crossingSegmentsVeryNearCells[index].Add(new Tuple<Cell, EPriority>(currentCell, priority));            
        }

        internal Cell GetBuiltCell(Point point)
        {            
            for (int i = 0; i < crossingSegments.Count; i++)
            {
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - point).Length <= World.cellLength / 2.0)
                {
                    if (crossingSegments[i].TOfCrossingSegment == 0.0 && crossingSegments[i].CrossingSegment.StartCell != null)
                        return crossingSegments[i].CrossingSegment.StartCell;
                    else if (crossingSegments[i].TOfCrossingSegment == 1.0 && crossingSegments[i].CrossingSegment.EndCell != null)
                        return crossingSegments[i].CrossingSegment.EndCell;
                }
            }
            return null;
        }

        internal void SetEndCell(Cell currentCell)
        {
            EndCell = currentCell;
            for (int i = crossingSegments.Count - 1; i >= 0; i--)
            {
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - EndPoint).Length <= World.cellLength / 2.0)
                    Crossings(currentCell, i, false);
                else
                    break;
            }
            SetCellAsDirectionCell(currentCell);
        }

        internal void SetStartCell(Cell currentCell)
        {
            StartCell = currentCell;            
            for (int i = 0; i < crossingSegments.Count; i++)
            {
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - StartPoint).Length <= World.cellLength / 2.0)
                    Crossings(currentCell, i, true);
                else
                    break;
            }
            SetCellAsDirectionCell(currentCell);
        }        

        internal void AddDirectionPoint(double tOfDirectionPoint)
        {
            //adjust
            if (tOfDirectionPoint == 0.0)
                tOfDirectionPoint = 0.000001;
            if (tOfDirectionPoint == 1.0)
                tOfDirectionPoint = 0.999999;
            DirectionPoint directionPoint = new DirectionPoint();
            AddDirectionPoint(tOfDirectionPoint, directionPoint);
            foreach (LaneSegment ls in new LaneMover(this).GetNextSegment())
            {
                if (ls != this)
                {
                    if ((this.Way == EWay.Normal && ls.Way == EWay.Normal) || 
                        (this.Way == EWay.Reversed && ls.Way == EWay.Reversed))
                        ls.AddDirectionPoint(tOfDirectionPoint, directionPoint);
                    else
                        ls.AddDirectionPoint(1.0 - tOfDirectionPoint, directionPoint);
                }
            }            
        }

        protected void AddDirectionPoint(double tOfDirectionPoint, DirectionPoint directionPoint)
        {
            directionPoint.AddLaneSegmentReference(new DirectionPoint.LaneSegmentReference() 
                { Segment = this, TOfSegment = tOfDirectionPoint });
            directionPoints.Add(new Tuple<double, DirectionPoint>(tOfDirectionPoint, directionPoint));
            directionPoints.Sort(delegate(Tuple<double, DirectionPoint> i1, Tuple<double, DirectionPoint> i2)
            { return i1.Item1.CompareTo(i2.Item1); });
        }

        internal void RemoveDirectionPoint(Tuple<double, DirectionPoint> i)
        {
            directionPoints.Remove(i);
        }

        internal double? GetPriority(EStartEnd eStartEnd)
        {
            if (eStartEnd == EStartEnd.START ?
                PreviousSegment.TOfCrossingSegment == 1.0 : FollowingSegment.TOfCrossingSegment == 0.0)
            {
                LaneSegment followingSegment = eStartEnd == EStartEnd.START ? this : FollowingSegment.CrossingSegment;
                LaneSegment previousSegment = eStartEnd == EStartEnd.START ? PreviousSegment.CrossingSegment : this;
                Tuple<double, double> minMaxPriority = followingSegment.ProcessStartPointPriorities();
                return previousSegment.ProcessEndPointPriorities(minMaxPriority);
            }
            else if (eStartEnd == EStartEnd.END)
            {
                return ProcessEndPointPriorities(FollowingSegment.MinPriority, FollowingSegment.MaxPriority);
            }
            else
                return null;
        }


        private double? ProcessEndPointPriorities(double minPriority, double maxPriority)
        {
            return ProcessEndPointPriorities(new Tuple<double, double>(minPriority, maxPriority));
        }
        private double? ProcessEndPointPriorities(Tuple<double, double> minMaxPriority)
        {
            List<int> crossingNearToStart = new List<int>();
            for (int i = crossingSegments.Count - 1; i >= 0; i--)
            {
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - EndPoint).Length <= World.cellLength / 2.0)
                {
                    if (crossingSegments[i].TOfCrossingSegment == 1.0)
                        crossingNearToStart.Add(i);
                }
                else
                    break;
            }
            Tuple<double,double> newMinMaxPriority = ProcessPriorities(crossingNearToStart, minMaxPriority);
            if (newMinMaxPriority == null)
                return null;
            return ComputePriorityMiddle(newMinMaxPriority);
        }

        private Tuple<double, double> ProcessStartPointPriorities()
        {
            List<int> crossingNearToStart = new List<int>();
            for (int i = 0; i < crossingSegments.Count; i++)
            {
                if ((ComputePoint(crossingSegments[i].TOfMyCrossing) - StartPoint).Length <= World.cellLength / 2.0)
                {
                    if (crossingSegments[i].TOfCrossingSegment == 1.0)
                        crossingNearToStart.Add(i);
                }
                else
                    break;
            }
            Tuple<double, double> minMaxPriority = ProcessPriorities(crossingNearToStart);
            PreviousSegment.MinPriority = PreviousSegment.CrossingSegment.FollowingSegment.MinPriority = minMaxPriority.Item1;
            PreviousSegment.MaxPriority = PreviousSegment.CrossingSegment.FollowingSegment.MaxPriority = minMaxPriority.Item2;
            return minMaxPriority;
        }

        internal void Restore()
        {
            crossingSegmentsVeryNearCells = new List<List<Tuple<Cell, EPriority>>>();
            foreach (var i in crossingSegments) 
                crossingSegmentsVeryNearCells.Add(new List<Tuple<Cell, EPriority>>());
            foreach (var i in crossingSegments)
                i.Processed = false;
        }
    }
}
