﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using BezierPoints = System.Tuple<System.Windows.Point,
    System.Windows.Point, System.Windows.Point, System.Windows.Point>;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Reprezents lane segment in bezier curve shape
    /// </summary>
    [Serializable]
    class BezierLaneSegment : LaneSegment
    {
        public const int LengthComputationSensitivity = 100000;        

        /// <summary>
        /// First control point
        /// </summary>
        public Point StartControlPoint { get; set; }

        /// <summary>
        /// Vector from start point to control start point
        /// </summary>
        public Vector StartControlVector
        {
            get { return StartControlPoint - StartPoint; }
        }

        /// <summary>
        /// Second control point
        /// </summary>
        public Point EndControlPoint { get; set; }

        /// <summary>
        /// Vector from end point to control end point
        /// </summary>
        public Vector EndControlVector
        {
            get { return EndControlPoint - EndPoint; }
        }

        internal override Point ComputePoint(double t)
        {
            return ComputePoint(new BezierPoints(StartPoint, StartControlPoint, EndControlPoint, EndPoint), t);
        }

        internal static Point ComputePoint(BezierPoints b, double t)
        {
            Point p = new Point();
            double u = 1 - t;
            p.X = u * u * u * b.Item1.X +
                3 * u * u * t * b.Item2.X +
                3 * u * t * t * b.Item3.X +
                t * t * t * b.Item4.X;
            p.Y = u * u * u * b.Item1.Y +
                3 * u * u * t * b.Item2.Y +
                3 * u * t * t * b.Item3.Y +
                t * t * t * b.Item4.Y;
            return p;
        }

        internal override Point ComputePoint(double t, double offset)
        {
            double distance = 0.0;
            double previousDistance = 0.0;
            Point previousPoint = ComputePoint(t);
            double i;
            for (i = t; distance < Math.Abs(offset); i = offset > 0 ? 
                i + 1.0 / LengthComputationSensitivity : i - 1.0 / LengthComputationSensitivity)
            {
                Point currentPoint = ComputePoint(i);
                previousDistance = distance;
                distance += (currentPoint - previousPoint).Length;
                previousPoint = currentPoint;
            }
            if (offset - previousDistance < distance - offset)
            {
                if (offset > 0)
                    return ComputePoint(i - 1.0 / LengthComputationSensitivity);
                else
                    return ComputePoint(i + 1.0 / LengthComputationSensitivity);
            }
            else
                return ComputePoint(i);             
        }

        internal override double Length
        {
            get 
            {
                return LengthPart(0.0, 1.0);
            }
        }

        internal override double LengthPart(double tStart, double tEnd)
        {
            return LengthPart(tStart, tEnd, LengthComputationSensitivity);
        }

        protected double LengthPart(double tStart, double tEnd, int lengthComputationSensitivity)
        {
            return LengthPart(tStart, tEnd, 1.0 / lengthComputationSensitivity);
        }

        protected double LengthPart(double tStart, double tEnd, double tStep)
        {
            if (tEnd < tStart)
            {
                throw new ArgumentException("Cannot compute length if end t-parameter is less than start-parameter");
            }

            double distance = 0.0;
            Point previousPoint = ComputePoint(tStart);
            for (double i = tStart; i <= tEnd; i += tStep)
            {
                Point currentPoint = ComputePoint(i);
                distance += (currentPoint - previousPoint).Length;
                previousPoint = currentPoint;
            }
            distance += (ComputePoint(tEnd) - previousPoint).Length;
            return distance;            
        }

        internal override IEnumerable<Tuple<Point, Vector, double>> GetLanePointsAndDirectionVectors(double distance, double tBegin, double tEnd)
        {
            if (tEnd < tBegin)
                throw new ArgumentException("Cannot get lane points backwards");
            if (distance <= 0.0)
                throw new ArgumentException("Distance must be positive");

            //compute step
            double lengthOfEnvelope = 0.0;
            lengthOfEnvelope += (StartControlPoint - StartPoint).Length;
            lengthOfEnvelope += (EndControlPoint - StartControlPoint).Length;
            lengthOfEnvelope += (EndPoint - EndControlPoint).Length;
            lengthOfEnvelope += (StartPoint - EndPoint).Length;
            double tStep = 1.0 / (lengthOfEnvelope / distance) / 10.0;
            
            //compute length of track
            double lengthOfTrack = LengthPart(tBegin, tEnd, tStep);

            //compute number of points
            int numberOfPoints = (int)Math.Round(lengthOfTrack / distance) + 1;

            //compute adjusted distance
            double adjustedDistance = numberOfPoints > 1 ? lengthOfTrack / (double)(numberOfPoints - 1)
                : lengthOfTrack;

            Point previousStepPoint = ComputePoint(tBegin);
            yield return new Tuple<Point, Vector, double>(ComputePoint(tBegin), ComputePoint(tBegin + tStep) - ComputePoint(tBegin), tBegin);

            double t = tBegin;

            for (int i = 0; i < numberOfPoints - 2; i++)
            {
                double distanceFromPreviousPoint = 0.0;
                Point beforePoint;
                double distanceBefore;

                do
                {
                    t += tStep;

                    distanceBefore = distanceFromPreviousPoint;
                    beforePoint = previousStepPoint;

                    distanceFromPreviousPoint += ComputeDistanceToNextPoint(t, ref previousStepPoint);
                } while (distanceFromPreviousPoint < adjustedDistance);

                if (Math.Abs(distanceBefore - adjustedDistance) < Math.Abs(distanceFromPreviousPoint - adjustedDistance))
                {
                    t -= tStep;
                    yield return new Tuple<Point, Vector, double>(beforePoint, ComputePoint(t + tStep) - beforePoint, t);
                    previousStepPoint = beforePoint;
                }
                else
                {
                    yield return new Tuple<Point, Vector, double>(previousStepPoint, ComputePoint(t + tStep) - previousStepPoint, t);
                }                
            }

            yield return new Tuple<Point, Vector, double>(ComputePoint(tEnd), ComputePoint(tEnd + tStep) - ComputePoint(tEnd), tEnd);
        }

        private double ComputeDistanceToNextPoint(double t, ref Point previousPoint)
        {
            Point currentPoint = ComputePoint(t);
            double distance = (previousPoint - currentPoint).Length;
            previousPoint = currentPoint;
            return distance;
        }
    }
}
