﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Class used to move through the road and get cells
    /// </summary>
    public class LaneMover
    {
        /// <summary>
        /// Indicates if direction delimiter lane was crossed
        /// </summary>
        protected bool directionDelimiterLaneCrossed = false;

        public bool DirectionDelimiterLineCrossed { get { return directionDelimiterLaneCrossed; } }
        
        /// <summary>
        /// Current cell
        /// </summary>
        protected Cell currentCell;
        protected LaneSegment currentSegment;

        /// <summary>
        /// Initialize by passing edge cell
        /// </summary>
        /// <param name="edgeCell">Edgecell</param>
        public LaneMover(Cell edgeCell)
        {
            //check if given cell is really on the edge
            if (edgeCell.CellOnTheLeft != null && edgeCell.CellOnTheRight != null)
                throw new ArgumentException("Cell must be on the edge of road", "edgeCell");
            currentCell = edgeCell;
        }
        public LaneMover(LaneSegment segment)
        {
            while (segment.RightNeighborLane != null)
                segment = segment.RightNeighborLane;
            currentSegment = segment;
        }

        /// <summary>
        /// Gets next cell
        /// </summary>
        /// <returns>Next cell in the road on the left/right</returns>
        public IEnumerable<Cell> GetNext()
        {
            //return first cell
            yield return currentCell;
            //is it one-lane road?
            if (currentCell.CellOnTheRight == null && currentCell.CellOnTheLeft == null)
                yield break;
            if (currentCell.CellOnTheLeft == null)
                //one-way road and given cell is on the 'faster' side of road
                directionDelimiterLaneCrossed = true;
            //go throught all lanes
            while ((currentCell.CellOnTheLeft != null && !directionDelimiterLaneCrossed) || (currentCell.CellOnTheRight != null && directionDelimiterLaneCrossed))
            {
                if (!directionDelimiterLaneCrossed)
                {                    
                    if (currentCell.Way != currentCell.CellOnTheLeft.Way)
                        //crossing direction delimiter line
                        directionDelimiterLaneCrossed = true;
                    currentCell = currentCell.CellOnTheLeft;
                }
                else
                {
                    currentCell = currentCell.CellOnTheRight;
                }
                //return current cell
                yield return currentCell;
            }
        }
        public IEnumerable<LaneSegment> GetNextSegment()
        {
            //return first cell
            yield return currentSegment;
            //is it one-lane road?
            if (currentSegment.RightNeighborLane == null && currentSegment.LeftNeighborLane == null)
                yield break;
            if (currentSegment.LeftNeighborLane == null)
                //one-way road and given cell is on the 'faster' side of road
                directionDelimiterLaneCrossed = true;
            //go throught all lanes
            while ((currentSegment.LeftNeighborLane != null && !directionDelimiterLaneCrossed) || 
                (currentSegment.RightNeighborLane != null && directionDelimiterLaneCrossed))
            {
                if (!directionDelimiterLaneCrossed)
                {
                    if (currentSegment.LeftNeighborLane.LeftNeighborLane != null && currentSegment.LeftNeighborLane.LeftNeighborLane.Equals(currentSegment))
                        //crossing direction delimiter line
                        directionDelimiterLaneCrossed = true;
                    currentSegment = currentSegment.LeftNeighborLane;
                }
                else
                {
                    currentSegment = currentSegment.RightNeighborLane;
                }
                //return current cell
                yield return currentSegment;
            }
        }        
    }
}
