﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Collections.ObjectModel;
using System.Numerics;

namespace TrafficMicroSimulator
{
    class InteractionEdgeToLaneMaker
    {        
        public InteractionEdgeToLaneMaker(Point worldPoint, double maxDistance, ReadOnlyCollection<LaneSegment> laneSegments,
            out LaneSegment nearestLaneSegment)
        {
            double minDistance = maxDistance;
            nearestLaneSegment = null;
            foreach (LaneSegment ls in laneSegments)
                CheckForDistance(ls, worldPoint, ref minDistance, ref nearestLaneSegment);            
        }

        public InteractionEdgeToLaneMaker(LaneSegment newSegment, double maxDistance, ReadOnlyCollection<LaneSegment> laneSegments)
        {
            if (newSegment.PreviousSegment != null && newSegment.FollowingSegment != null)
                return;
            double minStartEdgeDistance = maxDistance;
            double minEndEdgeDistance = maxDistance;
            LaneSegment nearestLaneToStart = null;
            LaneSegment nearestLaneToEnd = null;
            foreach (LaneSegment ls in laneSegments)
            {
                if (newSegment.PreviousSegment == null)
                    CheckForDistance(ls, newSegment.StartPoint, ref minStartEdgeDistance, ref nearestLaneToStart);
                if (newSegment.FollowingSegment == null)
                    CheckForDistance(ls, newSegment.EndPoint, ref minEndEdgeDistance, ref nearestLaneToEnd);
            }
            if (nearestLaneToStart != null)
            {
                Tuple<double, Point, double> tuple = DistanceFromLaneToPoint(nearestLaneToStart, newSegment.StartPoint);
                RealizeConnection(newSegment, EStartEnd.START, tuple.Item2, nearestLaneToStart, tuple.Item3);                            
            }
            if (nearestLaneToEnd != null)
            {
                Tuple<double, Point, double> tuple = DistanceFromLaneToPoint(nearestLaneToEnd, newSegment.EndPoint);
                RealizeConnection(newSegment, EStartEnd.END, tuple.Item2, nearestLaneToEnd, tuple.Item3);                
            }
        }

        public DirectionPoint ChosenDirectionPoint = null;
        public InteractionEdgeToLaneMaker(Point point, ReadOnlyCollection<LaneSegment> laneSegments, double maxDistance)
        {
            //search for existing direction point
            double minDistance = maxDistance;
            DirectionPoint nearestDirectionPoint = null;
            foreach (LaneSegment ls in laneSegments)
            {
                foreach (var i in ls.DirectionPoints)
                {
                    if ((point - ls.ComputePoint(i.Item1)).Length < minDistance)
                    {
                        minDistance = (point - ls.ComputePoint(i.Item1)).Length;
                        nearestDirectionPoint = i.Item2;
                    }
                }
            }

            if (nearestDirectionPoint != null)
            {
                ChosenDirectionPoint = nearestDirectionPoint;
                return;
            }
            
            //no near direction point found, try to create new direction point
            minDistance = maxDistance;
            LaneSegment nearestSegment = null;
            foreach (LaneSegment ls in laneSegments)
                CheckForDistance(ls, point, ref minDistance, ref nearestSegment);
            if (nearestSegment == null)
                return;
            Tuple<double, Point, double> tuple = DistanceFromLaneToPoint(nearestSegment, point);
            nearestSegment.AddDirectionPoint(tuple.Item3);
        }

        /// <summary>
        /// Remove nearest direction cell
        /// </summary>
        /// <param name="worldPoint"></param>
        /// <param name="maxDistance"></param>
        /// <param name="laneSegments"></param>
        public InteractionEdgeToLaneMaker(Point worldPoint, double maxDistance, ReadOnlyCollection<LaneSegment> laneSegments)
        {
            
            double minDistance = maxDistance;
            LaneSegment nearestLaneSegment = null;
            foreach (LaneSegment ls in laneSegments)
                CheckForDistance(ls, worldPoint, ref minDistance, ref nearestLaneSegment);
            if (nearestLaneSegment != null)
            {
                foreach (var i in nearestLaneSegment.DirectionPoints)
                {
                    if ((worldPoint - nearestLaneSegment.ComputePoint(i.Item1)).Length < maxDistance)
                    {
                        nearestLaneSegment.RemoveDirectionPoint(i);
                        return;
                    }
                }
            }
        }

        public InteractionEdgeToLaneMaker(Point worldPoint, double maxDistance, 
            ReadOnlyCollection<LaneSegment> laneSegments, out LaneSegment ls1, 
            out LaneSegment ls2, out CrossingSegmentInfo csi1, out CrossingSegmentInfo csi2)
        {            
            ls1 = null;
            ls2 = null;
            csi1 = null;
            csi2 = null;
            double minDistance = maxDistance;
            foreach (var i in laneSegments)
            {
                foreach (var j in i.CrossingSegments)
                {
                    if ((worldPoint - i.ComputePoint(j.TOfMyCrossing)).Length < minDistance &&
                        !j.IsJoin && j.TOfCrossingSegment != 0.0)
                    {
                        minDistance = (worldPoint - i.ComputePoint(j.TOfMyCrossing)).Length;
                        ls1 = i;
                        ls2 = j.CrossingSegment;
                        csi1 = j;
                        if (j.TOfCrossingSegment != 0.0 && j.TOfCrossingSegment != 1.0)
                            csi2 = j.CrossingSegment.CrossingSegments.Single(
                            k => k.CrossingSegment == i && k.TOfMyCrossing == j.TOfCrossingSegment);                        
                    }
                }
            }
        }
        
        private void RealizeConnection(LaneSegment newSegment, EStartEnd eStartEndOfNewSegment, Point pointOfCrossing,
            LaneSegment nearestLane, double tParameterOfNearestLane)
        {
            if (eStartEndOfNewSegment == EStartEnd.NONE)
                throw new ArgumentException("Start-End set to none!");

            double minIncrement = 0.0000000000000001;
            if (tParameterOfNearestLane == 0.0)
                tParameterOfNearestLane += minIncrement;
            if (tParameterOfNearestLane == 1.0)
                tParameterOfNearestLane -= minIncrement;
            

            if (eStartEndOfNewSegment == EStartEnd.START)
            {
                newSegment.StartPoint = pointOfCrossing;
                newSegment.PreviousSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = nearestLane,
                    TOfMyCrossing = 0.0,
                    TOfCrossingSegment = tParameterOfNearestLane,
                };
            }
            else
            {
                newSegment.EndPoint = pointOfCrossing;
                newSegment.FollowingSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = nearestLane,
                    TOfMyCrossing = 1.0,
                    TOfCrossingSegment = tParameterOfNearestLane,
                };
            }

            nearestLane.AddCrossingSegment(tParameterOfNearestLane, newSegment, 
                eStartEndOfNewSegment == EStartEnd.START ? 0.0 : 1.0);
        }
        
        private void CheckForDistance(LaneSegment ls, Point point, ref double minEdgeDistance, ref LaneSegment nearestLane)
        {
            double distance = DistanceFromLaneToPoint(ls, point).Item1;
            if (distance < minEdgeDistance)
            {
                nearestLane = ls;
                minEdgeDistance = distance;
            }
        }

        private Tuple<double, Point, double> DistanceFromLaneToPoint(LaneSegment ls, Point point)
        {
            if (ls is StraightLaneSegment)
            {
                //http://stackoverflow.com/a/3122532
                Vector a_to_p = point - ls.StartPoint;
                Vector a_to_b = ls.EndPoint - ls.StartPoint;
                
                double atb2 = a_to_b.LengthSquared;
                double atp_dot_atb = a_to_p * a_to_b;

                double t = atp_dot_atb / atb2;
                
                Point interactingLaneSegmentLanePoint;
                if (t < 0.0)
                    interactingLaneSegmentLanePoint = ls.StartPoint;
                else if (t > 1.0)
                    interactingLaneSegmentLanePoint = ls.EndPoint;
                else
                    interactingLaneSegmentLanePoint = ls.ComputePoint(t);                 

                return new Tuple<double, Point, double>((point - interactingLaneSegmentLanePoint).Length, interactingLaneSegmentLanePoint, t);
            }
            else if (ls is BezierLaneSegment)
            {
                //http://jazzros.blogspot.sk/2011/03/projecting-point-on-bezier-curve.html                    
                BezierLaneSegment bls = ls as BezierLaneSegment;

                Vector P0 = bls.StartPoint - point;
                Vector P1 = bls.StartControlPoint - point;
                Vector P2 = bls.EndControlPoint - point;
                Vector P3 = bls.EndPoint - point;

                Vector A = P3 - 3 * P2 + 3 * P1 - P0;
                Vector B = 3 * P2 - 6 * P1 + 3 * P0;
                Vector C = 3 * (P1 - P0);

                double Q5 = 3 * A * A;
                double Q4 = 5 * A * B;
                double Q3 = 4 * A * C + 2 * B * B;
                double Q2 = 3 * B * C + 3 * A * P0;
                double Q1 = C * C + 2 * B * P0;
                double Q0 = C * P0;

                List<double> a = new List<double>() { Q0, Q1, Q2, Q3, Q4, Q5 };
                bool wasNotNull = false;
                for (int i = 5; i >= 0; i--)
                {
                    if (a[i] == 0 && !wasNotNull)
                        a.RemoveAt(a.Count - 1);
                    else
                        wasNotNull = true;
                }

                Complex[] roots = RootComputer.ComputeRoots(a.ToArray<double>());
                roots = roots.Where<Complex>(i => i.Imaginary == 0.0).ToArray<Complex>();

                if (roots.Length == 0)
                    throw new ApplicationException("No solution to shortest distance point to given point");

                Point? nearestPoint = null;
                Point nearestPointCandidate = new Point();
                double? t = null;
                foreach (Complex complex in roots)
                {
                    double adjustedComplexReal = complex.Real;
                    if (adjustedComplexReal < 0.0)
                        adjustedComplexReal = 0.0;
                    else if (adjustedComplexReal > 1.0)
                        adjustedComplexReal = 1.0;
                    nearestPointCandidate = bls.ComputePoint(adjustedComplexReal);

                    if (nearestPoint == null ||
                        (nearestPointCandidate - point).Length < (nearestPoint.Value - point).Length)
                    {
                        nearestPoint = nearestPointCandidate;
                        t = adjustedComplexReal;
                    }
                }

                Point interactingLaneSegmentLanePoint = nearestPoint.Value;
                return new Tuple<double, Point, double>((point - interactingLaneSegmentLanePoint).Length, interactingLaneSegmentLanePoint, t.Value);
            }
            else
            {
                throw new ApplicationException("Unknown lane segment");
            }
        }        
    }

}