﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.Windows;

namespace TrafficMicroSimulator
{
    class InteractionEdgeToEdgeMaker
    {
        public InteractionEdgeToEdgeMaker(LaneSegment newSegment, double maxDistance, ReadOnlyCollection<LaneSegment> laneSegments)
        {
            double minStartEdgeDistance = maxDistance;
            double minEndEdgeDistance = maxDistance;
            LaneSegment nearestLaneToStart = null;
            LaneSegment nearestLaneToEnd = null;
            foreach (LaneSegment ls in laneSegments)
            {
                CheckForDistance(ls, newSegment.StartPoint, ref minStartEdgeDistance, ref nearestLaneToStart);
                CheckForDistance(ls, newSegment.EndPoint, ref minEndEdgeDistance, ref nearestLaneToEnd);                
            }
            if (nearestLaneToStart != null)
            {
                ConnectNearestLane(newSegment, nearestLaneToStart, EStartEnd.START);                
            }
            if (nearestLaneToEnd != null)
            {
                ConnectNearestLane(newSegment, nearestLaneToEnd, EStartEnd.END);                
            }
        }
        
        public InteractionEdgeToEdgeMaker(Point worldPoint, double maxDistance, ReadOnlyCollection<LaneSegment> laneSegments)
        {
            double minDistance = maxDistance;
            LaneSegment nearestSegment = null;
            foreach (LaneSegment ls in laneSegments)
                CheckForDistance(ls, worldPoint, ref minDistance, ref nearestSegment);
            if (nearestSegment != null)
            {
                if ((worldPoint - nearestSegment.StartPoint).Length < 
                    (worldPoint - nearestSegment.EndPoint).Length)
                    nearestSegment.Generator = null;                
            }
        }

        private void ConnectNearestLane(LaneSegment newSegment, LaneSegment nearestLane, EStartEnd eStartEndOfNewSegment)
        {
            if (eStartEndOfNewSegment == EStartEnd.NONE)
                throw new ArgumentException("Start End is None!");

            EStartEnd eStartEndOfExistingSegment = GetStartEndOfExistingSegment(
                eStartEndOfNewSegment == EStartEnd.START ? newSegment.StartPoint : newSegment.EndPoint, nearestLane);

            //do not connect anything if there is not place for edge segment
            if (eStartEndOfExistingSegment == EStartEnd.START ? nearestLane.PreviousSegment != null : nearestLane.FollowingSegment != null)
                return;

            //check if road direction ways of nearest segments are compatible
            CheckRoadWaysCompatibility(newSegment, eStartEndOfNewSegment, nearestLane, eStartEndOfExistingSegment);
            
            //slight move of new segment neighbor point
            if (eStartEndOfNewSegment == EStartEnd.START)
                newSegment.StartPoint = DistanceFromEdgeToPoint(nearestLane, newSegment.StartPoint).Item2;
            else
                newSegment.EndPoint = DistanceFromEdgeToPoint(nearestLane, newSegment.EndPoint).Item2;            
                        
            ConnectNearestLane(newSegment, eStartEndOfNewSegment, nearestLane, eStartEndOfExistingSegment);
        }

        private void ConnectNearestLane(LaneSegment newSegment, EStartEnd eStartEndOfNewSegment, LaneSegment nearestLane, EStartEnd eStartEndOfExistingSegment)
        {
            if (eStartEndOfNewSegment == EStartEnd.NONE)
                throw new ArgumentException("Start-End set to NONE");
            if (eStartEndOfExistingSegment == EStartEnd.NONE)
                throw new ArgumentException("Start-End set to NONE");

            if (eStartEndOfExistingSegment == EStartEnd.START)
                nearestLane.PreviousSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = newSegment,
                    TOfMyCrossing = 0.0,
                    TOfCrossingSegment = 1.0,
                };
            else
                nearestLane.FollowingSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = newSegment,
                    TOfMyCrossing = 1.0,
                    TOfCrossingSegment = 0.0,
                };

            if (eStartEndOfNewSegment == EStartEnd.START)
                newSegment.PreviousSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = nearestLane,
                    TOfMyCrossing = 0.0,
                    TOfCrossingSegment = 1.0,
                };
            else
                newSegment.FollowingSegment = new CrossingSegmentInfo()
                {
                    CrossingSegment = nearestLane,
                    TOfMyCrossing = 1.0,
                    TOfCrossingSegment = 0.0,
                };
        }

        private EStartEnd GetStartEndOfExistingSegment(Point point, LaneSegment existingSegment)
        {
            return (point - existingSegment.StartPoint).Length <
                (point - existingSegment.EndPoint).Length ? EStartEnd.START : EStartEnd.END;            
        }

        private void CheckRoadWaysCompatibility(LaneSegment newSegment, EStartEnd eStartEndOfNewSegment, LaneSegment existingSegment,
            EStartEnd eStartEndOfExistingSegment)
        {
            if (eStartEndOfNewSegment == EStartEnd.NONE)
                throw new ArgumentException("Start-End of new segment not defined!");
            if (eStartEndOfExistingSegment == EStartEnd.NONE)
                throw new ArgumentException("Start-End of existing segment not defined!");            

            bool incompatibleWays = false;
            if ((eStartEndOfNewSegment == EStartEnd.START && eStartEndOfExistingSegment == EStartEnd.START) ||
                (eStartEndOfNewSegment == EStartEnd.END && eStartEndOfExistingSegment == EStartEnd.END))
                    incompatibleWays = true;
            
            if (incompatibleWays)
                throw new ApplicationException("Incompatible direction of new segment with following segment!");
        }

        private void CheckForDistance(LaneSegment ls, Point point, ref double minEdgeDistance, ref LaneSegment nearestLane)
        {
            double distance = DistanceFromEdgeToPoint(ls, point).Item1;
            if (distance < minEdgeDistance)
            {
                nearestLane = ls;
                minEdgeDistance = distance;
            }
        }

        private Tuple<double, Point> DistanceFromEdgeToPoint(LaneSegment ls, Point point)
        {
            return new Tuple<double, Point>(
                Math.Min(
                (point - ls.StartPoint).Length,
                (point - ls.EndPoint).Length),
                (point - ls.StartPoint).Length < (point - ls.EndPoint).Length ? 
                ls.StartPoint : ls.EndPoint);
        }
    }
}
