﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
//using System.Drawing;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Windows.Forms.Integration;
using ZedGraph;
using System.Windows;
using System.ComponentModel;
using System.Globalization;
using System.Resources;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Interaction logic for GraphWindow.xaml
    /// </summary>
    public partial class GraphWindow : Window
    {
        ZedGraphControl speedFlow;
        ZedGraphControl speedDensity;
        ZedGraphControl flowDensity;
        int maxSpeed;
        const double toKmh = (World.cellLength / World.SimulationStepTime) * World.mps2kph;
        Point point = new Point();

        public GraphWindow(int maxSpeed, System.Windows.Point cellLocation)
        {
            InitializeComponent();
            DataContext = this;
            this.Title = Properties.Resources.ResourceManager.GetString("GraphWinTitle",
                new CultureInfo(Properties.Settings.Default.Language)) +
                cellLocation.X.ToString("0.000") + " Y: " + cellLocation.Y.ToString("0.000") + ")";
            point = cellLocation;
            speedFlow = new ZedGraphControl();
            firstHost.Child = speedFlow;
            speedDensity = new ZedGraphControl();
            secondHost.Child = speedDensity;
            flowDensity = new ZedGraphControl();
            thirdHost.Child = flowDensity;
            this.maxSpeed = maxSpeed;
            CreateGraphs();
            SetSize();
        }

        public void UpdatePlots(Cell cell)
        {

            double? averageSpeed = cell.GetAverageSpeed();
            int? flow = cell.GetFlow();
            if (averageSpeed == null)
                return;
            double density;
            if (averageSpeed == 0)
                density = 100;
            else
                density = (60 / toKmh) * flow.Value / averageSpeed.Value;

            //GraphPane myPane = zg1.GraphPane;
            GraphPane pane1 = speedFlow.GraphPane;
            GraphPane pane2 = speedDensity.GraphPane;
            GraphPane pane3 = flowDensity.GraphPane;

            ((IPointListEdit)pane1.CurveList[0].Points).Add(flow.Value, averageSpeed.Value * toKmh);
            ((IPointListEdit)pane2.CurveList[0].Points).Add(density, averageSpeed.Value * toKmh);
            ((IPointListEdit)pane3.CurveList[0].Points).Add(density, flow.Value);
            // force redraw
            speedFlow.Invalidate();
            speedDensity.Invalidate();
            flowDensity.Invalidate();
        }

        private void CreateGraphs()
        {
            // get a reference to the GraphPane
            GraphPane myPane1 = speedFlow.GraphPane;
            GraphPane myPane2 = speedDensity.GraphPane;
            GraphPane myPane3 = flowDensity.GraphPane;

            // Set the Titles
            SetTitles();
            speedFlow.GraphPane.XAxis.Scale.Min = 0;
            speedFlow.GraphPane.XAxis.Scale.Max = 60;
            speedFlow.GraphPane.YAxis.Scale.Min = 0;
            speedFlow.GraphPane.YAxis.Scale.Max = (maxSpeed * toKmh) + 10;
            speedDensity.GraphPane.XAxis.Scale.Min = 0;
            speedDensity.GraphPane.XAxis.Scale.Max = 150;
            speedDensity.GraphPane.YAxis.Scale.Min = 0;
            speedDensity.GraphPane.YAxis.Scale.Max = (maxSpeed * toKmh) + 10;
            flowDensity.GraphPane.XAxis.Scale.Min = 0;
            flowDensity.GraphPane.XAxis.Scale.Max = 150;
            flowDensity.GraphPane.YAxis.Scale.Min = 0;
            flowDensity.GraphPane.YAxis.Scale.Max = 60;

            LineItem myCurve1 = myPane1.AddCurve("", new PointPairList(), System.Drawing.Color.Blue, SymbolType.Circle);
            LineItem myCurve2 = myPane2.AddCurve("", new PointPairList(), System.Drawing.Color.Blue, SymbolType.Circle);
            LineItem myCurve3 = myPane3.AddCurve("", new PointPairList(), System.Drawing.Color.Blue, SymbolType.Circle);
            myCurve1.Line.IsVisible = false;
            myCurve2.Line.IsVisible = false;
            myCurve3.Line.IsVisible = false;

            // Tell ZedGraph to refigure the
            // axes since the data have changed
            speedFlow.AxisChange();
            speedDensity.AxisChange();
            flowDensity.AxisChange();
        }

        private void Grid_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            SetSize();
        }

        private void SetSize()
        {
            speedFlow.Location = new System.Drawing.Point(0, 0);
            // Leave a small margin around the outside of the control
            speedFlow.Size = new System.Drawing.Size((int)firstDock.ActualWidth, (int)firstDock.ActualHeight);

            speedDensity.Location = new System.Drawing.Point(0, 0);
            // Leave a small margin around the outside of the control
            speedDensity.Size = new System.Drawing.Size((int)secondDock.ActualWidth, (int)secondDock.ActualHeight);

            flowDensity.Location = new System.Drawing.Point(0, 0);
            // Leave a small margin around the outside of the control
            flowDensity.Size = new System.Drawing.Size((int)thirdDock.ActualWidth, (int)thirdDock.ActualHeight);
        }

        public void Translate()
        {
            this.Title = Properties.Resources.ResourceManager.GetString("GraphWinTitle",
                new CultureInfo(Properties.Settings.Default.Language)) +
                point.X.ToString("0.000") + " Y: " + point.Y.ToString("0.000") + ")";
            // Set the Titles
            SetTitles();
        }

        private void SetTitles()
        {
            // get a reference to the GraphPane
            ResourceManager resourceManager = Properties.Resources.ResourceManager;
            string language = Properties.Settings.Default.Language;
            GraphPane myPane1 = speedFlow.GraphPane;
            GraphPane myPane2 = speedDensity.GraphPane;
            GraphPane myPane3 = flowDensity.GraphPane;
            myPane1.Title.Text = resourceManager.GetString("GraphSpeedFlow", new CultureInfo(language));
            myPane1.XAxis.Title.Text = resourceManager.GetString("Flow", new CultureInfo(language));
            myPane1.YAxis.Title.Text = resourceManager.GetString("Speed", new CultureInfo(language));
            myPane2.Title.Text = resourceManager.GetString("GraphSpeedDensity", new CultureInfo(language));
            myPane2.XAxis.Title.Text = resourceManager.GetString("Density", new CultureInfo(language));
            myPane2.YAxis.Title.Text = resourceManager.GetString("Speed", new CultureInfo(language));
            myPane3.Title.Text = resourceManager.GetString("GraphFlowDensity", new CultureInfo(language));
            myPane3.XAxis.Title.Text = resourceManager.GetString("Density", new CultureInfo(language));
            myPane3.YAxis.Title.Text = resourceManager.GetString("Flow", new CultureInfo(language));
            // Tell ZedGraph to refigure the
            // axes since the data have changed
            speedFlow.Invalidate();
            speedDensity.Invalidate();
            flowDensity.Invalidate();
        }

        private void Window_Closing(object sender, CancelEventArgs e)
        {
            if (!CloseWindow)
                e.Cancel = true;
        }

        public bool CloseWindow = false;
    }
}

