﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Threading;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Represents generator which puts cars on the specified cellColumnPair and count times
    /// </summary>
    [Serializable]
    public abstract class Generator
    {
        /// <summary>
        /// Timer used to generate new vehicles
        /// </summary>
        [NonSerialized]
        protected DispatcherTimer generationTimer;

        /// <summary>
        /// Cell where generator puts cars on
        /// </summary>
        [NonSerialized]
        private Cell conntectedCell = null;
        public Cell ConnectedCell { get { return conntectedCell; } set { conntectedCell = value; } }

        /// <summary>
        /// Segment where generator puts car on
        /// </summary>
        public LaneSegment ConnectedSegment { get; protected set; }

        /// <summary>
        /// Time of last generated car
        /// </summary>
        protected DateTime lastTick;

        /// <summary>
        /// Time where was simulation paused
        /// </summary>
        protected TimeSpan? elapsedTime;

        /// <summary>
        /// Number of tickets which causes to generate new car
        /// </summary>
        protected int tickets;

        /// <summary>
        /// Generator has ticket to generate new car
        /// </summary>
        public bool HasTicket { get { return tickets > 0; } }

        /// <summary>
        /// Simulation speed
        /// </summary>
        private double simulationSpeed = 1;
        
        /// <summary>
        /// Part of initializing generator
        /// </summary>
        /// <param name="connectedCell">Cell instance where generator will put cars on</param>
        protected Generator(Cell connectedCell)
        {
            //set connectedCell and simulation speed
            ConnectedCell = connectedCell;
            //initialize timer
            generationTimer = new DispatcherTimer(DispatcherPriority.Normal);
            generationTimer.Tick += new EventHandler(generationTimer_Tick);
            //initialize tickets
            tickets = 0;
        }

        protected Generator(LaneSegment laneSegment)
        {
            //set connectedCell and simulation speed
            ConnectedSegment = laneSegment;
            //initialize timer
            generationTimer = new DispatcherTimer(DispatcherPriority.Normal);
            generationTimer.Tick += new EventHandler(generationTimer_Tick);
            //initialize tickets
            tickets = 0;
        }

        /// <summary>
        /// Remove one ticket from generator. After calling this
        /// function should be created new car in the World
        /// </summary>
        /// <exception cref="ApplicationException">No tickets left</exception>
        public void GetTicket()
        {
            if (!HasTicket)
                throw new ApplicationException("No tickets left");
            tickets--;
        }

        /// <summary>
        /// Time for generating new vehicle elapsed. Call appropriate method
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void generationTimer_Tick(object sender, EventArgs e)
        {
            //set new random interval
            generationTimer.Interval = GetNextInterval();
            //update lastTick
            lastTick = DateTime.Now;
            //add new ticket to enable generating new car
            tickets++;
        }

        /// <summary>
        /// Generates new random number by a distribution
        /// </summary>
        /// <returns>Random number</returns>
        abstract protected TimeSpan GetNextInterval();

        /// <summary>
        /// Starts generator timer. Decreases timer interval
        /// if simulation was resumed
        /// </summary>
        public void StartTimer()
        {
            //check elapsed time 
            if (elapsedTime != null && generationTimer.Interval > elapsedTime.Value)
            {
                //timer has been already running
                generationTimer.Interval -= elapsedTime.Value;
            }
            else
            {
                //compute new interval
                generationTimer.Interval = GetNextInterval();
            }
            //start timer
            generationTimer.Start();
            //get time of starting timer
            lastTick = DateTime.Now;
        }

        /// <summary>
        /// Stops generator timer
        /// </summary>
        public void StopTimer()
        {
            //stop timer
            generationTimer.Stop();
            //set elapsed time from lasttick
            elapsedTime = DateTime.Now - lastTick;
        }

        /// <summary>
        /// Restore generator after loading World from file
        /// </summary>
        public void RestoreGenerator()
        {
            tickets = 0;
            generationTimer = new DispatcherTimer(DispatcherPriority.Normal);
            generationTimer.Interval = GetNextInterval();
            generationTimer.Tick +=new EventHandler(generationTimer_Tick);
        }
    }
}
