﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Implements generator as gauss generator
    /// </summary>
    [Serializable]
    class GaussGenerator : Generator
    {
        /// <summary>
        /// Mi parameter of gauss distribution
        /// </summary>
        protected TimeSpan mi;

        /// <summary>
        /// Sigma parameter of gauss distribution
        /// </summary>
        protected TimeSpan sigma;

        /// <summary>
        /// Random instance
        /// </summary>
        protected Random random;

        /// <summary>
        /// Algorithm in GetNextInterval method computes two 
        /// results. This is the second one and it is used
        /// in every second call of GetNextInterval method
        /// </summary>
        protected TimeSpan? nextInterval;        
        
        /// <summary>
        /// Initialize generator
        /// </summary>
        /// <param name="connectedCell">Cell connected to</param>
        /// <param name="mi">Mi parameter of distribution</param>
        /// <param name="sigma">Sigma parameter of distribution</param>
        public GaussGenerator(Cell connectedCell, TimeSpan mi, TimeSpan sigma)
            : base(connectedCell)
        {
            this.mi = mi;
            this.sigma = sigma;
            random = new Random(((int)DateTime.Now.Ticks & 0x0000FFFF) + GetHashCode());
            nextInterval = null;
        }

        public GaussGenerator(LaneSegment nearestLaneSegment, TimeSpan mi, TimeSpan sigma)
            : base(nearestLaneSegment)
        {
            this.mi = mi;
            this.sigma = sigma;
            random = new Random(((int)DateTime.Now.Ticks & 0x0000FFFF) + GetHashCode());
            nextInterval = null;
        }

        /// <summary>
        /// Computes random interval in normal distribution using the polar method
        /// </summary>
        /// <returns>Random interval in normal distribution</returns>
        protected override TimeSpan GetNextInterval()
        {
            if (nextInterval != null)
            {
                //it is possible to use already computed nextInterval
                TimeSpan returnNextValue = nextInterval.Value;
                nextInterval = null;
                if (returnNextValue <= TimeSpan.FromMilliseconds(0.0))
                    //prevent having negative interval
                    returnNextValue = TimeSpan.FromMilliseconds(1.0);
                return returnNextValue;
            }
            //compute values by polar method
            double V1, V2, S;
            do
            {
                V1 = 2 * random.NextDouble() - 1;
                V2 = 2 * random.NextDouble() - 1;
                S = V1 * V1 + V2 * V2;
            } while (S >= 1 || S == 0);
            //store first computed value to nextInterval to use it in the next call of this method
            nextInterval = TimeSpan.FromMilliseconds(mi.TotalMilliseconds + 
                sigma.TotalMilliseconds * V1 * Math.Sqrt(-2.0 * Math.Log(S) / S));
            //return second computed value
            TimeSpan valueToReturn = TimeSpan.FromMilliseconds(mi.TotalMilliseconds + 
                sigma.TotalMilliseconds * V2 * Math.Sqrt(-2.0 * Math.Log(S) / S));
            if (valueToReturn <= TimeSpan.FromMilliseconds(0.0))
                //prevent having negative interval
                valueToReturn = TimeSpan.FromMilliseconds(1.0);
            return valueToReturn;
        }
    }
}
