﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using TrafficMicroSimulator.Properties;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Interaction logic for DirectionsWindow.xaml
    /// </summary>
    public partial class DirectionsWindow : Window
    {
        private DirectionPoint chosenDirectionPoint;
        List<TextBox> probabilityTextBoxesOfNormalWaySegments = new List<TextBox>();
        List<TextBox> probabilityTextBoxesOfReversedWaySegments = new List<TextBox>();
        List<TextBox> probabilityTextBoxesOfNormalWaySegmentsPercentage = new List<TextBox>();
        List<TextBox> probabilityTextBoxesOfReversedWaySegmentsPercentage = new List<TextBox>();
        private const double CanvasSize = 100.0;

        protected DirectionsWindow()
        {
            InitializeComponent();
            DataContext = this;
        }

        internal DirectionsWindow(DirectionPoint chosenDirectionPoint)
            : this()
        {
            //find ways to other direction points
            this.chosenDirectionPoint = chosenDirectionPoint;
            if (chosenDirectionPoint.Ways.Count == 0)
                chosenDirectionPoint.FindWaysToOtherDirectionPoints();

            BuildControlsInDirectionsWindow();
        }

        private void BuildControlsInDirectionsWindow()
        {            
            probabilityTextBoxesOfNormalWaySegments.Clear();
            probabilityTextBoxesOfNormalWaySegments.Clear();
            probabilityTextBoxesOfNormalWaySegmentsPercentage.Clear();
            probabilityTextBoxesOfReversedWaySegmentsPercentage.Clear();
            wrapPanelSmerNormal.Children.Clear();
            wrapPanelSmerReversed.Children.Clear();

            //to draw minimaps on the direction points window compute max distance to other direction point
            //also compute sum of probabilities of both directions
            Point pointOfBeginning = chosenDirectionPoint.GetDirectionPoint();
            List<Point> pointOfEndingsDirectionNormal = new List<Point>();
            List<Point> pointOfEndingsDirectionReversed = new List<Point>();
            List<double> probabilitiesDirectionNormal = new List<double>();
            List<double> probabilitiesDirectionReversed = new List<double>();
            foreach (var i in chosenDirectionPoint.Ways)
            {
                if (i.SegmentWay == EWay.Normal)
                {
                    pointOfEndingsDirectionNormal.Add(i.DestinationDirectionPoint.GetDirectionPoint());
                    probabilitiesDirectionNormal.Add(i.Probability);            
                }
                else
                {
                    pointOfEndingsDirectionReversed.Add(i.DestinationDirectionPoint.GetDirectionPoint());
                    probabilitiesDirectionReversed.Add(i.Probability);                    
                }
            }

            //draw minimaps of ways from normal way segments
            if (pointOfEndingsDirectionNormal.Count > 1) //there is no matter to show minimap of only possible way
                DrawMinimapsControls(pointOfBeginning, pointOfEndingsDirectionNormal, probabilitiesDirectionNormal,
                    wrapPanelSmerNormal, probabilityTextBoxesOfNormalWaySegments, 
                    probabilityTextBoxesOfNormalWaySegmentsPercentage);
            if (pointOfEndingsDirectionReversed.Count > 1) //there is no matter to show minimap of only possible way
                DrawMinimapsControls(pointOfBeginning, pointOfEndingsDirectionReversed, probabilitiesDirectionReversed,
                    wrapPanelSmerReversed, probabilityTextBoxesOfReversedWaySegments,
                    probabilityTextBoxesOfReversedWaySegmentsPercentage);

            //set sum textboxes
            UpdateProbabilities();
        }

        private void DrawMinimapsControls(Point pointOfBeginning, List<Point> pointOfEndings, List<double> probabilites,
            WrapPanel wrapPanel, List<TextBox> probabilityTextBoxes, List<TextBox> probabilityTextBoxesPercentage)
        {
            Point averagePoint = Renderer.GetAveragePoint(pointOfEndings.Concat(new List<Point>() { pointOfBeginning }));
            double maxDistance = Renderer.GetMaxDistance(averagePoint,
                pointOfEndings.Concat(new List<Point>() { pointOfBeginning }));
            
            for (int i = 0; i < pointOfEndings.Count; i++)
            {
                //canvas
                Canvas canvas = new Canvas();
                canvas.LayoutTransform = new ScaleTransform(1, -1);
                DrawMinimap(canvas, pointOfBeginning, pointOfEndings, i, averagePoint, maxDistance);
                //border
                Border border = new Border();
                border.BorderBrush = Brushes.Black;
                border.BorderThickness = new Thickness(1.0);
                border.Child = canvas;
                //probability textbox
                TextBox probabilityTextbox = new TextBox();
                probabilityTextbox.Text = probabilites[i].ToString();
                probabilityTextBoxes.Add(probabilityTextbox);
                probabilityTextbox.LostFocus += probabilityTextbox_LostFocus;
                //probability textbox in percentage
                TextBox probabilityTextboxPercentage = new TextBox();
                probabilityTextboxPercentage.IsReadOnly = true;
                probabilityTextBoxesPercentage.Add(probabilityTextboxPercentage);
                //inner wrappanel 
                WrapPanel innerPanel = new WrapPanel();
                innerPanel.Orientation = Orientation.Vertical;
                innerPanel.Children.Add(border);
                innerPanel.Children.Add(probabilityTextBoxes[probabilityTextBoxes.Count - 1]);
                innerPanel.Children.Add(probabilityTextBoxesPercentage[probabilityTextBoxesPercentage.Count - 1]);
                //main wrappanel
                wrapPanel.Children.Add(innerPanel);
            }

        }

        private void DrawMinimap(Canvas canvas, Point pointOfBeginning, List<Point> pointOfEndings,
            int indexOfBiggerCircle, Point averagePoint, double maxDistanceToAveragePoint)
        {
            DrawingVisual dv = new DrawingVisual();
            using (DrawingContext dc = dv.RenderOpen())
            {
                dc.DrawEllipse(Brushes.LightGreen, null,
                    Renderer.FromWorldSystemToCanvasSystem(pointOfBeginning, averagePoint, 
                    maxDistanceToAveragePoint, CanvasSize),
                    3.0, 3.0);

                for (int i = 0; i < pointOfEndings.Count; i++)
                {
                    Brush brush = i == indexOfBiggerCircle ? Brushes.Yellow : Brushes.Green;
                    dc.DrawEllipse(brush, null,
                        Renderer.FromWorldSystemToCanvasSystem(pointOfEndings[i], averagePoint, 
                        maxDistanceToAveragePoint, CanvasSize),
                            3.0, 3.0);
                }
            }
            canvas.Width = CanvasSize;
            canvas.Height = CanvasSize;
            VisualBrush visualBrush = new VisualBrush(dv);
            visualBrush.Viewbox = new Rect(0, 0, CanvasSize, CanvasSize);
            visualBrush.ViewboxUnits = BrushMappingMode.Absolute;
            canvas.Background = visualBrush;
        }
                        
        private void buttonReset_Click(object sender, RoutedEventArgs e)
        {
            chosenDirectionPoint.FindWaysToOtherDirectionPoints();
            BuildControlsInDirectionsWindow();
        }

        private void probabilityTextbox_LostFocus(object sender, RoutedEventArgs e)
        {
            TextBox currentTextBox = (TextBox)sender;
            double result;
            if ((double.TryParse(currentTextBox.Text, out result) == false || result < 0.0) &&
                !currentTextBox.IsFocused)
            {
                MessageBox.Show(GetString("BadFormatDouble"), GetString("Error"),                         
                    MessageBoxButton.OK, MessageBoxImage.Exclamation);
                currentTextBox.Text = 1.ToString();
                return;
            }

            UpdateProbabilities();            
        }

        private string GetString(string resourceStringIdentifier)
        {
            return Properties.Resources.ResourceManager.GetString(resourceStringIdentifier,
                new CultureInfo(Properties.Settings.Default.Language));
        }

        private void UpdateProbabilities()
        {
            //update sums of probabilities
            int counterNormalTextBoxes = 0;
            int counterReversedTextBoxes = 0;
            double sumOfProbabilitiesOfNormal = 0.0;
            double sumOfProbabilitiesOfReversed = 0.0;
            foreach (var way in chosenDirectionPoint.Ways)
            {
                if (way.SegmentWay == EWay.Normal && probabilityTextBoxesOfNormalWaySegments.Count > 0)
                {
                    way.Probability = double.Parse(probabilityTextBoxesOfNormalWaySegments[counterNormalTextBoxes++].Text);
                    sumOfProbabilitiesOfNormal += way.Probability;
                }
                else if (probabilityTextBoxesOfReversedWaySegments.Count > 0)
                {
                    way.Probability = double.Parse(probabilityTextBoxesOfReversedWaySegments[counterReversedTextBoxes++].Text);
                    sumOfProbabilitiesOfReversed += way.Probability;
                }
            }
            textBoxSumSmer1.Text = sumOfProbabilitiesOfNormal.ToString();
            textBoxSumSmer2.Text = sumOfProbabilitiesOfReversed.ToString();

            //update percentage probabilites
            int counterNormalTextBoxesPercentage = 0;
            int counterReversedTextBoxesPercentage = 0;
            foreach (var way in chosenDirectionPoint.Ways)
            {
                if (way.SegmentWay == EWay.Normal && probabilityTextBoxesOfNormalWaySegmentsPercentage.Count > 0)
                {
                    probabilityTextBoxesOfNormalWaySegmentsPercentage[counterNormalTextBoxesPercentage++].Text =
                        (way.Probability / sumOfProbabilitiesOfNormal * 100.0).ToString("00.##") + " %";
                }
                else if (probabilityTextBoxesOfReversedWaySegmentsPercentage.Count > 0)
                {
                    probabilityTextBoxesOfReversedWaySegmentsPercentage[counterReversedTextBoxesPercentage++].Text =
                        (way.Probability / sumOfProbabilitiesOfReversed * 100.0).ToString("00.##") + " %";
                }
            }
        }

        public object TranslationInvoker { get { return new Object(); } set { } }
    }
}
