﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Windows;

namespace TrafficMicroSimulator
{
    [Serializable]
    public class DirectionPoint
    {
        [Serializable]
        internal class LaneSegmentReference
        {
            internal LaneSegment Segment { get; set; }
            internal double TOfSegment { get; set; }                
        }
        private List<LaneSegmentReference> laneSegmentReferences = new List<LaneSegmentReference>();
        internal ReadOnlyCollection<LaneSegmentReference> LaneSegmentReferences 
        { 
            get { return laneSegmentReferences.AsReadOnly(); }
        }
        internal void AddLaneSegmentReference(LaneSegmentReference laneSegmentReference)
        {
            laneSegmentReferences.Add(laneSegmentReference);
        }

        [Serializable]
        internal class Way
        {
            [Serializable]
            internal class Instructions
            {
                internal LaneSegmentReference LaneStartSegmentReference { get; set; }
                private List<EInstruction> instructions = new List<EInstruction>();
                internal ReadOnlyCollection<EInstruction> InstructionList { get { return instructions.AsReadOnly(); } }
                internal void SetInstructionSet(List<EInstruction> instructionSet) { instructions = instructionSet; }
            }

            internal DirectionPoint DestinationDirectionPoint { get; set; }
            private Dictionary<LaneSegmentReference, Instructions> possibleWaysForOneDestination =
                new Dictionary<LaneSegmentReference, Instructions>();
            internal ReadOnlyCollection<Instructions> PossibleWaysForOneDestination 
                { get { return possibleWaysForOneDestination.Values.ToList().AsReadOnly(); } }
            internal void AddPossibleWayForOneDestination(LaneSegmentReference laneSegmentReference, Instructions instructions)
            {
                possibleWaysForOneDestination.Add(laneSegmentReference, instructions);                    
            }
            internal double Probability { get; set; }
            internal EWay SegmentWay { get; set; }
        }

        private Dictionary<Tuple<EWay, DirectionPoint>, Way> ways = new Dictionary<Tuple<EWay, DirectionPoint>, Way>();
        internal ReadOnlyCollection<Way> Ways { get { return ways.Values.ToList().AsReadOnly(); } }                

        internal void FindWaysToOtherDirectionPoints()
        {            
            ways.Clear();

            //for every lane segment find all ways to other direction points
            foreach (var i in laneSegmentReferences)
                FindWaysToOtherDirectionPointsFromLaneSegment(new Stack<EInstruction>(), i, i);            
        }

        private void FindWaysToOtherDirectionPointsFromLaneSegment(Stack<EInstruction> stack, LaneSegmentReference current, LaneSegmentReference first)
        {
            //is other direction point before crossing?
            DirectionPoint otherDirectionPoint = OtherDirectionPointNow(current);
            if (otherDirectionPoint != null)
            {
                //add new destination point if not was found previously
                if (!ways.ContainsKey(new Tuple<EWay, DirectionPoint>(first.Segment.Way, otherDirectionPoint)))
                    ways.Add(new Tuple<EWay, DirectionPoint>(first.Segment.Way, otherDirectionPoint),
                        new Way() { DestinationDirectionPoint = otherDirectionPoint, 
                            Probability = 1, SegmentWay = first.Segment.Way });

                //set instructions
                Way.Instructions instructions = new Way.Instructions()
                {
                    LaneStartSegmentReference = first
                };
                List<EInstruction> instructionsList = stack.ToList();
                instructionsList.Reverse();
                instructions.SetInstructionSet(instructionsList);
                //add this way to destination point
                ways[new Tuple<EWay, DirectionPoint>(first.Segment.Way, otherDirectionPoint)]
                    .AddPossibleWayForOneDestination(first, instructions);
                if (stack.Count > 0)
                    stack.Pop();
                return;
            }
            //is split now?
            LaneSegmentReference continuation, turn;
            if (SplitNow(current, out continuation, out turn))
            {
                stack.Push(EInstruction.Continue);
                FindWaysToOtherDirectionPointsFromLaneSegment(stack, continuation, first);
                stack.Push(EInstruction.Turn);
                FindWaysToOtherDirectionPointsFromLaneSegment(stack, turn, first);
                if (stack.Count > 0)
                    stack.Pop();
                return;
            }
            //end of road
            //go back
            if (stack.Count > 0)
                stack.Pop();
            return;
        }

        private bool SplitNow(LaneSegmentReference current, out LaneSegmentReference continuation, out LaneSegmentReference turn)
        {
            //search for split            
            double? tOfNextSplit = null;
            while (current != null)
            {
                //search in current segment
                int indexOfNextSplit;                
                tOfNextSplit = GetTOfNextSplit(current, out indexOfNextSplit);

                if (tOfNextSplit == null)
                    current = GetFollowingSegment(current.Segment);
                else
                {
                    continuation = new LaneSegmentReference()
                    {
                        Segment = current.Segment,
                        TOfSegment = tOfNextSplit.Value
                    };
                    turn = new LaneSegmentReference()
                    {
                        Segment = current.Segment.CrossingSegments[indexOfNextSplit].CrossingSegment,
                        TOfSegment = current.Segment.CrossingSegments[indexOfNextSplit].TOfCrossingSegment
                    };
                    return true;
                }
            }
            //end of road
            continuation = null;
            turn = null;
            return false;
        }

        private DirectionPoint OtherDirectionPointNow(LaneSegmentReference current)
        {
            //search for direction point
            double? tOfNextDirectionPoint = null;
            double? tOfNextSplit = null;
            while (current != null)
            {
                //search in current segment
                int indexOfNextDirectionPoint, foo;
                tOfNextDirectionPoint = GetTOfNextDirectionPoint(current, out indexOfNextDirectionPoint);
                tOfNextSplit = GetTOfNextSplit(current, out foo);

                if (tOfNextSplit == null && tOfNextDirectionPoint == null)
                    current = GetFollowingSegment(current.Segment);
                else if (tOfNextDirectionPoint == null)
                    return null;
                else if (tOfNextSplit == null)
                    return current.Segment.DirectionPoints[indexOfNextDirectionPoint].Item2;
                else if (tOfNextDirectionPoint.Value > tOfNextSplit.Value)
                    return null;
                else
                    return current.Segment.DirectionPoints[indexOfNextDirectionPoint].Item2;
            }
            //end of road
            return null;
        }

        private LaneSegmentReference GetFollowingSegment(LaneSegment laneSegment)
        {            
            if (laneSegment.FollowingSegment == null)
                return null;
            LaneSegment nextSegment = laneSegment.FollowingSegment.CrossingSegment;
            return new LaneSegmentReference() { Segment = nextSegment, TOfSegment = laneSegment.FollowingSegment.TOfCrossingSegment };
        }

        private double? GetTOfNextSplit(LaneSegmentReference current, out int index)
        {            
            var crossingSegments = current.Segment.CrossingSegments;
            for (int i = 0; i < crossingSegments.Count; i++)                
            {
                if (crossingSegments[i].TOfCrossingSegment == 0.0 &&
                    crossingSegments[i].TOfMyCrossing > current.TOfSegment)
                {
                    index = i;
                    return crossingSegments[i].TOfMyCrossing;
                }
            }
            index = -1;
            return null;
        }

        private double? GetTOfNextDirectionPoint(LaneSegmentReference current, out int index)
        {
            var directionPoints = current.Segment.DirectionPoints;
            for (int i = 0; i < directionPoints.Count; i++)

                if (directionPoints[i].Item1 > current.TOfSegment)
                {
                    index = i;
                    return directionPoints[i].Item1;
                }
            index = -1;
            return null;
        }

        internal Point GetDirectionPoint()
        {
            return laneSegmentReferences[laneSegmentReferences.Count / 2].Segment.ComputePoint(
                laneSegmentReferences[laneSegmentReferences.Count / 2].TOfSegment);
        }

        internal void ClearWays()
        {
            ways.Clear();            
        }

        internal void RemoveThisSegment(LaneSegment ls)
        {
            laneSegmentReferences.RemoveAll(i => i.Segment == ls);
        }

        internal List<EInstruction> GetInstructions(EWay wayOfCurrentCell, out LaneSegment segment, Random random)
        {
            List<Way> waysInProperWay = ways.Values.Where(i => i.SegmentWay == wayOfCurrentCell).ToList();
            if (waysInProperWay.Count == 0)
            {
                segment = null;
                return null;
            }
            double probabilitiesSum = waysInProperWay.Sum(i => i.Probability);
            double vote = random.NextDouble() * probabilitiesSum;
            double probabilitiesSumToDetermineWayToChoose = 0.0;
            foreach (var i in waysInProperWay)
            {
                probabilitiesSumToDetermineWayToChoose += i.Probability;
                if (probabilitiesSumToDetermineWayToChoose > vote)
                {
                    int indexOfRandomSegmentToChoose = random.Next(i.PossibleWaysForOneDestination.Count);
                    segment = i.PossibleWaysForOneDestination[indexOfRandomSegmentToChoose].LaneStartSegmentReference.Segment;
                    return new List<EInstruction>(i.PossibleWaysForOneDestination[indexOfRandomSegmentToChoose].InstructionList);                     
                }
            }
            throw new ApplicationException("Vote too big");
        }
    }
}
