﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Text.RegularExpressions;
using System.Windows.Threading;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Interaction logic for CrossroadWindow.xaml
    /// </summary>
    public partial class CrossroadWindow : Window
    {
        public object TranslationInvoker { get { return new Object(); } set { } }
        protected Dictionary<Vector, KeyValuePair<Cell, Cell>> directions;
        protected Dictionary<Vector, Dictionary<Vector, List<KeyValuePair<Cell, Cell>>>> laneMappings;
        protected Dictionary<Vector, List<KeyValuePair<Vector, int>>> probabilities;
        protected Dictionary<int, Vector> directionIds;
        protected Dictionary<int, List<KeyValuePair<Vector, Vector>>> semaphoreSignals;
        protected Dictionary<DispatcherTimer, int> semaphoreTimers;
        protected Dictionary<int, List<KeyValuePair<bool, TimeSpan>>> semaphoreIntervals;
        protected List<Vector> mainroads;

        public CrossroadWindow(Dictionary<Vector, KeyValuePair<Cell, Cell>> directions,
            Dictionary<Vector, Dictionary<Vector, List<KeyValuePair<Cell, Cell>>>> laneMappings,
            Dictionary<Vector, List<KeyValuePair<Vector, int>>> probabilities,
            Dictionary<int, List<KeyValuePair<Vector, Vector>>> semaphoreSignals,
            Dictionary<DispatcherTimer, int> semaphoreTimers,
            Dictionary<int, List<KeyValuePair<bool, TimeSpan>>> semaphoreIntervals,
            List<Vector> mainroads)
        {
            InitializeComponent();
            DataContext = this;
            this.directions = directions;
            this.laneMappings = laneMappings;
            this.probabilities = probabilities;
            this.semaphoreSignals = semaphoreSignals;
            this.semaphoreTimers = semaphoreTimers;
            this.semaphoreIntervals = semaphoreIntervals;
            this.mainroads = mainroads;
            int counter = 1;
            directionIds = new Dictionary<int, Vector>();
            foreach (var i in directions)
                directionIds.Add(counter++, i.Key);
            //create window
            CreateControls();
        }

        private void CreateControls()
        {
            //create new stack panel for each direction
            StackPanel stackPanel = new StackPanel();
            //set stack panel as child to main stack panel
            mainGrid.Children.Add(stackPanel);
            stackPanel.Orientation = Orientation.Horizontal;
            stackPanel.SetValue(Grid.RowProperty, 1);
            stackPanel.HorizontalAlignment = System.Windows.HorizontalAlignment.Center;
            foreach (var i in directionIds)
            {
                //add controls to stack
                //add label with direction id
                Label label = new Label();
                label.FontSize = 30;
                label.Content = i.Key.ToString() + ":";
                stackPanel.Children.Add(label);
                //add picture with arrow indicating direction
                Canvas canvas = new Canvas();
                double size = 40;
                canvas.Width = canvas.Height = size;
                canvas.LayoutTransform = new ScaleTransform(1, -1);
                DrawingVisual dv = new DrawingVisual();
                using (DrawingContext dc = dv.RenderOpen())
                {
                    //dc.DrawLine(new Pen(Brushes.Red, 1), new Point(0, 0), new Point(20, 20));
                    dc.DrawLine(new Pen(Brushes.Black, 1), new Point(size / 2, size / 2),
                        new Point(size / 2, size / 2) + i.Value * size / 2);
                    dc.DrawRectangle(null, new Pen(Brushes.Black, 1), new Rect(new Point(0,0), new Point(40,40)));
                    dc.DrawRectangle(null, new Pen(Brushes.Blue, 4), new Rect(new Point(20, 20), new Point(20, 20)));
                }
                VisualBrush visualBrush = new VisualBrush(dv);
                visualBrush.Viewbox = new Rect(0, 0, 50, 50);
                visualBrush.ViewboxUnits = BrushMappingMode.Absolute;
                canvas.Background = visualBrush;// new SolidColorBrush(Colors.Red);
                stackPanel.Children.Add(canvas);
            }
        }

        private void confMapping()
        {
            //reset lane mappings
            laneMappings.Clear();
            //create regex to split input string to numbers
            int[] ids = GetNumbersFromString(textMapping.Text);
            //parse input string
            for (int i = 0; i < ids.Length; i += 4)
            {
                if (!laneMappings.ContainsKey(directionIds[ids[i]]))
                    laneMappings.Add(directionIds[ids[i]], new Dictionary<Vector, List<KeyValuePair<Cell, Cell>>>());
                Dictionary<Vector, List<KeyValuePair<Cell, Cell>>> laneMappingOfVectorIdsI = laneMappings[directionIds[ids[i]]];
                if (!laneMappingOfVectorIdsI.ContainsKey(directionIds[ids[i + 2]]))
                    laneMappingOfVectorIdsI.Add(directionIds[ids[i + 2]], new List<KeyValuePair<Cell, Cell>>());
                //get cells
                Cell cell1 = null;
                Cell cell2 = null;
                //get cell from incoming direction
                Cell edgeCell = directions[directionIds[ids[i]]].Key;
                if (edgeCell.DirectionVector.RoundCoordinates() == directionIds[ids[i]])
                    edgeCell = directions[directionIds[ids[i]]].Value;
                int counter = 1;
                foreach (Cell cell in new LaneMover(edgeCell).GetNext())
                {
                    if (counter++ == ids[i + 1])
                    {
                        cell1 = cell;
                        break;
                    }
                }
                //get cell from outgoing direction
                edgeCell = directions[directionIds[ids[i + 2]]].Key;
                if (edgeCell.DirectionVector.RoundCoordinates() != directionIds[ids[i + 2]])
                    edgeCell = directions[directionIds[ids[i + 2]]].Value;
                counter = 1;
                foreach (Cell cell in new LaneMover(edgeCell).GetNext())
                {
                    if (counter++ == ids[i + 3])
                    {
                        cell2 = cell;
                        break;
                    }
                }
                //add to list
                laneMappingOfVectorIdsI[directionIds[ids[i + 2]]].Add(new KeyValuePair<Cell, Cell>(cell1, cell2));
            }
        }

        private void confProbability()
        {
            probabilities.Clear();
            //create regex to split input string to numbers
            int[] ids = GetNumbersFromString(textProbability.Text);
            //parse input string
            for (int i = 0; i < ids.Length; i += 3)
            {
                if (!probabilities.ContainsKey(directionIds[ids[i]]))
                    probabilities.Add(directionIds[ids[i]], new List<KeyValuePair<Vector, int>>());
                List<KeyValuePair<Vector, int>> probabilitiesOfVectorIdI = probabilities[directionIds[ids[i]]];
                int lastProbability = 0;
                if (probabilitiesOfVectorIdI.Count > 0)
                    lastProbability = probabilitiesOfVectorIdI[probabilitiesOfVectorIdI.Count - 1].Value;
                probabilitiesOfVectorIdI.Add(new KeyValuePair<Vector, int>(directionIds[ids[i + 1]], ids[i + 2] + lastProbability));
            }
        }

        private int[] GetNumbersFromString(string stringWithNumbers)
        {
            if (stringWithNumbers == "")
                return new int[0] { };
            Regex regex = new Regex(@"[^\d]+");
            string[] stringIds = regex.Split(stringWithNumbers);
            int[] ids = new int[stringIds.Length];
            for (int i = 0; i < ids.Length; i++)
                ids[i] = int.Parse(stringIds[i]);
            return ids;
        }

        private void confSignals()
        {
            semaphoreSignals.Clear();
            semaphoreTimers.Clear();
            //split string
            int[] ids = GetNumbersFromString(textSignals.Text);
            //parse input string

            for (int i = 0; i < ids.Length; i += 3)
            {
                if (!semaphoreSignals.ContainsKey(ids[i]))
                {   //add new semaphore signal if does not exist
                    semaphoreSignals.Add(ids[i], new List<KeyValuePair<Vector, Vector>>());
                    DispatcherTimer timer = new DispatcherTimer(DispatcherPriority.Normal);
                    semaphoreTimers.Add(timer, ids[i]);
                }
                semaphoreSignals[ids[i]].Add(
                    new KeyValuePair<Vector, Vector>(directionIds[ids[i + 1]], directionIds[ids[i + 2]]));
            }
        }

        private void confSignalTimes()
        {
            semaphoreIntervals.Clear();
            //split string
            int[] ids = GetNumbersFromString(textSignalTimes.Text);
            //parse input string

            for (int i = 0; i < ids.Length; i += 3)
            {
                if (!semaphoreIntervals.ContainsKey(ids[i]))
                {
                    semaphoreIntervals.Add(ids[i], new List<KeyValuePair<bool, TimeSpan>>());
                }
                bool signalRedFalseGreenTrue = false;
                if (ids[i + 1] == 1)
                    signalRedFalseGreenTrue = true;
                semaphoreIntervals[ids[i]].Add(new KeyValuePair<bool, TimeSpan>(signalRedFalseGreenTrue, TimeSpan.FromSeconds(ids[i + 2])));
            }
        }

        private void confMainRoads()
        {
            int[] ids = GetNumbersFromString(textMainRoads.Text);
            if (ids.Count() != 2 && ids.Count() != 0)
            {
                throw new ApplicationException("Two or none main roads must be set");
            }
            if (ids.Count() == 2)
            {
                mainroads.Add(directionIds[ids[0]]);
                mainroads.Add(directionIds[ids[1]]);
            }
        }

        private void btnCancel_Click(object sender, RoutedEventArgs e)
        {
            CancelConfiguration();
            Close();
        }

        private void CancelConfiguration()
        {
            laneMappings.Clear();
            probabilities.Clear();
            semaphoreSignals.Clear();
            semaphoreTimers.Clear();
            semaphoreIntervals.Clear();
            mainroads.Clear();
        }

        /// <summary>
        /// Configure crossroad
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnConfirm_Click(object sender, RoutedEventArgs e)
        {
            bool doNotClose = false;
            try
            {
                confMapping();
            }
            catch (Exception ex)
            {
                CancelConfiguration();
                MessageBox.Show("Impossible to set lane mappings (" + ex.Message + ")", "Error!",
                    MessageBoxButton.OK, MessageBoxImage.Error);
                doNotClose = true;
            }

            try
            {
                confProbability();
            }
            catch (Exception ex)
            {
                CancelConfiguration();
                MessageBox.Show("Impossible to set probabilities (" + ex.Message + ")", "Error!",
                    MessageBoxButton.OK, MessageBoxImage.Error);
                doNotClose = true;
            }

            try
            {
                confMainRoads();
            }
            catch (Exception ex)
            {
                CancelConfiguration();
                MessageBox.Show("Impossible to set main roads (" + ex.Message + ")", "Error!",
                    MessageBoxButton.OK, MessageBoxImage.Error);
                doNotClose = true;
            }

            try
            {
                confSignals();
            }
            catch (Exception ex)
            {
                CancelConfiguration();
                MessageBox.Show("Impossible to set signals (" + ex.Message + ")", "Error!",
                MessageBoxButton.OK, MessageBoxImage.Error);
                doNotClose = true;
            }

            try
            {
                confSignalTimes();
            }
            catch (Exception ex)
            {
                CancelConfiguration();
                MessageBox.Show("Impossible to set signals intervals (" + ex.Message + ")", "Error!",
                MessageBoxButton.OK, MessageBoxImage.Error);
                doNotClose = true;
            }

            if (!doNotClose)
                Close();
        }
    }
}
