﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Threading;

namespace TrafficMicroSimulator
{
    [Serializable]
    public class Crossroad
    {
        /// <summary>
        /// Location of crossroad
        /// </summary>
        public Point Location { get; protected set; }

        /// <summary>
        /// List of main roads
        /// </summary>
        protected List<Vector> mainroads = new List<Vector>();

        /// <summary>
        /// Simulation speed
        /// </summary>
        protected double simulationSpeed;

        /// <summary>
        /// List of directions
        /// </summary>
        protected Dictionary<Vector, KeyValuePair<Cell, Cell>> directions;

        /// <summary>
        /// Lane mappings between roads of crossrad
        /// </summary>
        protected Dictionary<Vector, Dictionary<Vector, List<KeyValuePair<Cell, Cell>>>> laneMappings;

        /// <summary>
        /// Provides for every input direction probabilities of 
        /// selection of every output directions
        /// </summary>
        protected Dictionary<Vector, List<KeyValuePair<Vector, int>>> probabilities = 
            new Dictionary<Vector,List<KeyValuePair<Vector,int>>>();

        protected Dictionary<int, List<KeyValuePair<Vector, Vector>>> semaphoreSignals =
            new Dictionary<int,List<KeyValuePair<Vector,Vector>>>();

        [NonSerialized]
        protected Dictionary<DispatcherTimer, int> semaphoreTimers =
            new Dictionary<DispatcherTimer, int>();

        protected Dictionary<int, List<KeyValuePair<bool, TimeSpan>>> semaphoreIntervals =
            new Dictionary<int, List<KeyValuePair<bool, TimeSpan>>>();

        protected Dictionary<int, int> currentPartOfPeriodOfSignalSwitching = new Dictionary<int, int>();

        /// <summary>
        /// Random instance
        /// </summary>
        protected Random random;

        public int NumberOfRoads { get { return directions.Count; } }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="probabilities"></param>
        /// <param name="entryCells"></param>
        public Crossroad(Point location)
        {
            this.Location = location;
            random = new Random(((int)DateTime.Now.Ticks & 0x0000FFFF) + GetHashCode());
            directions = new Dictionary<Vector, KeyValuePair<Cell, Cell>>();
            laneMappings = new Dictionary<Vector, Dictionary<Vector, List<KeyValuePair<Cell, Cell>>>>();
            probabilities = new Dictionary<Vector, List<KeyValuePair<Vector, int>>>();
        }

        /// <summary>
        /// Randomly choose direction for input direction
        /// </summary>
        /// <param name="entryCell">Entry crossroad cellColumnPair</param>
        /// <returns>Output direction id</returns>
        public Vector GetDirection(Vector entryCellDirection)
        {
            entryCellDirection.Negate();
            //get list of probabilities for input direction
            List<KeyValuePair<Vector, int>> inputDirectionProbabilities = probabilities[entryCellDirection.RoundCoordinates()];
            //get random number which is less than highest probability value. For probability
            //values is rule: index 0 -> probability value of 0. direction, index n -> probability value of n
            //+ probability values of <0,n - 1>  
            int randomNumber = random.Next(inputDirectionProbabilities[inputDirectionProbabilities.Count - 1].Value);
            //by random number choose output direction and return it
            foreach (var i in inputDirectionProbabilities)
                if (randomNumber < i.Value)
                    return i.Key;
            //should not reach this code
            throw new ApplicationException("GetDirection did not found direction");
        }

        public void AddRoad(KeyValuePair<Cell, Cell> keyValuePair)
        {
            //add to directions dictionary
            Vector directionFromCrossroadVector = keyValuePair.Key.DirectionVector;
            if (keyValuePair.Key.FollowingCell == null)
                directionFromCrossroadVector.Negate();
            directions.Add(directionFromCrossroadVector.RoundCoordinates(), keyValuePair);
            //reset probabilities dictionary
            int probability = 0;
            probabilities = new Dictionary<Vector, List<KeyValuePair<Vector, int>>>();
            foreach (var i in directions)
            {
                probabilities.Add(i.Key, new List<KeyValuePair<Vector, int>>());
                foreach (var j in directions)
                {
                    if (j.Key != i.Key)
                    {
                        if (IsOneWayToCrossroad(j.Key))
                            probabilities[i.Key].Add(new KeyValuePair<Vector, int>(j.Key, probability));
                        else
                            probabilities[i.Key].Add(new KeyValuePair<Vector, int>(j.Key, ++probability));
                    }
                }
            }
            //show dialog to set probabilities and lane mappings
            if (NumberOfRoads > 2)
            {
                CrossroadWindow crossroadWindow = new CrossroadWindow(directions, laneMappings, probabilities, 
                    semaphoreSignals, semaphoreTimers, semaphoreIntervals, mainroads);
                crossroadWindow.ShowDialog();
                foreach (var i in semaphoreTimers)
                    i.Key.Tick += new EventHandler(timer_Tick);
                foreach (var i in semaphoreSignals)
                    currentPartOfPeriodOfSignalSwitching.Add(i.Key, -1);

            }
            //set directions and connections
            SetCrossroadInfo();

            //enable semaphore timers
            foreach (var i in semaphoreTimers)
            {
                i.Key.Interval = TimeSpan.FromSeconds(0.1);
                i.Key.Start();
            }
            
        }

        public void SetCrossroadInfo()
        {
            foreach (var i in laneMappings)
            {
                foreach (var j in i.Value)
                {
                    for (int k = 0; k < laneMappings[i.Key][j.Key].Count; k++)
                    {
                        Cell cell = laneMappings[i.Key][j.Key][k].Key;
                        //initialize bool which indicates if current lane is edge lane of current direction
                        //if lane will be split set crossroad information also to split cells
                        bool noLaneOnTheLeft = false;
                        bool noLaneOnTheRight = false;
                        if (cell.CellOnTheLeft == null || cell.CellOnTheLeft.CellOnTheLeft == cell)
                            noLaneOnTheLeft = true;
                        if (cell.CellOnTheRight == null)
                            noLaneOnTheRight = true;
                        while (cell != null)
                        {

                            //set crossroad information
                            cell.NearestCrossroad = this;
                            cell.AddCrossroadDirection(j.Key);
                            if (mainroads.Contains(i.Key))
                                cell.IsMainRoad = true;
                            //if lane was split set crossroad information to split lanes
                            if (noLaneOnTheLeft)
                            {
                                Cell side = cell;
                                while (side.CellOnTheLeft != null && cell.CellOnTheLeft.CellOnTheLeft != cell)
                                {
                                    side = side.CellOnTheLeft;
                                    side.NearestCrossroad = this;
                                    side.AddCrossroadDirection(j.Key);
                                    if (mainroads.Contains(j.Key))
                                        side.IsMainRoad = true;
                                }
                            }
                            if (noLaneOnTheRight)
                            {
                                Cell side = cell;
                                while (side.CellOnTheRight != null)
                                {
                                    side = side.CellOnTheRight;
                                    side.NearestCrossroad = this;
                                    side.AddCrossroadDirection(j.Key);
                                    if (mainroads.Contains(j.Key))
                                        side.IsMainRoad = true;
                                }
                            }
                            //move to next cell and try to go to edge road when number
                            //of lanes is decreased
                            while (cell.PreviousCell == null && cell.CellOnTheRight != null)
                            {
                                cell = cell.CellOnTheRight;
                            }
                            if (cell.PreviousCell == null)
                                break;
                            cell = cell.PreviousCell;
                        }
                    }
                }
            }
        }
        
        /// <summary>
        /// Signal switching
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void timer_Tick(object sender, EventArgs e)
        {
            DispatcherTimer timer = sender as DispatcherTimer;
            //set new interval of current light
            if (++currentPartOfPeriodOfSignalSwitching[semaphoreTimers[timer]] == semaphoreIntervals[semaphoreTimers[timer]].Count)
                currentPartOfPeriodOfSignalSwitching[semaphoreTimers[timer]] = 0;
            timer.Interval = semaphoreIntervals[semaphoreTimers[timer]]
                [currentPartOfPeriodOfSignalSwitching[semaphoreTimers[timer]]].Value;
        }

        public bool IsGreenLight(Vector currentVector, Vector destinationVector)
        {
            //find transition group id
            int transitionGroupId = -1;
            foreach (var i in semaphoreSignals)
            {
                if (semaphoreSignals[i.Key].Contains(new KeyValuePair<Vector, Vector>(currentVector, destinationVector)))
                {
                    transitionGroupId = i.Key;
                    break;
                }
            }
            if (transitionGroupId == -1)
                //crossroad does not have semaphores
                return true;
            //get light status of this transition group
            if (semaphoreIntervals[transitionGroupId][currentPartOfPeriodOfSignalSwitching[transitionGroupId]].Key)
                return true;
            else
                return false;
        }

        /// <summary>
        /// Use before cells of different roads are connected
        /// </summary>
        /// <param name="vector"></param>
        /// <returns></returns>
        private bool IsOneWayToCrossroad(Vector vector)
        {
            return (directions[vector.RoundCoordinates()].Key.FollowingCell == null && directions[vector.RoundCoordinates()].Value.FollowingCell == null);
        }

        public Cell GetConnectedCell(Cell currentCell, Vector directionVectorOfNextRoad)
        {
            Vector currentCellNegatedVector = currentCell.DirectionVector;
            currentCellNegatedVector.Negate();
            foreach (var i in laneMappings[currentCellNegatedVector.RoundCoordinates()][directionVectorOfNextRoad.RoundCoordinates()]
                .Where(j => j.Key == currentCell))
            {
                return i.Value;
            }
            return null;
            //throw new ApplicationException("Could not find connected cell");
        }

        /// <summary>
        /// Get edge arriving cell of crossroad in given direction
        /// </summary>
        /// <param name="vector"></param>
        /// <returns></returns>
        public Cell GetCellOfDirectionOut(Vector vector)
        {
            KeyValuePair<Cell,Cell> pair = directions[vector];
            if (pair.Key.DirectionVector.RoundCoordinates() == vector)
                return pair.Key;
            else
                return pair.Value;
        }

        /// <summary>
        /// Get edge arriving cell of crossroad in given direction
        /// </summary>
        /// <param name="vector"></param>
        /// <returns></returns>
        public Cell GetCellsOfDirectionIn(Vector vector)
        {
            KeyValuePair<Cell, Cell> pair = directions[vector];
            if (pair.Key.DirectionVector.RoundCoordinates() != vector)
                return pair.Key;
            else
                return pair.Value;
        }

        /// <summary>
        /// Gets next direction 
        /// </summary>
        /// <param name="currentVector"></param>
        /// <returns></returns>
        public Vector GetNextDirectionToRight(Vector currentVector)
        {
            List<Vector> vectorsOnTheRight = new List<Vector>();
            List<Vector> vectorsOnTheLeft = new List<Vector>();
            //divide all direction vectors which are on the left and right side of current direction vector
            foreach (var i in directions)
            {
                if (i.Key == currentVector.RoundCoordinates())
                    continue;
                if (CountDeterminant(currentVector, i.Key) >= 0)
                    vectorsOnTheRight.Add(i.Key);
                else
                    vectorsOnTheLeft.Add(i.Key);
            }
            //if there is a road on the right side choose the nearest
            double distance;
            if (vectorsOnTheRight.Count > 0)
            {
                distance = double.MaxValue;
                Vector nearestRoadDirectionVector = new Vector();
                foreach (var i in vectorsOnTheRight)
                {
                    if (((Location + currentVector) - (Location + i)).Length < distance)
                    {
                        distance = ((Location + currentVector) - (Location + i)).Length;
                        nearestRoadDirectionVector = i;
                    }
                }
                return nearestRoadDirectionVector;
            }
            //there is only road on the left side. Choose the furthest
            distance = 0;
            Vector furthestRoadDirectionVector = new Vector();
            foreach (var i in vectorsOnTheLeft)
            {
                if (((Location + currentVector) - (Location + i)).Length > distance)
                {
                    distance = ((Location + currentVector) - (Location + i)).Length;
                    furthestRoadDirectionVector = i;
                }
            }
            return furthestRoadDirectionVector;
        }

        /// <summary>
        /// Count determinant of two vectors
        /// </summary>
        /// <param name="first"></param>
        /// <param name="second"></param>
        /// <returns></returns>
        private double CountDeterminant(Vector first, Vector second)
        {
            return first.X * second.Y - first.Y * second.X;
        }

        public bool IsDirectionBetweenTwoOtherDirections(Vector crossingCarVector, 
            Vector directionOfMyCurrentRoad, Vector directionOfMyDestinationRoad)
        {
            Vector currentVector = directionOfMyCurrentRoad.RoundCoordinates();
            while ((currentVector = GetNextDirectionToRight(currentVector)) != directionOfMyDestinationRoad.RoundCoordinates())
            {
                if (currentVector == crossingCarVector.RoundCoordinates())
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// Check crossroad if it has main roads
        /// </summary>
        /// <param name="mainRoadDirectionVectors"></param>
        /// <returns></returns>
        public bool HasMainRoads(out KeyValuePair<Vector,Vector> mainRoadDirectionVectors)
        {
            mainRoadDirectionVectors = new KeyValuePair<Vector, Vector>();
            bool hasMainRoads = false;
            foreach (var i in directions)
            {
                if (GetCellsOfDirectionIn(i.Key).IsMainRoad)
                {
                    if (!hasMainRoads)
                    {
                        mainRoadDirectionVectors = new KeyValuePair<Vector, Vector>(i.Key, new Vector());
                        hasMainRoads = true;
                    }
                    else
                    {
                        mainRoadDirectionVectors = new KeyValuePair<Vector, Vector>(mainRoadDirectionVectors.Key, i.Key);
                        break;
                    }
                }
            }
            return hasMainRoads;
        }

        public IEnumerable<Cell> GetCrossroadOutCells()
        {
            return directions.Select(d => GetCellOfDirectionOut(d.Key));
        }

        public bool HasSemaphores()
        {
            return semaphoreSignals.Count > 0;
        }

        /// <summary>
        /// Restore crossroads after loading file
        /// </summary>
        public void RestoreCrossroad()
        {
            semaphoreTimers = new Dictionary<DispatcherTimer, int>();
            currentPartOfPeriodOfSignalSwitching.Clear();
            foreach (var i in semaphoreSignals)
            {
                if (!semaphoreTimers.ContainsValue(i.Key))
                {
                    DispatcherTimer semaphoreTimer = new DispatcherTimer(DispatcherPriority.Normal);
                    semaphoreTimer.Tick += new EventHandler(timer_Tick);
                    currentPartOfPeriodOfSignalSwitching.Add(i.Key, -1);
                    semaphoreTimer.Interval = TimeSpan.FromSeconds(0.1);
                    semaphoreTimers.Add(semaphoreTimer, i.Key);
                }
            }
            foreach (var i in semaphoreTimers)
                i.Key.Start();
        }
    }
}
