﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Collections.ObjectModel;
using System.Numerics;
using System.Collections;
using BezierPoints = System.Tuple<System.Windows.Point,
    System.Windows.Point, System.Windows.Point, System.Windows.Point>;

namespace TrafficMicroSimulator
{
    class CrossingInteractionMaker
    {
        const double eps = 0.000001;

        public CrossingInteractionMaker(LaneSegment newSegment, ReadOnlyCollection<LaneSegment> laneSegments)
        {
            foreach (LaneSegment ls in laneSegments)
            {
                List<Tuple<double, double>> intersections = new List<Tuple<double, double>>();
                if (ls is StraightLaneSegment && newSegment is StraightLaneSegment)
                {
                    intersections.Add(CrossingInteractionTwoStraightsLanes(newSegment, ls));
                }
                else if ((ls is StraightLaneSegment && newSegment is BezierLaneSegment) ||
                    (ls is BezierLaneSegment && newSegment is StraightLaneSegment))
                {
                    intersections.AddRange(CrossingInteractionStraightAndBezier(newSegment, ls));
                }
                else if (ls is BezierLaneSegment && newSegment is BezierLaneSegment)
                {
                    intersections.AddRange(CrossingInteractionTwoBeziers(newSegment as BezierLaneSegment, ls as BezierLaneSegment));
                }
                else
                {
                    throw new ApplicationException("Unknown lane segment");
                }
                foreach (var i in ReduceNearSolutions(intersections.Where<Tuple<double, double>>
                    (j => j.Item1 >= 0.0 && j.Item1 <= 1.0 && j.Item2 >= 0.0 && j.Item2 <= 1.0), 
                    newSegment, ls))
                {
                    newSegment.AddCrossingSegment(i.Item1, ls, i.Item2);
                    ls.AddCrossingSegment(i.Item2, newSegment, i.Item1);
                }
            }
        }
                
        private IEnumerable<Tuple<double, double>> CrossingInteractionTwoBeziers(BezierLaneSegment bls1, BezierLaneSegment bls2)
        {
            return CrossingInteractionTwoBeziers(new BezierPoints(bls1.StartPoint, bls1.StartControlPoint,
                bls1.EndControlPoint, bls1.EndPoint), 0.0, 1.0, new BezierPoints(bls2.StartPoint,
                    bls2.StartControlPoint, bls2.EndControlPoint, bls2.EndPoint), 0.0, 1.0);
        }

        private IEnumerable<Tuple<double, double>> CrossingInteractionTwoBeziers(BezierPoints bls1, double tl1, double tu1,
            BezierPoints bls2, double tl2, double tu2)
        {
            //http://stackoverflow.com/a/4041286
            Rect bls1Box = MakeBoundingBox(bls1);
            Rect bls2Box = MakeBoundingBox(bls2);
            if (!bls1Box.IntersectsWith(bls2Box))
                return new List<Tuple<double, double>>();
            //if (bls1Box.Height * bls1Box.Width + bls2Box.Height * bls2Box.Width < eps)
            if (bls1Box.Height < eps && bls1Box.Width < eps && bls2Box.Height < eps && bls2Box.Width < eps)
                return new List<Tuple<double, double>>() { new Tuple<double, double>((tl1 + tu1) / 2, (tl2 + tu2) / 2) };
            Tuple<BezierPoints, BezierPoints> bls1ab = SplitBezier(bls1);
            Tuple<BezierPoints, BezierPoints> bls2ab = SplitBezier(bls2);
            double tm1 = tl1 + (tu1 - tl1) / 2;
            double tm2 = tl2 + (tu2 - tl2) / 2;
            return CrossingInteractionTwoBeziers(bls1ab.Item1, tl1, tm1, bls2ab.Item1, tl2, tm2).Concat<Tuple<double, double>>(
                CrossingInteractionTwoBeziers(bls1ab.Item1, tl1, tm1, bls2ab.Item2, tm2, tu2)).Concat<Tuple<double, double>>(
                CrossingInteractionTwoBeziers(bls1ab.Item2, tm1, tu1, bls2ab.Item1, tl2, tm2)).Concat<Tuple<double, double>>(
                CrossingInteractionTwoBeziers(bls1ab.Item2, tm1, tu1, bls2ab.Item2, tm2, tu2));
        }

        private IEnumerable<Tuple<double, double>> ReduceNearSolutions(IEnumerable<Tuple<double, double>> enumerable, 
            LaneSegment segment1, LaneSegment segment2)
        {
            List<List<Tuple<double, double>>> reduced = new List<List<Tuple<double, double>>>();
            foreach (var i in enumerable)
            {
                bool wasReduction = false;
                for (int j = 0; j < reduced.Count; j++)
                {
                    for (int k = 0; k < reduced[j].Count; k++)
                    {
                        if ((segment1.ComputePoint(i.Item1) - segment1.ComputePoint(reduced[j][k].Item1)).Length < World.MinCellLength * 0.1 &&
                            (segment2.ComputePoint(i.Item2) - segment2.ComputePoint(reduced[j][k].Item2)).Length < World.MinCellLength * 0.1)
                        {
                            wasReduction = true;
                            reduced[j].Add(new Tuple<double, double>(i.Item1, i.Item2));
                            break;
                        }
                    }
                    if (wasReduction)
                        break;
                }
                if (!wasReduction)
                {
                    reduced.Add(new List<Tuple<double, double>>());
                    reduced[reduced.Count - 1].Add(i);
                }
            }

            List<Tuple<double, double>> reducedToReturn = new List<Tuple<double, double>>();
            foreach (var i in reduced)
            {
                Tuple<double, double> toAdd = new Tuple<double, double>(0.0, 0.0);
                foreach (var j in i)
                    toAdd = new Tuple<double, double>(toAdd.Item1 + j.Item1, toAdd.Item2 + j.Item2);
                toAdd = new Tuple<double, double>(toAdd.Item1 / i.Count, toAdd.Item2 / i.Count);
                if (segment1.PreviousSegment != null && segment1.PreviousSegment.CrossingSegment == segment2 &&
                    (segment1.ComputePoint(toAdd.Item1) - segment1.StartPoint).Length < World.MinCellLength * 0.1)
                    continue;
                if (segment1.FollowingSegment != null && segment1.FollowingSegment.CrossingSegment == segment2 &&
                    (segment1.ComputePoint(toAdd.Item1) - segment1.EndPoint).Length < World.MinCellLength * 0.1)
                    continue;
                if (segment2.PreviousSegment != null && segment2.PreviousSegment.CrossingSegment == segment1 &&
                    (segment2.ComputePoint(toAdd.Item2) - segment2.StartPoint).Length < World.MinCellLength * 0.1)
                    continue;
                if (segment2.FollowingSegment != null && segment2.FollowingSegment.CrossingSegment == segment1 &&
                    (segment2.ComputePoint(toAdd.Item2) - segment2.EndPoint).Length < World.MinCellLength * 0.1)
                    continue;
                reducedToReturn.Add(toAdd);
            }
            return reducedToReturn;
        }

        private Tuple<BezierPoints, BezierPoints> SplitBezier(BezierPoints bls)
        {
            Point P12 = bls.Item1 + (bls.Item2 - bls.Item1) / 2;
            Point P23 = bls.Item2 + (bls.Item3 - bls.Item2) / 2;
            Point P34 = bls.Item3 + (bls.Item4 - bls.Item3) / 2;
            Point P12_23 = P12 + (P23 - P12) / 2;
            Point P23_34 = P23 + (P34 - P23) / 2;
            Point P1223_2334 = P12_23 + (P23_34 - P12_23) / 2;
            return new Tuple<BezierPoints, BezierPoints>(new BezierPoints(bls.Item1, P12, P12_23, P1223_2334),
                new BezierPoints(P1223_2334, P23_34, P34, bls.Item4));
        }



        private Rect MakeBoundingBox(BezierPoints bls)
        {
            return new Rect(new Point(Math.Min(bls.Item1.X, Math.Min(bls.Item2.X, Math.Min(bls.Item3.X, bls.Item4.X))),
                                      Math.Min(bls.Item1.Y, Math.Min(bls.Item2.Y, Math.Min(bls.Item3.Y, bls.Item4.Y)))),
                            new Point(Math.Max(bls.Item1.X, Math.Max(bls.Item2.X, Math.Max(bls.Item3.X, bls.Item4.X))),
                                      Math.Max(bls.Item1.Y, Math.Max(bls.Item2.Y, Math.Max(bls.Item3.Y, bls.Item4.Y)))));
        }

        private IEnumerable<Tuple<double, double>> CrossingInteractionStraightAndBezier(LaneSegment ls1, LaneSegment ls2)
        {
            BezierLaneSegment bls = ls1 is BezierLaneSegment ? ls1 as BezierLaneSegment : ls2 as BezierLaneSegment;
            StraightLaneSegment sls = ls1 is StraightLaneSegment ? ls1 as StraightLaneSegment : ls2 as StraightLaneSegment;
            
            double slsAx = sls.StartPoint.X;
            double slsAy = sls.StartPoint.Y;
            double slsBx = sls.EndPoint.X;
            double slsBy = sls.EndPoint.Y;
            double blsAx = bls.StartPoint.X;
            double blsAy = bls.StartPoint.Y;
            double blsBx = bls.StartControlPoint.X;
            double blsBy = bls.StartControlPoint.Y;
            double blsCx = bls.EndControlPoint.X;
            double blsCy = bls.EndControlPoint.Y;
            double blsDx = bls.EndPoint.X;
            double blsDy = bls.EndPoint.Y;

            //make translation of coord system - move startpoint of straight segment to [0,0]
            slsBx -= slsAx;
            slsBy -= slsAy;
            blsAx -= slsAx;
            blsAy -= slsAy;
            blsBx -= slsAx;
            blsBy -= slsAy;
            blsCx -= slsAx;
            blsCy -= slsAy;
            blsDx -= slsAx;
            blsDy -= slsAy;
            slsAx = 0;
            slsAy = 0;

            //make rotation
            //compute angle of the second point of line
            double angle = Math.Atan2(slsBx, slsBy);
            //move slsBx to y axis
            slsBy = new Vector(slsBx, slsBy).Length;
            slsBx = 0;
            //rotate all points of Bezier Curve
            Rotate(ref blsAx, ref blsAy, angle);
            Rotate(ref blsBx, ref blsBy, angle);
            Rotate(ref blsCx, ref blsCy, angle);
            Rotate(ref blsDx, ref blsDy, angle);

            //find when Bezier intersects y-axis
            // + a
            // -3 a t + 3 b t
            // + 3 a t^2 - 6 b t^2 + 3 c t^2
            // -a t^3 + 3 b t^3 - 3 c t^3 + d t^3
            IEnumerable<Complex> roots = RootComputer.ComputeRoots(new double[] 
            { 
                blsAx,
                -3 * blsAx + 3 * blsBx,
                3 * blsAx - 6 * blsBx + 3 * blsCx,
                -blsAx + 3 * blsBx - 3 *blsCx + blsDx
            });
            roots = roots.Where<Complex>(i => i.Imaginary == 0);

            List<Tuple<double, double>> toReturn = new List<Tuple<double, double>>();
            //compute points crossing y axis
            foreach (Complex root in roots)
            {
                Point p = BezierLaneSegment.ComputePoint(new BezierPoints(new Point(blsAx, blsAy),
                    new Point(blsBx, blsBy), new Point(blsCx, blsCy), new Point(blsDx, blsDy)), root.Real);
                double tSls = p.Y / slsBy;
                if (ls1 is StraightLaneSegment && ls2 is BezierLaneSegment)
                    toReturn.Add(new Tuple<double, double>(tSls, root.Real));
                else if (ls1 is BezierLaneSegment && ls2 is StraightLaneSegment)
                    toReturn.Add(new Tuple<double, double>(root.Real, tSls));
                else
                    throw new ApplicationException("Unknown lane segment");
            }

            return toReturn;
        }

        private void Rotate(ref double x, ref double y, double angle)
        {
            double x_ = x * Math.Cos(angle) - y * Math.Sin(angle);
            double y_ = x * Math.Sin(angle) + y * Math.Cos(angle);
            x = x_;
            y = y_;
        }

        private Tuple<double, double> CrossingInteractionTwoStraightsLanes(LaneSegment newSegment, LaneSegment otherSegment)
        {
            //from parametric expression of line segment: [x,y] = A + B*t
            double Ax = newSegment.StartPoint.X;
            double Bx = newSegment.EndPoint.X - newSegment.StartPoint.X;
            double Cx = otherSegment.StartPoint.X;
            double Dx = otherSegment.EndPoint.X - otherSegment.StartPoint.X;
            double Ay = newSegment.StartPoint.Y;
            double By = newSegment.EndPoint.Y - newSegment.StartPoint.Y;
            double Cy = otherSegment.StartPoint.Y;
            double Dy = otherSegment.EndPoint.Y - otherSegment.StartPoint.Y;

            //get out t2
            double old_dx = Dx;
            double old_dy = Dy;
            Ax *= old_dy; Bx *= old_dy; Cx *= old_dy; Dx *= old_dy;
            Ay *= -old_dx; By *= -old_dx; Cy *= -old_dx; Dy *= -old_dx;

            //make one equation by addition of two equations
            double Axy = Ax + Ay;
            double Bxy = Bx + By;
            double Cxy = Cx + Cy;

            if (Bxy == 0 || Dx == 0)
                return new Tuple<double, double>(double.PositiveInfinity, double.PositiveInfinity);
            //compute t1
            double t1 = (Cxy - Axy) / Bxy;
            //compute t2
            double t2 = (Ax + Bx * t1 - Cx) / Dx;

            return new Tuple<double, double>(t1, t2);           
        }
    }
}
