﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.IO;
using System.Collections.ObjectModel;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Represents cellColumnPair
    /// </summary>
    public class Cell
    {
        protected class CarSpeedDistance { public int Speed; public int Distance; public double Priority;}

        protected List<int> carSpeeds = new List<int>();
        protected List<DateTime> timesOfCarMoves = new List<DateTime>();        
        protected Car lastCar = null;

        protected List<Cell> crossingCells = new List<Cell>();
        protected List<EPriority> crossingCellsPriorites = new List<EPriority>();
        public ReadOnlyCollection<EPriority> CrossingCellsPriorities { get { return crossingCellsPriorites.AsReadOnly(); } }
        public ReadOnlyCollection<Cell> CrossingCells { get { return crossingCells.AsReadOnly(); } }

        protected Dictionary<Car, CarSpeedDistance> nearCars = new Dictionary<Car, CarSpeedDistance>();

        public double T { get; set; }

        public int id;

        /// <summary>
        /// Directions from this cellColumnPair's lane from crossroad
        /// </summary>
        protected HashSet<Vector> nextCrossroadDirections = new HashSet<Vector>();

        /// <summary>
        /// Determines if this cellColumnPair is part of main road of crossroad
        /// </summary>
        public bool IsMainRoad { get; set; }

        /// <summary>
        /// Reference to crossroad instance. It is set only on entry crossroad
        /// cells to compute which crossroad's direction should car take
        /// </summary>
        public Crossroad NearestCrossroad { get; set; }

        /// <summary>
        /// Location of cellColumnPair
        /// </summary>
        public Point Location { get; protected set; }
        
        /// <summary>
        /// Way of cellColumnPair (direction of car on this cellColumnPair)
        /// </summary>
        protected Vector directionVector;
        /// <summary>
        /// Way of cellColumnPair (direction of car on this cellColumnPair)
        /// </summary>
        public Vector DirectionVector
        {
            get { return directionVector; }
            set { value.Normalize(); directionVector = value; }
        }

        /// <summary>
        /// To prevent cars from changing lane for example when they are close to crossroad
        /// </summary>
        public bool IsPossibleToChangleLaneLeft { get; set;}

        /// <summary>
        /// To prevent cars from changing lane for example when they are close to crossroad
        /// </summary>
        public bool IsPossibleToChangleLaneRight { get; set; }

        /// <summary>
        /// Maximum allowed speed on the cellColumnPair in cells per simulation step
        /// </summary>
        public int AllowedSpeed { get; set; }

        abstract class FollowingCellNode { }
        class FollowingCellNonLeafNode : FollowingCellNode
        {
            public int rank = -1;
            public FollowingCellNode continueNode = null;
            public FollowingCellNode turnNode = null;
        }
        class FollowingCellLeafNode : FollowingCellNode
        {
            public Cell followingCell = null;
        }        

        FollowingCellNode followingCellRoot = null;

        //just crossroad cells can have more than one following cells
        private List<Cell> followingCells = new List<Cell>();
        private Dictionary<Cell, List<int>> followingCellsRanks = new Dictionary<Cell, List<int>>();
        /// <summary>
        /// Cell connected in the front
        /// </summary>
        public Cell FollowingCell
        {
            get 
            {
                if (followingCells.Count == 0)
                    return null;
                return followingCells[0]; 
            }
            set
            {
                if (value == null)
                    followingCells.Clear();
                else if (followingCells.Count == 0)
                    followingCells.Add(value);
                else
                    followingCells[0] = value;
            }
        }

        public double? Priority { get; set; }

        public ReadOnlyCollection<Cell> FollowingCells { get { return followingCells.AsReadOnly(); } }
        public Cell FollowingCellLastAdded { get { return followingCells.Count > 0 ? followingCells[followingCells.Count - 1] : FollowingCell; } }
        public void AddFollowingCell(Cell cell)
        {
            if (followingCells.Count == 0)
            {
                throw new ApplicationException( "This method should be used when this cell becomes " +
                                                "crossroad cell. Use FollowingCell property instead");
            }
            if (followingCells.Contains(cell))
                throw new ApplicationException("Cell is again added as the following cell");
            followingCells.Add(cell);
        }

        //just crossroad cells can have more than one previous cells
        private List<Cell> previousCells = new List<Cell>();
        private Dictionary<Cell, List<int>> previousCellsRanks = new Dictionary<Cell, List<int>>();
        /// <summary>
        /// Cell connected in the back
        /// </summary>
        public Cell PreviousCell
        {
            get 
            {
                if (previousCells.Count == 0)
                    return null;
                return previousCells[0]; 
            }
            set
            {
                if (value == null)
                {
                    previousCells.Clear();
                    previousCellsConnections.Clear();
                    return;
                }
                if (previousCells.Count == 0)
                    previousCells.Add(value);
                else
                    previousCells[0] = value;                
            }
        }
        public ReadOnlyCollection<Cell> PreviousCells { get { return previousCells.AsReadOnly(); } }
        public Cell PreviousCellLastAdded { get { return previousCells.Count > 0 ? previousCells[previousCells.Count - 1] : null; } }
        public void AddPreviousCell(Cell cell)
        {
            if (previousCells.Count == 0)
            {
                throw new ApplicationException("This method should be used when this cell becomes " +
                                                "crossroad cell. Use PreviousCell property instead");
            }
            if (previousCells.Contains(cell))
                throw new ApplicationException("Cell is again added as the previous cell");
            previousCells.Add(cell);      
        }

        public bool IsCrossroadCell { get { return followingCells.Count > 1 || previousCells.Count > 1; } }

        /// <summary>
        /// List of crossing segments infos to previous cell to determine its final priority
        /// </summary>
        protected Dictionary<Cell, List<CrossingSegmentInfo>> previousCellsConnections = 
            new Dictionary<Cell, List<CrossingSegmentInfo>>();
        
        /// <summary>
        /// Cell on the left side of this cellColumnPair
        /// </summary>
        public Cell CellOnTheLeft
        {
            get 
            {
                if (cellsOnTheLeft.Count == 0)
                    return null;
                return cellsOnTheLeft[0]; 
            }
            set
            {
                if (value == null)
                    cellsOnTheLeft.Clear();
                else if (cellsOnTheLeft.Count == 0)
                    cellsOnTheLeft.Add(value);
                else
                    cellsOnTheLeft[0] = value;
            }
        }
        //just crossroad cells can have more than one cells on the left
        private List<Cell> cellsOnTheLeft = new List<Cell>();
        public ReadOnlyCollection<Cell> CellsOnTheLeft { get { return cellsOnTheLeft.AsReadOnly(); } }
        public void AddCellOnTheLeft(Cell cell)
        {
            if (cellsOnTheLeft.Count == 0)
            {
                throw new ApplicationException("This method should be used when this cell becomes " +
                                                "crossroad cell. Use CellOnTheLeft property instead");
            }
            if (cellsOnTheLeft.Contains(cell))
                return;//throw new ApplicationException("Cell is again added as the cell on the left");
            cellsOnTheLeft.Add(cell);
        }


        /// <summary>
        /// Cell on the right size of this cellColumnPair
        /// </summary>
        public Cell CellOnTheRight
        {
            get 
            {
                if (cellsOnTheRight.Count == 0)
                    return null;
                return cellsOnTheRight[0]; 
            }
            set
            {
                if (value == null)
                    cellsOnTheRight.Clear();
                else if (cellsOnTheRight.Count == 0)
                    cellsOnTheRight.Add(value);
                else
                    cellsOnTheRight[0] = value;
            }
        }
        //just crossroad cells can have more than one cells on the right
        private List<Cell> cellsOnTheRight = new List<Cell>();
        public ReadOnlyCollection<Cell> CellsOnTheRight { get { return cellsOnTheRight.AsReadOnly(); } }
        public void AddCellOnTheRight(Cell cell)
        {
            if (cellsOnTheRight.Count == 0)
            {
                throw new ApplicationException("This method should be used when this cell becomes " +
                                                "crossroad cell. Use CellOnTheRight property instead");
            }
            if (cellsOnTheRight.Contains(cell))
                return;//throw new ApplicationException("Cell is again added as the cell on the right");
            cellsOnTheRight.Add(cell);
        }



        /// <summary>
        /// Instance of car located on the cellColumnPair
        /// </summary>        
        protected Car car;
        public Car Car { get { return car; } set { car = value; } }

        /// <summary>
        /// Determines if cellColumnPair is not seized by car
        /// </summary>
        public bool IsFree { get { return Car == null; } }

        public bool IsCompletelyFree
        {
            get
            {
                if (!IsFree)
                    return false;
                foreach (var i in crossingCells)
                    if (!i.IsFree)
                        return false;
                return true;
            }
        }
        
        protected StreamWriter file = null;        
        protected GraphWindow graphWindow = null;

        public EWay Way { get; set; }

        /// <summary>
        /// Initialize cellColumnPair
        /// </summary>
        /// <param name="cellLocation">Location in the World</param>
        /// <param name="way">Way in the World</param>
        /// <param name="isPossibleToChangeLaneLeft">Determines if splitting line on the left is full</param>
        /// <param name="isPossibleToChangeLaneRight">Determines if splitting line on the right is full</param>
        /// <param name="allowedSpeed">Max allowed speed on the cellColumnPair</param>
        /// <param name="followingCell">Reference to following cellColumnPair</param>
        /// <param name="previousCell">Reference to previous cellColumnPair</param>
        /// <param name="cellOnTheLeft">Reference to cellColumnPair on the left side</param>
        /// <param name="cellOnTheRight">Reference to cellColumnPair on the right side</param>
        public Cell(int id, Point location, Vector directionVector, int allowedSpeed, double T, EWay way,
                    Cell followingCell = null, Cell previousCell = null,
                    Cell cellOnTheLeft = null, Cell cellOnTheRight = null,
                    bool isPossibleToChangeLaneLeft = false, bool isPossibleToChangeLaneRight = false) 
        {
            //initialize fields
            this.id = id;
            Location = location;            
            DirectionVector = directionVector;
            this.T = T;
            IsPossibleToChangleLaneLeft = isPossibleToChangeLaneLeft;
            IsPossibleToChangleLaneRight = isPossibleToChangeLaneRight;
            AllowedSpeed = allowedSpeed;
            CellOnTheLeft = cellOnTheLeft;
            CellOnTheRight = cellOnTheRight;
            FollowingCell = followingCell;
            PreviousCell = previousCell;            
            Car = null;
            Way = way;
        }

        /// <summary>
        /// Seize cellColumnPair by part of car
        /// </summary>
        /// <param name="newCar">Instance of car which takes this cellColumnPair</param>
        public void Seize(Car newCar)
        {
            //check if it is actually free
            if (!IsFree)
                throw new ApplicationException("Cell is already seized");
            //Console.WriteLine("SEIZE: Cell: " + id + ", car " + newCar.Id);
            //remove nearcar because that car is already crossing this cell
            if (PreviousCells.Count > 1 || CrossingCells.Count > 0)
            {
                nearCars.Remove(newCar);
                newCar.RemoveCrossroadCellWhereItIsPlanningToGo(this);
            }
            Car = newCar;
        }

        /// <summary>
        /// Remove car from cellColumnPair
        /// </summary>
        public void Free()
        {
            if (Car == null)
                throw new ApplicationException("Trying to free already free cell");
            //Console.WriteLine("FREE : Cell: " + id + ", car " + Car.Id);
            Car = null;
        }

        public override string ToString()
        {
            string returnString = "Cell on ";
            returnString += Location.X + " " + Location.Y + ";";
            if (FollowingCell == null)
                returnString += "Cell is the last.";
            if (PreviousCell == null)
                returnString += "Cell is the first.";
            if (!IsFree)
                returnString += "Cell contains car.";
            return returnString;
        }

        public int NumberOfDirections { get { return nextCrossroadDirections.Count; } }

        public void UpdateCellStatistics(Car car)
        {
            DateTime now = DateTime.Now;
            int numberOfDeletedItems = timesOfCarMoves.RemoveAll(i => now - i > TimeSpan.FromSeconds(60));
            carSpeeds.RemoveRange(0, numberOfDeletedItems);
            if (car != lastCar)
            {
                lastCar = car;
                timesOfCarMoves.Add(now);
                carSpeeds.Add(car.CurrentSpeed);
            }

            
        }

        public void SaveAndShowStatistics()
        {
            /*
            if (file != null)
            {
                string statistics = "";
                double? averageSpeed = GetAverageSpeed();
                int? flow = GetFlow();
                if (averageSpeed == null || averageSpeed == 0.0)
                {
                    statistics += "Speed\tN/A\t";
                    statistics += "Flow\tN/A\t";
                    statistics += "Density\tN/A\t";
                }
                else
                {
                    double toKmh = (World.cellLength / World.SimulationStepTime) * World.mps2kph;
                    statistics += "Speed\t" + (averageSpeed.Value * toKmh).ToString("0.000") + "\t";
                    statistics += "Flow\t" + flow.Value + "\t";
                    statistics += "Density\t" + ((60 / toKmh) * flow.Value / averageSpeed.Value).ToString("0.000") + "\t";
                }
                file.WriteLine(statistics);
            }*/

            if (graphWindow != null)
                graphWindow.UpdatePlots(this);
        }

        public double? GetAverageSpeed()
        {
            DateTime now = DateTime.Now;
            int numberOfDeletedItems = timesOfCarMoves.RemoveAll(i => now - i > TimeSpan.FromSeconds(60));
            carSpeeds.RemoveRange(0, numberOfDeletedItems);
            if (timesOfCarMoves.Count == 0)
                return null;
            int sum = 0;
            for (int i = 0; i < timesOfCarMoves.Count; i++)
            {
                sum += carSpeeds[i];
            }
            return (double)sum / timesOfCarMoves.Count;
        }

        public int? GetFlow()
        {
            DateTime now = DateTime.Now;
            int numberOfDeletedItems = timesOfCarMoves.RemoveAll(i => now - i > TimeSpan.FromSeconds(60));
            carSpeeds.RemoveRange(0, numberOfDeletedItems);
            if (timesOfCarMoves.Count == 0)
                return null;
            return timesOfCarMoves.Count;
        }

        public void AddCrossroadDirection(Vector direction)
        {
            nextCrossroadDirections.Add(direction.RoundCoordinates());
        }

        public bool ContainsDirection(Vector vector)
        {
            return nextCrossroadDirections.Contains(vector.RoundCoordinates());
        }

        public void SetAsStatistics()
        {
            //file = new StreamWriter(Location.X.ToString("00000") + "_" + Location.Y.ToString("00000"));
            graphWindow = new GraphWindow(AllowedSpeed, Location);
        }

        public void OpenStatisticsWindow()
        {
            if (graphWindow == null)
                graphWindow = new GraphWindow(AllowedSpeed, Location);
            graphWindow.Show();
        }

        public void CloseStatisticsFile()
        {
            //if (file != null)
            //    file.Close();
            if (graphWindow != null)
            {
                graphWindow.CloseWindow = true;
                graphWindow.Close();
            }
            graphWindow = null;
            file = null;
        }

        public void TranslateGraphWindow()
        {
            if (graphWindow != null)
                graphWindow.Translate();
        }
        
        internal void AddCrossingCell(Cell crossingCell, EPriority priority)
        {
            crossingCells.Add(crossingCell);
            crossingCellsPriorites.Add(priority);
        }

        internal Cell GetConnectedCell(DirectionPoint.Way.Instructions currentInstructions)
        {
            //int numberOfCells
            throw new NotImplementedException();
        }

        internal Cell GetConnectedCell(Vector? currentInstructions)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Determine gap. Also determine gaps on the lanes which car has to change to or change through
        /// </summary>
        /// <param name="currentInstructions"></param>
        /// <param name="numberOfLanesNeededToChangeToLeft"></param>
        /// <param name="numberOfLanesNeededToChangeToRight"></param>
        /// <param name="gap"></param>
        /// <param name="acc"></param>
        internal void Gap(List<EInstruction> currentInstructions, int numberOfLanesNeededToChangeToLeft,
            int numberOfLanesNeededToChangeToRight, out int gap, out int acc, out int maxTurnSpeed, out int allowedSpeed)
        {            
            this.Gap(currentInstructions, out gap, out acc, out maxTurnSpeed, out allowedSpeed);
            this.Gap(currentInstructions, numberOfLanesNeededToChangeToLeft, ELeftRight.Left, ref gap, ref acc, ref maxTurnSpeed, ref allowedSpeed);
            this.Gap(currentInstructions, numberOfLanesNeededToChangeToRight, ELeftRight.Right, ref gap, ref acc, ref maxTurnSpeed, ref allowedSpeed);                       
        }

        private void Gap(List<EInstruction> currentInstructions, int numberOfLanesNeededToChange, ELeftRight eLeftRight, ref int gap, ref int acc,
            ref int maxTurnSpeed, ref int allowedSpeed)
        {
            Cell cell = this;
            for (int i = 0; i < numberOfLanesNeededToChange; i++)
            {
                cell = eLeftRight == ELeftRight.Left ? cell.CellOnTheLeft : cell.CellOnTheRight;
                if (cell == null)
                    break;
                int gapInTheSide, accInTheSide, maxTurnSpeedInTheSide, allowdSpeedInTheSide;
                cell.Gap(currentInstructions, out gapInTheSide, out accInTheSide, out maxTurnSpeedInTheSide, out allowdSpeedInTheSide);
                //choose the least gap
                if (gap > gapInTheSide)
                {
                    gap = gapInTheSide;
                    acc = accInTheSide;
                }
                //choose the least maxTurnSpeedInTheSide
                if (maxTurnSpeed > maxTurnSpeedInTheSide)
                {
                    maxTurnSpeed = maxTurnSpeedInTheSide;
                }
                if (allowedSpeed > allowdSpeedInTheSide)
                {
                    allowedSpeed = allowdSpeedInTheSide;
                }
            }
        }

        private void Gap(List<EInstruction> currentInstructions, out int gap, out int acc, out int maxTurnSpeed, out int allowedSpeed)
        {
            Cell cell = this;
            Cell cellBefore = null;
            int currentInstructionsPointer = 0;
            gap = int.MaxValue;
            acc = int.MaxValue;
            maxTurnSpeed = int.MaxValue;
            allowedSpeed = int.MaxValue;
            for (int i = 0; i < Math.Min(AllowedSpeed, MaxTurnSpeed) * 2; i++)
            {                
                if (cell.FollowingCells.Count == 0)
                    return;
                //detect sharpness of turn and return max turn speed
                if (cell.MaxTurnSpeed < maxTurnSpeed)
                    maxTurnSpeed = cell.MaxTurnSpeed;
                if (cell.AllowedSpeed < allowedSpeed)
                    allowedSpeed = cell.AllowedSpeed;
                int foo1 = 0, foo2 = 0;
                bool notDecidedAboutWay;
                bool mustGiveWay;
                cellBefore = cell;
                cell = cell.GetNextFollowingCell(currentInstructions, i, Car.CurrentSpeed, 
                    ref currentInstructionsPointer, ref foo1, ref foo2, out notDecidedAboutWay, out mustGiveWay);
                if (cellBefore.FollowingCells.Count > 1 && cell.PreviousCells.Count > 1)
                {
                    //possible very short segment (2 cells length) - car must evaluate max turn speed
                    int maxTurnSpeed2CellSegment = cell.GetMaxTurnSpeed(cellBefore.Location, cellBefore.DirectionVector);
                    if (maxTurnSpeed2CellSegment < maxTurnSpeed)
                        maxTurnSpeed = maxTurnSpeed2CellSegment;
                }
                if (!cell.IsCompletelyFree || notDecidedAboutWay || mustGiveWay)
                {
                    gap = i;
                    if (!cell.IsFree)
                        acc = cell.Car.CurrentSpeed - cell.Car.PreviousSpeed;
                    else //else if (!notDecidedAboutWay || mustGiveWay)
                        acc = 0;
                    return;
                }
            }
            return;
        }

        internal Cell GetNextFollowingCell(List<EInstruction> currentInstructions, ref int currentInstructionsPointer,
            ref int changeLaneToLeft, ref int changeLaneToRight)
        {
            bool foo;
            return GetNextFollowingCell(currentInstructions, ref currentInstructionsPointer,
                ref changeLaneToLeft, ref changeLaneToRight, out foo);
        }

        internal Cell GetNextFollowingCell(List<EInstruction> currentInstructions, int distanceFromCar, int carSpeed,
            ref int currentInstructionsPointer, ref int changeLaneToLeft, ref int changeLaneToRight, 
            out bool instructionsMissing, out bool mustYieldRightOfWay)
        {
            Cell cell = GetNextFollowingCell(currentInstructions, ref currentInstructionsPointer, ref changeLaneToLeft,
                ref changeLaneToRight, out instructionsMissing);
            Cell previousCell = this;
            mustYieldRightOfWay = false;
            if (cell.PreviousCells.Count > 1 && cell.CrossingCells.Count == 0)
            {
                mustYieldRightOfWay = MustYield(distanceFromCar, carSpeed, previousCell.Priority.Value, new List<Cell>() {cell});
            }
            else
            {                                
                mustYieldRightOfWay = MustYield(distanceFromCar, carSpeed, cell.ComputeCompletePriorityValue(previousCell), 
                    new List<Cell>(cell.CrossingCells).Concat(new List<Cell>() {cell}).ToList());
            }
            return cell;
        }

        public double ComputeCompletePriorityValue(Cell previousCell)
        {
            double completePriority = 0.0;
            foreach (var i in crossingCellsPriorites)
                if (i == EPriority.High)
                    completePriority += 10.0;
            if (PreviousCells.Count > 1)
                completePriority += previousCell.Priority.Value;
            return completePriority;
        }

        internal Cell GetNextFollowingCell(List<EInstruction> currentInstructions, ref int currentInstructionsPointer,
            ref int changeLaneToLeft, ref int changeLaneToRight, out bool instructionsMissing)
        {
            if (changeLaneToLeft > 0)
            {
                changeLaneToLeft--;
                if (CellOnTheLeft == null)
                    throw new ApplicationException("Car's trying to change to left non-existing lane");
                return CellOnTheLeft.GetNextFollowingCell(currentInstructions, ref currentInstructionsPointer, 
                    ref changeLaneToLeft, ref changeLaneToRight, out instructionsMissing);
            }
            else if (changeLaneToRight > 0)
            {
                changeLaneToRight--;
                if (CellOnTheRight == null)
                    throw new ApplicationException("Car's trying to change to right non-existing lane");
                return CellOnTheRight.GetNextFollowingCell(currentInstructions, ref currentInstructionsPointer, 
                    ref changeLaneToLeft, ref changeLaneToRight, out instructionsMissing);
            }
            
            instructionsMissing = false;            
            if (FollowingCells.Count == 0)                            
                return null;
            else if (FollowingCells.Count == 1)                
                return FollowingCell;           
            else
            {
                FollowingCellNode node = followingCellRoot;
                while (!(node is FollowingCellLeafNode))
                {
                    FollowingCellNonLeafNode nonLeafNode = node as FollowingCellNonLeafNode;
                    if (currentInstructions == null || currentInstructionsPointer >= currentInstructions.Count)
                    {
                        instructionsMissing = true;
                        node = nonLeafNode.continueNode;
                    }
                    else
                    {
                        if (currentInstructionsPointer >= currentInstructions.Count ||
                           currentInstructions[currentInstructionsPointer++] == EInstruction.Continue)
                            node = nonLeafNode.continueNode;
                        else
                            node = nonLeafNode.turnNode;
                    }
                }                
                return (node as FollowingCellLeafNode).followingCell;
            }
        }

        protected bool MustYield(int distanceFromCar, int carSpeed, double myPriority, List<Cell> crossroadCells)
        {
            foreach (var c in crossroadCells)
            {
                foreach (var i in c.nearCars)
                {
                    if (myPriority < i.Value.Priority)
                    {
                        //get speed difference
                        int carsSpeedDifference = i.Value.Speed - carSpeed;
                        if (carsSpeedDifference < 0)
                            carsSpeedDifference = 0;

                        //minimum time
                        double minimumTimeOfArriveOfNearestCar = 2.0;
                        //get enhanced minimum time to arrive of nearest car with respect to speed of the other car.                     
                        double carsSpeedDifferenceKph = ToKph(carsSpeedDifference);
                        double increaseOfMiniimumTimeToArriveOfNearestCar = carsSpeedDifferenceKph * 0.1;
                        double enhancedMinimumTimeToArriveOfNearestCar = minimumTimeOfArriveOfNearestCar + increaseOfMiniimumTimeToArriveOfNearestCar;

                        //estimate my time to arrive to crossroad
                        double myTime = carSpeed == 0 ? 0.0 :
                            //(distanceFromCar * World.cellLength) / (carSpeed * (World.cellLength / World.SimulationStepTime)) * World.mps2kph; -> equal to next line                        
                            (distanceFromCar * World.SimulationStepTime * World.mps2kph) / carSpeed;
                        //estimate potential collision car's time to arrive to crossroad
                        double othersTime = i.Value.Speed == 0 ? 0.0 :
                            //(i.Value.Distance * World.cellLength) / (i.Value.Speed * (World.cellLength / World.SimulationStepTime)) * World.mps2kph; -> equal to next line
                            (i.Value.Distance * World.SimulationStepTime * World.mps2kph) / i.Value.Speed;
                        //compare times
                        //Console.WriteLine("My car speed: " + carSpeed + ", Crossroad distance: " + distanceFromCar);
                        //Console.WriteLine("Others car speed: " + i.Value.Speed + ", Others crossroad distance: " + i.Value.Distance);
                        //Console.Write("My time: " + myTime + ", Others time: " + othersTime + ", Enhanced min: " + enhancedMinimumTimeToArriveOfNearestCar);
                        if (myTime <= othersTime)
                        {
                            //when my time is less than time of other car, use enhanced minimum time
                            if (othersTime - myTime < enhancedMinimumTimeToArriveOfNearestCar)
                            {
                                //Console.WriteLine(", I must yield");
                                //Console.WriteLine("###################################################################");
                                return true;
                            }
                        }
                        else
                        {
                            //use classic minum time
                            if (myTime - othersTime < minimumTimeOfArriveOfNearestCar)
                            {
                                //Console.WriteLine(", I must yield");
                                //Console.WriteLine("###################################################################");
                                return true;
                            }
                        }
                        //Console.WriteLine(", I can go on");
                        //Console.WriteLine("###################################################################");
                    }
                }
            }
            return false;            
        }

        private double ToKph(int carsSpeedDifference)
        {
            return ((World.cellLength * carsSpeedDifference) / World.SimulationStepTime) * World.mps2kph;
        }
        
        internal void AddFollowingNonLeafNode(int rank, List<int> rankList)
        {
            if (followingCellRoot == null)
            {                
                followingCellRoot = new FollowingCellNonLeafNode() { rank = rank };                
                return;
            }
            FollowingCellNode node = followingCellRoot;
            FollowingCellNonLeafNode nonLeafNode;
            FollowingCellNonLeafNode previousTurn = null;
            foreach (var i in rankList)
            {
                nonLeafNode = node as FollowingCellNonLeafNode;
                while (nonLeafNode.rank != i)
                    nonLeafNode = nonLeafNode.continueNode as FollowingCellNonLeafNode;
                previousTurn = nonLeafNode;
                node = nonLeafNode.turnNode;
            }
            
            FollowingCellNonLeafNode previousContinue = null;
            while (node != null)
            {
                nonLeafNode = node as FollowingCellNonLeafNode;
                previousContinue = nonLeafNode;
                node = nonLeafNode.continueNode;                
            }
            if (node == null)
            {
                node = new FollowingCellNonLeafNode() { rank = rank };
                if (previousContinue != null)
                    previousContinue.continueNode = node;
                else if (previousTurn != null)
                    previousTurn.turnNode = node;
                else
                    throw new ApplicationException("Place for non-leaf node was not found");
            }
            else //node is FollowingCellSegmentJoinerNode
            {
                FollowingCellNonLeafNode newNode = new FollowingCellNonLeafNode() { rank = rank, continueNode = node };
                if (previousContinue != null)
                    previousContinue.continueNode = newNode;                
                else
                    followingCellRoot = newNode;
            }            
        }

        public void SetFollowingCell(Cell followingCell, List<int> rankList)
        {
            if (followingCellRoot == null)
                return;
            FollowingCellNode node = followingCellRoot;
            FollowingCellNonLeafNode nonLeafNode = null;
            FollowingCellNonLeafNode previousTurn = null;            
            if (rankList != null)
            {
                foreach (var i in rankList)
                {
                    nonLeafNode = node as FollowingCellNonLeafNode;                    
                    while (nonLeafNode.rank != i)
                    {
                        nonLeafNode = nonLeafNode.continueNode as FollowingCellNonLeafNode;                        
                    }
                    previousTurn = nonLeafNode;
                    node = nonLeafNode.turnNode;
                }
            }
            FollowingCellNonLeafNode previousContinue = null;
            while (node != null)
            {
                nonLeafNode = node as FollowingCellNonLeafNode;
                previousContinue = nonLeafNode;
                node = nonLeafNode.continueNode;
            }
            node = new FollowingCellLeafNode() { followingCell = followingCell };
            if (previousContinue != null)
                previousContinue.continueNode = node;
            else if (previousTurn != null)
                previousTurn.turnNode = node;
            else
                throw new ApplicationException("Leaf not found");

            //FollowingCellNode node = followingCellRoot;
            //FollowingCellNode turnNode;
            //if (rankList == null)
            //    rankList = new List<int>();
            //for (int i = 0; i < rankList.Count; i++)
            //{

            //}
            //if (rankList == null || rankList.Count == 0)
            //{                                
            //    FollowingCellNode previous = null;
            //    while (node != null)
            //    {
            //        previous = node;
            //        FollowingCellNonLeafNode nonLeafNode = node as FollowingCellNonLeafNode;
            //        node = nonLeafNode.continueNode;
            //    }
            //    if (previous != null)
            //        (previous as FollowingCellNonLeafNode).continueNode = new FollowingCellLeafNode() { followingCell = followingCell };
            //    else
            //        previous = new FollowingCellLeafNode() { followingCell = followingCell };
            //}
            //else
            //{
            //    for (int i = 0; i < rankList.Count; i++)
            //    {
            //        FollowingCellNonLeafNode nonLeafNode = node as FollowingCellNonLeafNode;
            //        while (nonLeafNode.rank != rankList[i])
            //            nonLeafNode = nonLeafNode.continueNode as FollowingCellNonLeafNode;
            //        if (i == rankList.Count - 1)
            //            node = new FollowingCellLeafNode() { followingCell = followingCell };
            //        else
            //            node = nonLeafNode.turnNode;
            //    }
            //}

        }

        internal double GetTimeToArriveOfTheNearestCar()
        {
            return GetTimeToArriveOfTheNearestCar(0);            
        }

        private double GetTimeToArriveOfTheNearestCar(int distanceFromCrossing)
        {
            double shortestTime = double.PositiveInfinity; //in seconds           
            Cell currentCell = this;
            while (currentCell != null)
            {
                if (!currentCell.IsFree)
                {
                    double timeOfThisCar = (distanceFromCrossing * World.cellLength) /
                        (distanceFromCrossing / currentCell.Car.CurrentSpeed * World.SimulationStepTime);
                    if (shortestTime > timeOfThisCar)
                        shortestTime = timeOfThisCar;
                    return shortestTime;
                }
                distanceFromCrossing++;
                for (int i = 1; i < currentCell.PreviousCells.Count; i++)
                {
                    double shortestTimeFromConnectingRoad = GetTimeToArriveOfTheNearestCar(distanceFromCrossing);
                    if (shortestTime > shortestTimeFromConnectingRoad)
                        shortestTime = shortestTimeFromConnectingRoad;
                }
                currentCell = currentCell.PreviousCell;
            }
            return shortestTime;
        }
       
        internal IEnumerable<Cell> CarCells(Car car)
        {
            Cell cell = this;
            bool found;
            do
            {
                found = false;
                yield return cell;

                foreach (var j in cell.PreviousCells)
                {
                    if (j.Car == car)
                    {
                        cell = j;
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    foreach (var j in new LaneMover(World.GetEdgeCell(cell)).GetNext())
                    {
                        foreach (var k in j.PreviousCells)
                        {
                            if (k.Car == car)
                            {
                                cell = k;
                                found = true;
                                break;
                            }
                        }
                        if (found)
                            break;
                    }
                }
            } while (found);
        }

        internal void RemoveCar(Car car)
        {
            foreach (var i in CarCells(car))
                i.Free();            
        }

        private int MaxTurnSpeed = int.MaxValue;

        public DirectionPoint DirectionPoint { get; protected set; }
        protected Dictionary<LaneSegment, Tuple<int, int>> mappingSegmentToLaneChanges = null;
        protected EWay wayOfCurrentCell;        
        internal void SetDirections(DirectionPoint directionPoint, LaneSegment segment)
        {
            DirectionPoint = directionPoint;
            wayOfCurrentCell = segment.Way;
            mappingSegmentToLaneChanges = new Dictionary<LaneSegment, Tuple<int, int>>();
            foreach (var i in DirectionPoint.LaneSegmentReferences)
            {
                LaneSegment currentSegment = segment;
                if (i.Segment == currentSegment)
                    mappingSegmentToLaneChanges.Add(currentSegment, new Tuple<int, int>(0, 0));
                else
                {
                    bool next = false;
                    int counter = 1;
                    while (currentSegment.RightNeighborLane != null)
                    {
                        if (currentSegment.RightNeighborLane == i.Segment)
                        {
                            mappingSegmentToLaneChanges.Add(i.Segment, new Tuple<int, int>(0, counter));
                            next = true;
                            break;
                        }
                        currentSegment = currentSegment.RightNeighborLane;
                        counter++;
                    }
                    if (next)
                        continue;
                    currentSegment = segment;
                    counter = 1;
                    while (currentSegment.LeftNeighborLane != null && currentSegment.LeftNeighborLane.Way == wayOfCurrentCell)
                    {
                        if (currentSegment.LeftNeighborLane == i.Segment)
                        {
                            mappingSegmentToLaneChanges.Add(i.Segment, new Tuple<int, int>(counter, 0));
                            break;
                        }
                        currentSegment = currentSegment.LeftNeighborLane;
                        counter++;
                    }
                }
            }
        }

        internal List<EInstruction> GetInstructions(out int numberOfLanesToChangeLeftToFollowInstructions, out int numberOfLanesToChangeRightToFollowInstructions,
            Random random)
        {
            LaneSegment segment;
            List<EInstruction> instructions = DirectionPoint.GetInstructions(wayOfCurrentCell, out segment, random);
            if (instructions == null)
            {
                numberOfLanesToChangeLeftToFollowInstructions = numberOfLanesToChangeRightToFollowInstructions = 0;
            }
            else
            {
                Tuple<int, int> changeLanes = mappingSegmentToLaneChanges[segment];
                numberOfLanesToChangeLeftToFollowInstructions = changeLanes.Item1;
                numberOfLanesToChangeRightToFollowInstructions = changeLanes.Item2;
            }
            return instructions;
        }

        internal void SetNearCar(Car car, int distance, int speed, double priority)
        {
            CarSpeedDistance csd = new CarSpeedDistance() { Distance = distance, Speed = speed, Priority = priority };
            if (!nearCars.ContainsKey(car))
                nearCars.Add(car, csd);
            else
                nearCars[car] = csd;
            car.AddCrossroadCellWhereItIsPlanningToGo(this);
        }

        internal void ComputeMaxTurnSpeed(Point otherCellLocation, Vector otherCellDirectionVector)
        {
            MaxTurnSpeed = GetMaxTurnSpeed(otherCellLocation, otherCellDirectionVector);
        }

        internal int GetMaxTurnSpeed(Point otherCellLocation, Vector otherCellDirection)
        {
            //compute angle
            double dotProduct = DirectionVector * otherCellDirection;
            if (dotProduct > 1.0)
                dotProduct = 1.0;
            if (dotProduct < -1.0)
                dotProduct = -1.0;
            double angle = Math.Acos(dotProduct);
            //compute distance
            double distance = (otherCellLocation - Location).Length;
            //compute speed
            double maxTurnSpeedMps = (distance / angle) / World.mps2kph;
            if (double.IsInfinity(maxTurnSpeedMps))
            {
                //Console.WriteLine("Dot product: " + dotProduct + ", Angle: " + angle + ", distance = " + distance + ", Speed mps: " +
                //    maxTurnSpeedMps + ", Speed: " + int.MaxValue);
                return int.MaxValue;
            }
            else
            {
                //Console.WriteLine("Dot product: " + dotProduct + ", Angle: " + angle + ", distance = " + distance + ", Speed mps: " +
                //    maxTurnSpeedMps + ", Speed: " + (int)Math.Round(maxTurnSpeedMps * (World.cellLength / World.SimulationStepTime)));
                int maxTurnSpeed = (int)Math.Round(maxTurnSpeedMps * (World.cellLength / World.SimulationStepTime));
                return maxTurnSpeed > 0 ? maxTurnSpeed : 1;
            }
        }

        internal void ClearNearCars()
        {
            nearCars.Clear();
        }
    }
}
