﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Media;
using System.Windows;

namespace TrafficMicroSimulator
{
    /// <summary>
    /// Represents car
    /// </summary>
    public class Car
    {
        /// <summary>
        /// Id of car used in gethashcode
        /// </summary>
        protected int id;
        public int Id { get { return id; } }

        /// <summary>
        /// Car's current speed
        /// </summary>
        public int CurrentSpeed { get; protected set; }

        /// <summary>
        /// Car's speed in previous step
        /// </summary>
        public int PreviousSpeed { get; protected set; }

        /// <summary>
        /// Car's length in number of cells which it occupies
        /// </summary>
        public int Length { get; protected set; }

        /// <summary>
        /// Car's max speed
        /// </summary>
        public int MaxSpeed { get; protected set; }

        /// <summary>
        /// Car's color which is drawn by
        /// </summary>
        public Color Color { get; protected set; }        

        /// <summary>
        /// Next crossroad direction influences car's selection of lane
        /// </summary>
        //protected Vector? nextCrossroadDirection = null;

        /// <summary>
        /// Random generator
        /// </summary>
        protected Random random;

        private bool increaseSpeed = true;

        HashSet<Car> carsWhichToGivePriority = new HashSet<Car>();
        private int deadlockCounter;

        private List<EInstruction> currentInstructions = null;
        protected int numberOfLanesToChangeLeftToFollowInstructions = 0;
        protected int numberOfLanesToChangeLeftAllowed = 0;
        protected int numberOfLanesToChangeRightToFollowInstructions = 0;
        protected int numberOfLanesToChangeRightAllowed = 0;
        Vector? nextCrossroadDirection = null;

        /// <summary>
        /// Creates new car instance
        /// </summary>
        /// <param name="id">Id which uniquely identifies car</param>
        public Car(int id, int length, int maxSpeed)
        {
            //initialize fields
            this.id = id;
            Length = length;
            MaxSpeed = maxSpeed;
            //initialize random instance
            random = new Random(((int)DateTime.Now.Ticks & 0x0000FFFF) + GetHashCode());
            //set random color to car
            Color = Color.FromRgb((byte)random.Next(256), (byte)random.Next(256), (byte)random.Next(256));
        }

        /// <summary>
        /// Analyse current situation of car and change current speed or plan change lanes
        /// </summary>
        /// <param name="currentLocation"></param>
        public Cell Move(Cell currentPositionFrontCell)
        {
            //store previous speed
            PreviousSpeed = CurrentSpeed;

            //find out if car can change lanes
            numberOfLanesToChangeLeftAllowed = 0;
            numberOfLanesToChangeRightAllowed = 0;
            if (numberOfLanesToChangeLeftToFollowInstructions != 0)
                numberOfLanesToChangeLeftAllowed = CanChangeLane(ELeftRight.Left, ref numberOfLanesToChangeLeftToFollowInstructions, currentPositionFrontCell);
            else if (numberOfLanesToChangeRightToFollowInstructions != 0)
                numberOfLanesToChangeRightAllowed = CanChangeLane(ELeftRight.Right, ref numberOfLanesToChangeRightToFollowInstructions, currentPositionFrontCell);
            //if car cannot change all lanes which need, it must slow down
            bool slowdownBecauseCannotChangeLane = (numberOfLanesToChangeLeftToFollowInstructions == numberOfLanesToChangeLeftAllowed &&
                numberOfLanesToChangeRightToFollowInstructions == numberOfLanesToChangeRightAllowed) ? false : true;

            //perform transition function respecting also changing lanes
            PerformEnhancedLocalTransitionFunction(currentPositionFrontCell, slowdownBecauseCannotChangeLane);
            //Seize new cells
            return Seize(currentPositionFrontCell);
        }

        private Cell Seize(Cell currentPositionFrontCell)
        {
            Cell cellBefore = null;
            Cell cellToReturn = null;
            Cell cell = currentPositionFrontCell;
            for (int i = 0; i < CurrentSpeed; i++)
            {
                int currentInstructionPointer = 0;
                cellBefore = cell;
                cell = cell.GetNextFollowingCell(currentInstructions, ref currentInstructionPointer,
                    ref numberOfLanesToChangeLeftAllowed, ref numberOfLanesToChangeRightAllowed);

                if (cell == null)
                {
                    //end of road
                    cellBefore.RemoveCar(this);
                    return null;
                }
                if (currentInstructions != null)
                    currentInstructions.RemoveRange(0, currentInstructionPointer);
                //seize next cell
                cell.Seize(this);

                //get directions
                if (cell.DirectionPoint != null)
                    currentInstructions = cell.GetInstructions(out numberOfLanesToChangeLeftToFollowInstructions,
                        out numberOfLanesToChangeRightToFollowInstructions, random);                
            }
            cellToReturn = cell;

            //set crossroad near cars
            int instructionPointer = 0;
            cellBefore = cell;
            cell = cell.GetNextFollowingCell(currentInstructions, ref instructionPointer,
                ref numberOfLanesToChangeLeftAllowed, ref numberOfLanesToChangeRightAllowed);
            int distance = 0;
            while (cell != null && cell.IsFree && cell.DirectionPoint == null)
            {                
                distance++;
                if (cell != null)
                {
                    if (cell.PreviousCells.Count > 1 && cell.CrossingCells.Count == 0)
                    {
                        cell.SetNearCar(this, distance, CurrentSpeed, cellBefore.Priority.Value);
                    }
                    else if (cell.CrossingCells.Count > 0)
                    {                        
                        cell.SetNearCar(this, distance, CurrentSpeed, cell.ComputeCompletePriorityValue(cellBefore));
                    }                    
                }
                cellBefore = cell;
                cell = cell.GetNextFollowingCell(currentInstructions, ref instructionPointer,
                    ref numberOfLanesToChangeLeftAllowed, ref numberOfLanesToChangeRightAllowed);
            } 

            return cellToReturn;
        }

        /// <summary>
        /// Send car command to execute one step and move forwards
        /// </summary>
        /// <param name="currentPositionFrontCell">Current first cellColumnPair which it seizes</param>
        /// <returns>First cellColumnPair of new car's position</returns>        
        public void Release(Cell currentPositionFrontCell)
        {
            int counter = 0;
            foreach (var i in currentPositionFrontCell.CarCells(this))            
                if (++counter > Length)
                    i.Free();           
        }                

        private int CanChangeLane(ELeftRight eLeftRight, ref int numberOfLanesToChangeToFollowInstructions, Cell currentCell)
        {
            int numberOfPossibleChanges = 0;
            Cell cellOnTheSide = currentCell;
            for (int i = 0; i < (eLeftRight == ELeftRight.Left ?
                numberOfLanesToChangeToFollowInstructions : numberOfLanesToChangeRightToFollowInstructions); i++)
            {
                cellOnTheSide = eLeftRight == ELeftRight.Left ? cellOnTheSide.CellOnTheLeft : cellOnTheSide.CellOnTheRight;
                if (cellOnTheSide == null)
                {
                    numberOfLanesToChangeToFollowInstructions = numberOfPossibleChanges;                    
                    return numberOfPossibleChanges;
                }
                if (cellOnTheSide.IsFree)
                    numberOfPossibleChanges++;
                else
                    break;
            }
            return numberOfPossibleChanges;
        }

        private int emergencyBreaks = 0;
        private int currentSpeedNegative = 0;
        /// <summary>
        /// Based on Evolutionary Approach to Calibration of Cellular Automaton
        /// Traffic Simulation Models by Pavol Korcek, Lukas Sekanina and Otto Fucik
        /// </summary>
        private void PerformEnhancedLocalTransitionFunction(Cell currentPositionFrontCell, bool slowDownBecauseOfChangingLane)
        {
            //parameters
            double p7 = 0.8;
            double p6 = currentPositionFrontCell.AllowedSpeed / 2;
            double p5 = 0.2;
            double p8 = 0.1;
            double p9 = 2.0;
            double p10 = 3.0;
            double pSlowDownBecauseOfChangingLane = 2.0;

            //acceleration step
            if (CurrentSpeed < MaxSpeed && CurrentSpeed < currentPositionFrontCell.AllowedSpeed)
            {
                if (random.NextDouble() < p7)
                    CurrentSpeed++;
            }
            
            //get gap length and acceleration of next vehicle
            int gap;
            int accOfNextCar;
            int maximumTurnSpeed;
            int allowedSpeed;
            currentPositionFrontCell.Gap(currentInstructions, numberOfLanesToChangeLeftToFollowInstructions,
                numberOfLanesToChangeRightToFollowInstructions, out gap, out accOfNextCar, out maximumTurnSpeed, out allowedSpeed);
            if (CurrentSpeed > allowedSpeed)
                CurrentSpeed = allowedSpeed;
            if (CurrentSpeed > maximumTurnSpeed) //deep turn so car must slow down
                CurrentSpeed = maximumTurnSpeed;
            if (gap == int.MaxValue || (gap + accOfNextCar) > CurrentSpeed)
            {
                if (CurrentSpeed > 1)
                {
                    //random slowing
                    if (CurrentSpeed < p6)
                    {
                        if (random.NextDouble() < p5)
                            CurrentSpeed--;
                    }
                    else
                    {
                        if (random.NextDouble() < p8)
                            CurrentSpeed--;
                    }
                }
            }
            else
            {
                //slowing because of too small gap
                if (accOfNextCar > 0)
                    CurrentSpeed = (int)Math.Round((gap + accOfNextCar) / p9);
                else
                    CurrentSpeed = (int)Math.Round((gap + accOfNextCar) / p10);
            }

            if (slowDownBecauseOfChangingLane)
                CurrentSpeed = (int)(Math.Round(CurrentSpeed / pSlowDownBecauseOfChangingLane));
            
            if (CurrentSpeed > gap)
            {
                //emergency break
                emergencyBreaks++;
                Console.WriteLine("Emergency break applied #" + emergencyBreaks);
                CurrentSpeed = gap;
            }
            if (CurrentSpeed < 0)
            {
                currentSpeedNegative++;
                Console.WriteLine("Current speed less than 0 #" + currentSpeedNegative);
                CurrentSpeed = 0;
            }
        }

        ///<summary>
        ///Send car command to execute one step and move forwards
        ///</summary>
        ///<param name="currentPositionFrontCell">Current first cellColumnPair which it seizes</param>
        ///<returns>First cellColumnPair of new car's position</returns>        
        //public Cell Move(Cell currentPositionFrontCell)
        //{
        //    //save current speed as previous
        //    PreviousSpeed = CurrentSpeed;
        //    bool crossroadCrossed = false;
        //    bool checkCrossroad = false;
        //    bool changeLaneToLeft = false;
        //    bool numberOfLanesToChangeRightAllowed = false;
        //    bool laneWillJoin = false;
        //    bool carWaitingBeforeLaneJoining = false;
        //    int numberOfLanesWhichCarHasToChange = 0;
        //    bool slowdownBecauseCannotChangeLane = false;
        //    int numberOfFreeCells = 0;
        //    int distanceToCrossroad = -1;
            
        //    Vector directionOfCurrentRoad = new Vector();
        //    ////get next crossroad direction
        //    //if (nextCrossroadDirection == null)
        //    //{
        //    //    Cell pomCell = currentPositionFrontCell;
        //    //    bool foo;
        //    //    while (pomCell.FollowingCell != null)
        //    //        pomCell = World.CellForwards(pomCell, nextCrossroadDirection, out foo);
        //    //    if (pomCell.NearestCrossroad != null)
        //    //        nextCrossroadDirection = currentPositionFrontCell.NearestCrossroad.GetDirection(pomCell.DirectionVector);
        //    //    else
        //    //        //any crossroad in front
        //    //        nextCrossroadDirection = new Vector();
        //    //}
        //    int accelerationOfFollowingVehicle = 0;
        //    Cell cellInTheFront = World.CellForwards(currentPositionFrontCell, currentInstructions, out crossroadCrossed);
        //    checkCrossroad = crossroadCrossed;
        //    Cell lastCrossroadCell = null;
        //    Cell backupCell = currentPositionFrontCell;
        //    if (crossroadCrossed)
        //    {
        //        checkCrossroad = crossroadCrossed;
        //        //store information about distance to crossroad and direction of cell nearest to crossroad
        //        distanceToCrossroad = numberOfFreeCells;
        //        lastCrossroadCell = backupCell;
        //        directionOfCurrentRoad = backupCell.DirectionVector;
        //        directionOfCurrentRoad.Negate();
        //    }
            
            
        //    int maxTurnSpeed = int.MaxValue;
        //    int numberOfCellsToTurn = 0;
        //    int numberOfCellsToJoin = 0;
        //    bool wasTryiedToChangeLaneWithLessCars = false;
        //    //go through cells in the front until possible or until neede
        //    while (cellInTheFront != null && cellInTheFront.IsFree && numberOfFreeCells <= CurrentSpeed * 5 + 5)
        //    {
        //        if (backupCell.CellOnTheLeft != null && backupCell.DirectionVector == backupCell.CellOnTheLeft.DirectionVector &&
        //            cellInTheFront.DirectionVector != cellInTheFront.CellOnTheLeft.DirectionVector && backupCell.FollowingCell != null)
        //            //lane will join
        //            laneWillJoin = true;
        //        if (cellInTheFront.CellOnTheLeft != null && cellInTheFront.DirectionVector == cellInTheFront.CellOnTheLeft.DirectionVector &&
        //            !cellInTheFront.CellOnTheLeft.IsFree && cellInTheFront.CellOnTheLeft.Car.CurrentSpeed <= 2)
        //            carWaitingBeforeLaneJoining = true;
        //        //check if there is not turn
        //        if (backupCell.DirectionVector != cellInTheFront.DirectionVector)
        //        {
        //            //compute angle of turn
        //            double angle = AngleOfTurn(backupCell.DirectionVector, cellInTheFront.DirectionVector);
        //            //convert to degrees
        //            angle = 180 * angle / Math.PI;
        //            //compute max turn speed
        //            int newMaxTurnSpeed = int.MaxValue;
        //            double speedConvertingValue = World.mps2kph / (World.cellLength / World.SimulationStepTime);
        //            if (angle < 10)
        //                newMaxTurnSpeed = int.MaxValue;
        //            else if (angle < 25)
        //                newMaxTurnSpeed = (int)Math.Round(70 / speedConvertingValue);
        //            else if (angle < 50)
        //                newMaxTurnSpeed = (int)Math.Round(40 / speedConvertingValue);
        //            else if (angle < 75)
        //                newMaxTurnSpeed = (int)Math.Round(30 / speedConvertingValue);
        //            else if (angle < 100)
        //                newMaxTurnSpeed = (int)Math.Round(20 / speedConvertingValue);
        //            else if (angle < 125)
        //                newMaxTurnSpeed = (int)Math.Round(15 / speedConvertingValue);
        //            else if (angle < 150)
        //                newMaxTurnSpeed = (int)Math.Round(10 / speedConvertingValue);
        //            else
        //                newMaxTurnSpeed = (int)Math.Round(5 / speedConvertingValue);
        //            if (newMaxTurnSpeed < maxTurnSpeed)
        //            {
        //                numberOfCellsToTurn = numberOfFreeCells;
        //                maxTurnSpeed = newMaxTurnSpeed;
        //            }
        //        }
        //        //change lane if needed
        //        if (!slowdownBecauseCannotChangeLane && !checkCrossroad &&
        //            nextCrossroadDirection != new Vector() && !cellInTheFront.ContainsDirection(nextCrossroadDirection.Value))
        //        {
        //            //it is needed to change lane
        //            //determine where is correct lane (on the left or right)
        //            bool neededToChangeLaneToLeftFalseToRightTrue = true;
        //            LaneMover laneMover = new LaneMover(World.GetEdgeCell(cellInTheFront));
        //            foreach (var i in laneMover.GetNext())
        //            {
        //                if (laneMover.DirectionDelimiterLineCrossed)
        //                    break;
        //                if (i == cellInTheFront)
        //                {
        //                    neededToChangeLaneToLeftFalseToRightTrue = false;
        //                    break;
        //                }
        //                if (i.ContainsDirection(nextCrossroadDirection.Value))
        //                    break;
        //            }

        //            while (!cellInTheFront.ContainsDirection(nextCrossroadDirection.Value))
        //            {   //go as much to side as needed
        //                if (neededToChangeLaneToLeftFalseToRightTrue)
        //                    cellInTheFront = cellInTheFront.CellOnTheRight;
        //                else
        //                    cellInTheFront = cellInTheFront.CellOnTheLeft;
        //                if (CheckForCollisionOnTheSideLane(cellInTheFront))
        //                {
        //                    //next lane is free
        //                    if (neededToChangeLaneToLeftFalseToRightTrue)
        //                    {
        //                        numberOfLanesToChangeRightAllowed = true;
        //                        numberOfLanesWhichCarHasToChange++;
        //                    }
        //                    else
        //                    {
        //                        changeLaneToLeft = true;
        //                        numberOfLanesWhichCarHasToChange++;
        //                    }
        //                }
        //                else
        //                {
        //                    //next lane is not free
        //                    slowdownBecauseCannotChangeLane = true;
        //                    numberOfCellsToJoin = numberOfFreeCells;
        //                    break;
        //                }
        //            }
        //        }
        //        else if (cellInTheFront.PreviousCell != backupCell)
        //        {
        //            //lanes joining. Car has to check for collision
        //            if (!slowdownBecauseCannotChangeLane && !CheckForCollisionOnTheSideLane(cellInTheFront))
        //            {
        //                slowdownBecauseCannotChangeLane = true;
        //                numberOfCellsToJoin = numberOfFreeCells;
        //            }
        //        }
        //        else if (!wasTryiedToChangeLaneWithLessCars && numberOfLanesWhichCarHasToChange == 0)
        //        {
        //            //determine if car approaches to crossroad
        //            if (CarApproachesToQueue(cellInTheFront) && CurrentSpeed > 2)
        //            {
        //                //try change lane to use that lane where is less cars
        //                wasTryiedToChangeLaneWithLessCars = true;
        //                //get cells of lanes which also have direction of this car on the next crossroad
        //                List<Cell> cellsOnTheLeftWhichCarCanUse = new List<Cell>();
        //                List<Cell> cellsOnTheRightWhichCarCanUse = new List<Cell>();
        //                Cell cellOnTheSide = cellInTheFront;
        //                while (cellOnTheSide.CellOnTheLeft != null && cellOnTheSide.CellOnTheLeft.CellOnTheLeft != cellOnTheSide)
        //                {
        //                    cellOnTheSide = cellOnTheSide.CellOnTheLeft;
        //                    if (cellOnTheSide.ContainsDirection(nextCrossroadDirection.Value) || cellOnTheSide.NearestCrossroad == null)
        //                        cellsOnTheLeftWhichCarCanUse.Add(cellOnTheSide);
        //                }
        //                cellOnTheSide = cellInTheFront;
        //                while (cellOnTheSide.CellOnTheRight != null)
        //                {
        //                    cellOnTheSide = cellOnTheSide.CellOnTheRight;
        //                    if (cellOnTheSide.ContainsDirection(nextCrossroadDirection.Value) || cellOnTheSide.NearestCrossroad == null)
        //                        cellsOnTheRightWhichCarCanUse.Add(cellOnTheSide);
        //                }
        //                int currentDistanceToVerySlowCar = DistanceToVerySlowCar(cellInTheFront);
        //                int distanceToVerySlowCar = currentDistanceToVerySlowCar + 1;// + (int)Math.Round(15 / World.cellLength);
        //                int laneToSwitch = 0;
        //                int pom;
        //                bool doNotChangeLane = false;
        //                for (int i = 0; i < cellsOnTheLeftWhichCarCanUse.Count; i++)
        //                {
        //                    pom = DistanceToVerySlowCar(cellsOnTheLeftWhichCarCanUse[i]);
        //                    if (pom == -1)
        //                    {
        //                        doNotChangeLane = true;
        //                        break;
        //                    }
        //                    if (pom > distanceToVerySlowCar)
        //                    {
        //                        distanceToVerySlowCar = pom;
        //                        laneToSwitch = -(i + 1);
        //                    }
        //                }
        //                if (!doNotChangeLane)
        //                {
        //                    for (int i = 0; i < cellsOnTheRightWhichCarCanUse.Count; i++)
        //                    {
        //                        pom = DistanceToVerySlowCar(cellsOnTheRightWhichCarCanUse[i]);
        //                        if (pom == -1)
        //                        {
        //                            doNotChangeLane = true;
        //                            break;
        //                        }
        //                        if (pom > distanceToVerySlowCar)
        //                        {
        //                            distanceToVerySlowCar = pom;
        //                            laneToSwitch = i + 1;
        //                        }
        //                    }
        //                }
        //                if (laneToSwitch != 0)
        //                {
        //                    if (laneToSwitch < 0)
        //                    {
        //                        changeLaneToLeft = true;
        //                        numberOfLanesWhichCarHasToChange = -laneToSwitch;
        //                    }
        //                    else
        //                    {
        //                        numberOfLanesToChangeRightAllowed = true;
        //                        numberOfLanesWhichCarHasToChange = laneToSwitch;
        //                    }
        //                }
        //                int j;
        //                for (j = 0; j < numberOfLanesWhichCarHasToChange; j++)
        //                {
        //                    if (changeLaneToLeft)
        //                    {
        //                        if (CheckForCollisionOnTheSideLane(cellInTheFront.CellOnTheLeft))
        //                            cellInTheFront = cellInTheFront.CellOnTheLeft;
        //                        else
        //                            break;
        //                    }
        //                    if (numberOfLanesToChangeRightAllowed)
        //                    {
        //                        if (CheckForCollisionOnTheSideLane(cellInTheFront.CellOnTheRight))
        //                            cellInTheFront = cellInTheFront.CellOnTheRight;
        //                        else
        //                            break;
        //                    }
        //                }
        //                numberOfLanesWhichCarHasToChange = j;
        //                if (numberOfLanesWhichCarHasToChange == 0)
        //                {
        //                    changeLaneToLeft = false;
        //                    numberOfLanesToChangeRightAllowed = false;
        //                }
        //            }
                    
        //        }

        //        backupCell = cellInTheFront;
        //        cellInTheFront = World.CellForwards(cellInTheFront, nextCrossroadDirection, out crossroadCrossed);
        //        numberOfFreeCells++;
        //        if (cellInTheFront != null && !cellInTheFront.IsFree)
        //        {
        //            accelerationOfFollowingVehicle = cellInTheFront.Car.CurrentSpeed - cellInTheFront.Car.PreviousSpeed;
        //        }
        //        if (crossroadCrossed)
        //        {
        //            checkCrossroad = crossroadCrossed;
        //            //store information about distance to crossroad and direction of cell nearest to crossroad
        //            distanceToCrossroad = numberOfFreeCells;
        //            lastCrossroadCell = backupCell;
        //            directionOfCurrentRoad = backupCell.DirectionVector;
        //            directionOfCurrentRoad.Negate();
        //        }                
        //    }

        //    //Nagel-Schreckenberg steps
        //    //1 step: Acceleration
        //    if (currentPositionFrontCell.AllowedSpeed < CurrentSpeed ||
        //        CurrentSpeed > maxTurnSpeed)
        //    {
        //        CurrentSpeed--;
        //        if (numberOfCellsToTurn < CurrentSpeed * 4)
        //        {
        //            CurrentSpeed = (int)Math.Round(CurrentSpeed / 1.5);
        //            if (CurrentSpeed == 0 && numberOfFreeCells != 0)
        //                CurrentSpeed = 1;
        //        }
        //    }
        //    else
        //    {
        //        if (increaseSpeed && CurrentSpeed < MaxSpeed)
        //        {
        //            increaseSpeed = false;
        //            CurrentSpeed++;
        //        }
        //        else
        //        {
        //            if (CurrentSpeed <= 6)
        //                CurrentSpeed++;
        //            increaseSpeed = true;
        //        }
        //    }
        //    if (slowdownBecauseCannotChangeLane)
        //    {
        //        if (CurrentSpeed <= 1)
        //            CurrentSpeed = 0;
        //        else
        //            CurrentSpeed -= 2;
        //        if (numberOfCellsToJoin < CurrentSpeed * 4)
        //        {
        //            if (Math.Round(CurrentSpeed / 1.5) < Math.Round((double)numberOfCellsToJoin / 4.0))
        //                CurrentSpeed = (int)Math.Round(CurrentSpeed / 1.5);
        //            else
        //                CurrentSpeed = (int)Math.Round((double)numberOfCellsToJoin / 4.0);
        //        }
        //    }
        //    if (laneWillJoin && carWaitingBeforeLaneJoining)
        //    {
        //        CurrentSpeed -= 2;
        //        if (CurrentSpeed <= 0)
        //        {
        //            if (random.NextDouble() < 0.75)
        //                CurrentSpeed = 1;
        //            else
        //                CurrentSpeed = 0;
        //        }
        //        if (CurrentSpeed > numberOfFreeCells)
        //            CurrentSpeed = numberOfFreeCells;
        //    }
        //    if (cellInTheFront == null || cellInTheFront.IsFree || numberOfFreeCells + accelerationOfFollowingVehicle > CurrentSpeed * 2)
        //    {
        //        //Randomization
        //        if (CurrentSpeed > 1 && random.NextDouble() < 0.2)
        //            CurrentSpeed--;
        //    }
        //    else
        //    {   //Slowing down
        //        double parameter; //parameter used to compute next speed
        //        if (accelerationOfFollowingVehicle > 0)
        //        {
        //            parameter = 2;
        //        }
        //        else
        //        {
        //            parameter = 3;
        //        }
        //        CurrentSpeed = (int)Math.Round((double)(numberOfFreeCells + accelerationOfFollowingVehicle) / parameter);
        //    }
        //    if (checkCrossroad)
        //    {
        //        if (!(lastCrossroadCell.ContainsDirection(nextCrossroadDirection.Value)) && distanceToCrossroad < 6)
        //            //do not cross crossroad if car is on the bad lane
        //            CurrentSpeed = 0;
        //        else if (CheckCrossroadForCollisions(currentPositionFrontCell, distanceToCrossroad, directionOfCurrentRoad.RoundCoordinates()))
        //        {   //car has to wait for other car
        //            if (currentPositionFrontCell.FollowingCell == null)
        //            //stop if it is behind the crossroad
        //            {
        //                CurrentSpeed = 0;
        //            }
        //            else if (CurrentSpeed > 0)
        //            {
        //                //slow down if car approaches to crossroad
        //                if (CurrentSpeed > distanceToCrossroad / 4.0)
        //                {
        //                    if (Math.Round(CurrentSpeed / 1.5) < Math.Round((double)distanceToCrossroad / 4.0))
        //                        CurrentSpeed = (int)Math.Round(CurrentSpeed / 1.5);
        //                    else
        //                        CurrentSpeed = (int)Math.Round((double)distanceToCrossroad / 4.0);
        //                }
        //            }
        //        }
        //    }
        //    //move car if it is moving
        //    if (CurrentSpeed > 0)
        //    {
        //        //cancel current cells
        //        Cell carCell = currentPositionFrontCell;
        //        for (int i = 0; i < Length; i++)
        //        {
        //            Vector directionVectorToCheck = carCell.DirectionVector;
        //            while (carCell.IsFree)
        //            //lanes are dividing
        //            {
        //                if (carCell.CellOnTheLeft.CellOnTheLeft == carCell || carCell.CellOnTheLeft == null)
        //                    //error to prevent free cell of the other car on the other side of road
        //                    //throw new ApplicationException("Illegal try to free cell");
        //                    break;
        //                carCell = carCell.CellOnTheLeft;
        //            }
        //            if (carCell.IsFree || carCell.Car != this)
        //                break;
        //            carCell.Free();
        //            /*while (carCell.PreviousCell == null && carCell.CellOnTheRight != null)
        //            {
        //                carCell = carCell.CellOnTheRight;
        //            }*/
        //            if (carCell.PreviousCell == null)
        //                break;
        //            carCell = carCell.PreviousCell;
        //        }
        //        //go to the new position
        //        Cell followingCell = currentPositionFrontCell;
        //        bool crossroadWasCrossed = false;
        //        Cell backupCell2 = currentPositionFrontCell;
        //        for (int i = 0; i < CurrentSpeed; i++)
        //        {
        //            backupCell2 = followingCell;
        //            bool laneChanged = false;
        //            //change lane if it car has to do that
        //            if (changeLaneToLeft && numberOfLanesWhichCarHasToChange > 0 && followingCell.CellOnTheLeft != null && 
        //                followingCell.CellOnTheLeft.CellOnTheLeft != followingCell)
        //            {
        //                if (backupCell2.DirectionVector != followingCell.DirectionVector)
        //                    throw new ApplicationException("Cannot change lane to lane of other direction");
        //                followingCell = followingCell.CellOnTheLeft;
        //                numberOfLanesWhichCarHasToChange--;
        //                laneChanged = true;
        //            }
        //            if (numberOfLanesToChangeRightAllowed && numberOfLanesWhichCarHasToChange > 0 && followingCell.CellOnTheRight != null)
        //            {
        //                followingCell = followingCell.CellOnTheRight;
        //                numberOfLanesWhichCarHasToChange--;
        //                laneChanged = true;
        //            }
        //            if (!laneChanged)
        //            {
        //                followingCell = World.CellForwards(followingCell, nextCrossroadDirection, out crossroadCrossed);
        //            }
        //            else
        //            {
        //                i--;
        //            }
        //            if (followingCell == null)
        //                //end of road
        //                return null;
        //            if (crossroadCrossed)
        //                crossroadWasCrossed = true;
        //            followingCell.UpdateCellStatistics(this);
        //            if (/*(crossroadCrossed && doNotCrossCrossroad) || */!followingCell.IsFree && !laneChanged)
        //            {
        //                followingCell = backupCell2;
        //                //nextCrossroadDirection = null;
        //                CurrentSpeed = CurrentSpeed - (CurrentSpeed - i);
        //                if (CurrentSpeed < 0)
        //                    CurrentSpeed = 0;
        //            }
                
        //        }

        //        bool end = false;
        //        while (!end)
        //        {
        //            end = true;
        //            //change lane if it car has to do that
        //            if (changeLaneToLeft && numberOfLanesWhichCarHasToChange > 0 && followingCell.CellOnTheLeft != null &&
        //                followingCell.CellOnTheLeft.CellOnTheLeft != followingCell)
        //            {
        //                //if (backupCell2.DirectionVector != followingCell.DirectionVector)
        //                //    throw new ApplicationException("Cannot change lane to lane of other direction");
        //                followingCell = followingCell.CellOnTheLeft;
        //                numberOfLanesWhichCarHasToChange--;
        //                end = false;
        //            }
        //            if (numberOfLanesToChangeRightAllowed && numberOfLanesWhichCarHasToChange > 0 && followingCell.CellOnTheRight != null)
        //            {
        //                followingCell = followingCell.CellOnTheRight;
        //                numberOfLanesWhichCarHasToChange--;
        //                end = false;
        //            }
        //        }

        //        if (crossroadWasCrossed)
        //        {
        //            nextCrossroadDirection = null;
        //            carsWhichToGivePriority.Clear();
        //        }
        //        Cell cellToReturn = followingCell;
        //        //seize cells by going backwards <Length> cells
        //        for (int i = 0; i < Length; i++)
        //        {
        //            if (!followingCell.IsFree)
        //                //needed when car changes lane because of its ending
        //                break;
        //            followingCell.Seize(this);
        //            /*while (followingCell.PreviousCell == null && followingCell.CellOnTheRight != null)
        //            {
        //                followingCell = followingCell.CellOnTheRight;
        //            }*/
        //            if (followingCell.PreviousCell == null)
        //                break;
        //            followingCell = followingCell.PreviousCell;
        //        }
        //        return cellToReturn;
        //    }
        //    return currentPositionFrontCell;
        //}

        private int DistanceToVerySlowCar(Cell cell)
        {
            int counter = 0;
            while (cell.FollowingCell != null && counter < 300 / World.cellLength)
            {
                cell = cell.FollowingCell;
                counter++;
                if (!cell.IsFree)
                {
                    if (cell.Car.CurrentSpeed <= 2)
                    {
                        return counter;
                    }
                    else
                        return -1;
                }
            }
            return int.MaxValue - 1;
        }
            
        private bool CarApproachesToQueue(Cell cell)
        {
            int counter = 0;
            while (cell.FollowingCell != null)
            {
                cell = cell.FollowingCell;
                counter++;
                if (!cell.IsFree)
                {
                    if (cell.Car.CurrentSpeed <= 2)
                    {
                        if (counter < 200 / World.cellLength)
                            return true;
                        else
                            return false;
                    }
                    else
                        return false;
                }
            }
            return false;
        }

        private double AngleOfTurn(Vector vector, Vector vector_2)
        {
            return Math.Acos(vector.X * vector_2.X + vector.Y * vector_2.Y);
        }

        private bool CheckForCollisionOnTheSideLane(Cell cellOnTheSide)
        {
            //find cell
            for (int i = 0; i < 400; i++)
            {
                if (!cellOnTheSide.IsFree)
                {
                    if (CurrentSpeed > 1)
                    {
                        //check if there is danger of collision with car going in this lane
                        if ((i <= Length + 5) ||
                            ((double)i / cellOnTheSide.Car.CurrentSpeed < 2.0 + (cellOnTheSide.Car.CurrentSpeed / 4) - CurrentSpeed)
                            )
                        {
                            return false;
                        }
                    }
                        
                    else
                    {
                        if (random.NextDouble() < 0.4)
                            //if (i <= Length + 1)
                            return true;//go
                        else
                            return false;//slow or stop
                    }
                    return true;
                }
                cellOnTheSide = cellOnTheSide.PreviousCell;
                if (cellOnTheSide == null)
                    break;
            }
            return true;
        }

        private bool CheckCrossroadForCollisions(Cell currentCell, int distanceToCrossroad, Vector directionOfMyCurrentRoad)
        {
            //if car is not on the crossroad cell yet, get nearest crossroadcell in the front to get information if it is main road
            //and if destination road is main road
            Cell cell = currentCell;
            bool currentRoadIsMain;
            bool destinationRoadIsMain;
            Crossroad crossingCrossroad;
            crossingCrossroad = cell.NearestCrossroad;
            currentRoadIsMain = cell.IsMainRoad;
            destinationRoadIsMain = crossingCrossroad.GetCellsOfDirectionIn
                (nextCrossroadDirection.Value).IsMainRoad;
            if (carsWhichToGivePriority.Count > 0 && !CheckIfCarsGivenPriorityAreFarFromCrossroad(crossingCrossroad))
                return true;
            if (!crossingCrossroad.IsGreenLight(directionOfMyCurrentRoad, nextCrossroadDirection.Value))
                //red light
                return true;
            //check for collisions
            if (currentRoadIsMain && !crossingCrossroad.HasSemaphores())
            {
                //current road is main
                if (destinationRoadIsMain)
                {
                    //destination road is also main. Crossroad is free for this car
                    return false;
                }
                else
                {
                    //destination road is not main
                    //check roads to the right in direction to destination road
                    Vector vectorInDirectionToRight = directionOfMyCurrentRoad;
                    while ((vectorInDirectionToRight = crossingCrossroad
                        .GetNextDirectionToRight(vectorInDirectionToRight)) != nextCrossroadDirection.Value)
                    {
                        Cell cellOfRoadInDirectionToRight = crossingCrossroad.GetCellsOfDirectionIn(vectorInDirectionToRight);
                        if (cellOfRoadInDirectionToRight.IsMainRoad)
                        {
                            //crossing main road. Need to check it and wait for cars which are not
                            //going to road to one of the roads between my current and destination road in direction to right
                            if (CheckDirectionForCollisions(vectorInDirectionToRight, directionOfMyCurrentRoad,
                                distanceToCrossroad, crossingCrossroad, true))
                            {
                                return true;
                            }
                        }
                    }
                    //car can go on in current road
                    return false;
                }
            }
            else
            {
                //car is not on the main road
                KeyValuePair<Vector, Vector> mainRoadDirectionVectors;
                if (!crossingCrossroad.HasSemaphores() && crossingCrossroad.HasMainRoads(out mainRoadDirectionVectors))
                {
                    if (CheckMainRoad(mainRoadDirectionVectors.Key, distanceToCrossroad, directionOfMyCurrentRoad, crossingCrossroad))
                        return true;
                    if (CheckMainRoad(mainRoadDirectionVectors.Value, distanceToCrossroad, directionOfMyCurrentRoad, crossingCrossroad))
                        return true;
                }
                //crossroad does not have main roads
                //check for deadlock
                if (!crossingCrossroad.HasSemaphores())
                {
                    Vector currentVector = directionOfMyCurrentRoad;
                    int numberOfWaitingCars = 0;
                    do
                    {
                        if (IsWaitingCar(currentVector, crossingCrossroad))
                            numberOfWaitingCars++;
                    } while ((currentVector = crossingCrossroad.GetNextDirectionToRight(currentVector)) != directionOfMyCurrentRoad);
                    deadlockCounter++;
                    if (numberOfWaitingCars >= 3 && deadlockCounter >= 3)
                    {
                        //possible deadlock. Do not give priority
                        numberOfWaitingCars = 0;
                        deadlockCounter = 0;
                        return false;
                    }
                }
                //check roads to the right in direction to destination road
                Vector vectorInDirectionToRight = directionOfMyCurrentRoad;
                while ((vectorInDirectionToRight = crossingCrossroad
                    .GetNextDirectionToRight(vectorInDirectionToRight)) != nextCrossroadDirection.Value)
                {
                    //checking road at the right. Need to check it and wait for cars which are not
                    //going to road to one of the roads between my current and destination road in direction to right
                    if (CheckDirectionForCollisions(vectorInDirectionToRight, directionOfMyCurrentRoad,
                        distanceToCrossroad, crossingCrossroad, true))
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        private bool IsWaitingCar(Vector currentVector, Crossroad crossingCrossroad)
        {
            Cell cell = crossingCrossroad.GetCellsOfDirectionIn(currentVector);
            LaneMover laneMover = new LaneMover(cell);
            foreach (Cell c in laneMover.GetNext())
            {
                if (laneMover.DirectionDelimiterLineCrossed)
                    break;
                Cell cellToCheck = c;
                for (int i = 0; i < 5; i++)
                {
                    if (!cellToCheck.IsFree && cellToCheck.Car.CurrentSpeed <= 1)
                        return true;
                    cellToCheck = cellToCheck.PreviousCell;
                    if (cellToCheck == null)
                        break;
                }
            }
            return false;            
        }

        private bool CheckIfCarsGivenPriorityAreFarFromCrossroad(Crossroad crossroad)
        {
            foreach (var firstCrossroadOutCell in crossroad.GetCrossroadOutCells())
            {
                LaneMover laneMover = new LaneMover(firstCrossroadOutCell);
                foreach (Cell laneCell in laneMover.GetNext())
                {
                    if (laneMover.DirectionDelimiterLineCrossed)
                        break;
                    Cell currentCell = firstCrossroadOutCell;
                    for (int i = 0; i < 10; i++)
                    {
                        if (currentCell == null)
                            break;
                        if (!currentCell.IsFree && carsWhichToGivePriority.Contains(currentCell.Car))
                            return false;
                        currentCell = currentCell.FollowingCell;
                        if (currentCell == null)
                            break;
                    }
                }
            }
            return true;
            
        }

        /// <summary>
        /// Checks if car has to wait for other cars in main roads
        /// </summary>
        /// <param name="mainRoadDirectionVector"></param>
        /// <param name="distanceToCrossroad"></param>
        /// <param name="directionOfMyCurrentRoad"></param>
        /// <returns></returns>
        private bool CheckMainRoad(Vector mainRoadDirectionVector, int distanceToCrossroad, 
            Vector directionOfMyCurrentRoad, Crossroad crossroad)
        {
            if (mainRoadDirectionVector == new Vector())
                return false;
            if (crossroad.IsDirectionBetweenTwoOtherDirections(mainRoadDirectionVector,
                directionOfMyCurrentRoad, nextCrossroadDirection.Value))
            {
                //main road at right
                if (CheckDirectionForCollisions(mainRoadDirectionVector, directionOfMyCurrentRoad, distanceToCrossroad, crossroad, true))
                {
                    return true;
                }
            }
            else
            {
                //main road at left
                if (CheckDirectionForCollisions(mainRoadDirectionVector, directionOfMyCurrentRoad, distanceToCrossroad, crossroad, false))
                {
                    return true;
                }
            }
            return false;
        }

        private bool CheckDirectionForCollisions(Vector vectorInDirectionToCheck, Vector directionOfMyCurrentRoad, 
            int distanceToCrossroad, Crossroad crossroad, bool rightDirectionsTrueLeftDirectionsFalse)
        {
            Cell cellOfDirectionToCheckEdge = crossroad.GetCellsOfDirectionIn(vectorInDirectionToCheck);
            Car crossingCar = null;
            for (int i = 0; i < 400; i++)
            {
                LaneMover laneMover = new LaneMover(cellOfDirectionToCheckEdge);
                //bool allCarsAreStopped = true;
                foreach (Cell cellOfDirectionToCheck in laneMover.GetNext())
                {
                    if (laneMover.DirectionDelimiterLineCrossed)
                        break;
                    if (!cellOfDirectionToCheck.IsFree && cellOfDirectionToCheck.Car != crossingCar)
                    {
                        //car found
                        crossingCar = cellOfDirectionToCheck.Car;

                        double myTimeToArriveToCrossroad;
                        if (CurrentSpeed == 0)
                            myTimeToArriveToCrossroad = 0;
                        else
                            myTimeToArriveToCrossroad = (double)distanceToCrossroad / CurrentSpeed;
                        double crossingCarsTimeToArriveToCrossroad;
                        if (crossingCar.CurrentSpeed == 0)
                            crossingCarsTimeToArriveToCrossroad = i;
                        else
                            crossingCarsTimeToArriveToCrossroad = (double)i / cellOfDirectionToCheck.Car.CurrentSpeed;
                        if (Math.Abs(crossingCarsTimeToArriveToCrossroad -
                            myTimeToArriveToCrossroad) < (double)5.0 / World.SimulationStepTime)
                        {
                            if (rightDirectionsTrueLeftDirectionsFalse)
                            {
                                //checking roads at the right
                                if (crossingCar.nextCrossroadDirection == null ||
                                    !crossroad.IsDirectionBetweenTwoOtherDirections(crossingCar.nextCrossroadDirection.Value,
                                    directionOfMyCurrentRoad, nextCrossroadDirection.Value))
                                    if (!(crossroad.HasSemaphores()) || (crossingCar.nextCrossroadDirection != null && crossroad.HasSemaphores() && crossroad.IsGreenLight(vectorInDirectionToCheck, crossingCar.nextCrossroadDirection.Value)))
                                        if (HasCollidingCarSameDestinationLane(crossingCar))
                                        {
                                            carsWhichToGivePriority.Add(crossingCar);
                                            return true;
                                        }
                            }
                            else
                            {
                                //checking roads on the left
                                if (crossingCar.nextCrossroadDirection == null ||
                                    !crossroad.IsDirectionBetweenTwoOtherDirections(crossingCar.nextCrossroadDirection.Value,
                                    nextCrossroadDirection.Value, directionOfMyCurrentRoad))
                                    if(crossingCar.nextCrossroadDirection != null && crossingCar.nextCrossroadDirection != directionOfMyCurrentRoad)
                                    if (!(crossroad.HasSemaphores()) || (crossingCar.nextCrossroadDirection == null && crossroad.HasSemaphores() && crossroad.IsGreenLight(vectorInDirectionToCheck, crossingCar.nextCrossroadDirection.Value)))
                                        if (HasCollidingCarSameDestinationLane(crossingCar))
                                        {
                                            carsWhichToGivePriority.Add(crossingCar);
                                            return true;
                                        }
                            }
                        }
                        //if (crossingCar.CurrentSpeed == 0)
                        //    allCarsAreStopped = true;
                    }
                }
                //go to the next cell
                cellOfDirectionToCheckEdge = World.CellBackwards(cellOfDirectionToCheckEdge);
                if (cellOfDirectionToCheckEdge == null)
                    return false;
            }
            return false;
        }



        private bool HasCollidingCarSameDestinationLane(Car collidingCar)
        {
            return true;
        }

        /// <summary>
        /// Gets unique id number
        /// </summary>
        /// <returns>Unique id number for hashing purposes</returns>
        public override int GetHashCode()
        {
            return id;
        }

        /// <summary>
        /// Comparing based on id of instance
        /// </summary>
        /// <param name="obj">Object to compare</param>
        /// <returns>Result of compare</returns>
        public override bool Equals(object obj)
        {
            if (obj is Car)
                return id == (obj as Car).id;
            return false;
        }

        public override string ToString()
        {
            return "Car " + id + ",  current speed is " + CurrentSpeed;
        }

        public int Acc { get; set; }        

        private HashSet<Cell> crossroadCellsWhereItIsPlanningToGo = new HashSet<Cell>();
    
        internal void AddCrossroadCellWhereItIsPlanningToGo(Cell cell)
        {
 	        crossroadCellsWhereItIsPlanningToGo.Add(cell);
        }

        internal void RemoveCrossroadCellWhereItIsPlanningToGo(Cell cell)
        {
            crossroadCellsWhereItIsPlanningToGo.Remove(cell);
        }

        internal void ClearCrossroadCellsWhereItIsPlanningToGo()
        {
            foreach (var i in crossroadCellsWhereItIsPlanningToGo)
                i.ClearNearCars();
            crossroadCellsWhereItIsPlanningToGo.Clear();
        }
    }
}
