<?php

require_once("Database_Xml.php");
require_once("Database_Json.php");
require_once(__DIR__ . "/../config.php");

class Databases
{
    private static $implementations = array("xml" => "Database_Xml", "json" => "Database_Json");

    /**
     * Returns list of supported extensions.
     * @return array
     */
    public static function getExtensions() { return array_keys(self::$implementations); }

    /**
     * Returns list of existing databases.
     * @return array
     */
    public static function all() { return Config::get('db'); }

    public static function create($name, $ext, $ns)
    {
        $ext = strtolower($ext);
        if (!array_key_exists($ext, self::$implementations))
            return null;

        $db = Config::get('db');
        if (array_key_exists($name, $db))
            return null;

        $class = self::$implementations[$ext];
        io_createNamespace($ns . ':.' . $name, 'media');

        $db[$name] = $ns;
        Config::set('db', $db);
        return new $class($name, $ns);
    }

    /**
     * @param string $name
     * @return null | Database_Base
     */
    public static function get($name)
    {
        $db = Config::get('db');
        if (!array_key_exists($name, $db))
            return null;

        $path = self::findDatabase($name, $db[$name]);
        if ($path == null)
            return null;

        $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        $class = self::$implementations[$ext];
        return new $class($name, $db[$name]);
    }

    /**
     * @param string $name
     * @return bool
     */
    public static function remove($name)
    {
        $db = Config::get('db');
        if (!array_key_exists($name, $db))
            return false;

        $path = self::findDatabase($name, $db[$name]);
        if ($path != null)
            @unlink($path);

        io_sweepNS($db[$name] . ":" . $name, 'datadir');
        io_sweepNS($db[$name] . ": " . $name, 'mediadir');

        unset($db[$name]);
        Config::set('db', $db);
        return true;
    }

    private static function findDatabase($name, $ns)
    {
        $ns = mediaFN($ns) . '/';
        if (!file_exists($ns))
            return null;
        $dir = opendir($ns);
        while (($file = readdir($dir)) !== false)
        {
            $path = $ns . $file;
            if (filetype($path) == 'file' && pathinfo($path, PATHINFO_FILENAME) == ".$name")
                return $path;
        }
        return null;
    }
}