<?php

require_once("Database_Base.php");

class Database_Xml extends Database_Base
{
    public function __construct($name, $ns)
    {
        parent::__construct($name, $ns);
        if (!@file_exists($this->getPath()))
        {
            $db = new SimpleXMLElement("<library nextP='1' nextA='1'></library>");
            $this->save($db);
        }
    }

    public function getExtension() { return "xml"; }

    public function get($id)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']");
        $result = (array)$result;
        if (empty($result))
            return null;

        $this->normalize($result, "comment");
        $this->normalize($result, "file");
        $this->normalize($result, "author");
        foreach ($result['author'] as &$author)
            $author = $this->getAuthor($db, $author);
        if (!empty($result['crossref']) && $result['crossref'] != $id)
            $result['crossref'] = $this->get($result['crossref']);
        return $result;
    }

    private function getAuthor(SimpleXMLElement $db, $id)
    {
        list($result) = $db->xpath("author[id='$id']");
        if (empty($result))
            return null;
        return (array)$result;
    }

    private function normalize(&$array, $field)
    {
        if (empty($array[$field]) && !is_string($array[$field]))
            $array[$field] = array();
        elseif (!is_array($array[$field]))
            $array[$field] = array($array[$field]);

        foreach ($array[$field] as &$f)
            if (!is_string($f))
                $f = (array)$f;
    }

    public function find($data)
    {
        $db = $this->load();

        $result = array();
        foreach ($db->children() as $tag => $element)
        {
            if ($tag != 'publication')
                continue;
            $element = (array)$element;
            $this->normalize($element, "author");
            foreach ($element['author'] as &$author)
                $author = $this->getAuthor($db, $author);
            if ($this->findFilter($element, $data))
                $result[] = $element;
        }

        return $result;
    }

    private function findFilter($element, $data)
    {
        foreach ($data as $key => $value)
        {
            if ($key == 'author')
            {
                $filter = false;
                foreach ($element[$key] as $author)
                    if (stripos($author['name'], $value) !== false)
                        $filter = true;
                if (!$filter)
                    return false;
            }
            else if (stripos($element[$key], $value) === false)
                return false;
        }
        return true;
    }

    public function add($data)
    {
        $db = $this->load();
        $child = $db->addChild("publication");

        $child->addChild("id", (int)$db["nextP"]);
        $db["nextP"] += 1;

        $this->addData($db, $child, $data);
        $this->save($db);
        return $child->id;
    }

    private function addData(SimpleXMLElement $db, SimpleXMLElement $element, $data)
    {
        $authors = array();
        if (!empty($data['author']))
            foreach ($data['author'] as $author)
                $authors[] = $this->addAuthor($db, $author);
        $data['author'] = $authors;

        foreach ($data as $key => $value)
        {
            if (empty($value))
                continue;
            if (is_array($value))
                foreach ($value as $v)
                    $element->addChild($key, $v);
            else
                $element->addChild($key, $value);
        }
    }

    private function addAuthor(SimpleXMLElement $db, $data)
    {
        $name = $data['name'];
        list($result) = $db->xpath("author[name='$name']");
        if (empty($result))
        {
            $result = $db->addChild("author");
            $result->addChild('name', $name);
            $result->addChild("id", (int)$db["nextA"]);
            $result->addChild("url", $data['url']);
            $db["nextA"] += 1;
        }

        if (!empty($data['url']))
        {
            if (filter_var($data['url'], FILTER_VALIDATE_URL) === FALSE)
                $data['url'] = null;
            $result->url = $data['url'];
        }
        return $result->id;
    }

    public function edit($id, $data)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']");
        if (empty($result))
            return;

        $toRemove = array();
        foreach ($result->children() as $key => $value)
            if ($key != 'comment' && $key != 'file' && $key != 'id')
                $toRemove[] = $value;

        foreach ($toRemove as $r)
            unset($r[0]);

        $this->addData($db, $result, $data);
        $this->save($db);
    }

    public function remove($id)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']");
        if (empty($result))
            return;

        $array = (array)$result;
        $this->normalize($array, "author");

        foreach ($array['author'] as $author)
        {
            $count = $db->xpath("publication[author='$author' and id!='$id']");
            if (count($count) > 0)
                continue;
            $this->removeNode("author[id='$author']");
        }
        unset($result[0]);
        $this->save($db);
    }

    public function addComment($id, $author, $text)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']");
        if (empty($result))
            return null;

        $comment = $result->addChild("comment");
        $comment->addChild("author", $author);
        $comment->addChild("date", time());
        $comment->addChild("text", $text);
        $this->save($db);
    }

    public function editComment($id, $date, $text)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']/comment[date='$date']");
        if (empty($result))
            return;
        $result->text = $text;
        $this->save($db);
    }

    public function removeComment($id, $date) { $this->removeNode("publication[id='$id']/comment[date='$date']"); }

    public function addFile($id, $name, $date)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication[id='$id']");
        if (empty($result))
            return;

        list($file) = $result->xpath("file[name='$name']");
        if (empty($file))
            $file = $result->addChild("file");

        $file->name = $name;
        $file->date = $date;
        $this->save($db);
    }

    public function removeFile($id, $name) { $this->removeNode("publication[id='$id']/file[name='$name']"); }

    public function isPublicFile($name)
    {
        $db = $this->load();
        list($result) = $db->xpath("publication/file[name='$name']");
        if (empty($result) || empty($result->date))
            return true;
        return new DateTime($result->date) <= new DateTime();
    }

    private function removeNode($xpath)
    {
        $db = $this->load();
        list($result) = $db->xpath($xpath);
        if (empty($result))
            return;
        unset($result[0]);
        $this->save($db);
    }

    protected function load() { return new SimpleXMLElement(parent::load()); }
    protected function save(SimpleXMLElement $db)
    {
        $dom = new DOMDocument("1.0");
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput = true;
        $dom->loadXML($db->asXML());
        $dom->save($this->getPath());
        parent::save($dom->saveXML());
    }
}